// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file CrossPeakList.hh
/// @author Oliver Lange

#ifndef INCLUDED_protocols_NoesyAssign_CrossPeakList_IMPL_HH
#define INCLUDED_protocols_NoesyAssign_CrossPeakList_IMPL_HH


// Unit Header
#include <protocols/NoesyAssign/CrossPeakList.hh>

// Package Headers
#include <protocols/NoesyAssign/CrossPeak.hh>
#include <protocols/NoesyAssign/PeakFileFormat.fwd.hh>
#include <protocols/NoesyAssign/PeakAssignmentResidueMap.hh>
#include <protocols/NoesyAssign/DistanceScoreMover.hh>
//#include <protocols/NoesyAssign/PeakAssignment.hh>
//#include <protocols/NoesyAssign/ResonanceList.fwd.hh>
#include <protocols/jumping/JumpSample.hh>

// Project Headers
#include <core/types.hh>
#include <core/id/NamedAtomID.fwd.hh>
#include <core/chemical/AA.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>
#include <core/pose/Pose.hh>

// Utility headers
#include <utility/exit.hh>
// #include <utility/excn/Exceptions.hh>
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
#include <core/util/Tracer.hh>
// #include <core/options/option.hh>
// #include <core/options/keys/abinitio.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
//#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <list>
#include <map>
#include <cmath>



namespace protocols {
namespace NoesyAssign {


template< class DecoyIterator >
void CrossPeakList::update_decoy_compatibility_score( DecoyIterator const& decoys_begin, DecoyIterator const& decoys_end ) {
  using namespace core;
  using namespace util;
  static core::util::Tracer tr("devel.NoesyAssign.crosspeaks");

  if ( decoys_begin == decoys_end ) return;
  tr.Info << "compute decoy compatibility" << std::endl;

  //compatibility with decoys.
  pose::Pose pose;
  decoys_begin->fill_pose( pose );
  protocols::jumping::JumpSample jumps( pose.fold_tree() );
  jumps.remove_chainbreaks( pose );
  DistanceScoreMoverOP cst_tool =  new DistanceScoreMover( *this, pose );
  cst_tool->prepare_scoring();

  for ( DecoyIterator iss = decoys_begin; iss != decoys_end; ++iss ) {
    pose::Pose pose;
    iss->fill_pose( pose );
    protocols::jumping::JumpSample jumps( pose.fold_tree() );
    jumps.remove_chainbreaks( pose );

    tr.Debug << "score decoys " << iss->decoy_tag() << std::endl;
    cst_tool->apply( pose );
  }

  cst_tool->finalize_scoring();
}


//void CrossPeakList::calibrate( io::silent::SilentFileData const& decoys ) {
template < class DecoyIterator >
void CrossPeakList::calibrate( DecoyIterator const& begin, DecoyIterator const& end ) {
  using namespace core;
  using namespace util;
  static core::util::Tracer tr("devel.NoesyAssign.crosspeaks");

  PeakAssignmentParameters const& params( *PeakAssignmentParameters::get_instance() );
  bool structure_independent_calibration( params.calibration_target_ >= 1.0 || begin == end );

  PeakCalibrator calibrator;
  PeakCalibrator calibrator_low( 1 );
  PeakCalibrator calibrator_high( 1e20 );

  /* there are a couple of problems with the description of this in the manuscript:
     1) does one compute the average dist only for the subset of atoms? how to select with peaks that
     have all sorts of mixed assignments?

     2) only if one has different average values can one also expect to converge the different constants independently...


  */
  DistanceScoreMoverOP cst_tool = NULL;
  if ( !structure_independent_calibration ) {
    pose::Pose pose;
    begin->fill_pose( pose );
    protocols::jumping::JumpSample jumps( pose.fold_tree() );
    jumps.remove_chainbreaks( pose );
    cst_tool =  new DistanceScoreMover( *this, pose );
  }

  tr.Info << "Calibration .... for " << size() << " crosspeaks " << std::endl;
  Real average_dist;
  bool finished = false;
  Size max_cycles = 30;
  Real tolerance = 0.005;
  while ( !finished && max_cycles ) {
    --max_cycles;
    calibrator.interpolate( calibrator_low, calibrator_high );
    average_dist = calibrate( calibrator );
    if ( std::isnan(average_dist) || std::isinf( average_dist)) break;
    bool too_small( false );
    bool too_big( false );
    if ( structure_independent_calibration ) {
      tr.Info << average_dist << " " << params.calibration_target_ << " " << calibrator.Q_backbone_ << std::endl;
      //if ( ( average_dist ) ) break;
      if ( average_dist - params.calibration_target_ < -0.1 ) {
	finished = false;
	too_small = true;
      } else if ( average_dist - params.calibration_target_ > 0.1 ) {
	finished = false;
	too_big = true;
      } else {
	finished = true;
      }
    } else { //structure dependent calibration
      runtime_assert( cst_tool );
      cst_tool->prepare_scoring( true /*used_for_calibration */ );
      Size ct_decoys( 1 );
      for ( DecoyIterator iss = begin; iss != end; ++iss ) {
	pose::Pose pose;
	iss->fill_pose( pose ); //has to reread RDC file for each pose!
	protocols::jumping::JumpSample jumps( pose.fold_tree() );
	jumps.remove_chainbreaks( pose );
	cst_tool->apply( pose );
	++ct_decoys;
      }
      Real violations=cst_tool->compute_violation_percentage();
      tr.Info << violations*100 << "% of constraints violated. "
	      << calibrator.Q_backbone_
	      << " on " << ct_decoys
	      << " structural models"<< std::endl;
      if ( violations - params.calibration_target_ < -tolerance ) {
	finished = false;
	too_big = true;
      } else if ( violations - params.calibration_target_ > tolerance ) {
	finished = false;
	too_small = true;
      } else {
	finished = true;
      }
    }
    if ( too_small ) {
      calibrator_low = calibrator;
    } else {
      calibrator_high = calibrator;
    }
  }

  ///if we have used decoys throw out violated constraints with final call
  if ( cst_tool ) cst_tool->eliminate_violated_constraints();
}

}
}

#endif
