// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @detailed
///
///
///
/// @author Oliver Lange

// Unit Headers
#include <protocols/NoesyAssign/Resonance.hh>
#include <protocols/NoesyAssign/ResonanceList.hh>



// Package Headers
#include <protocols/NoesyAssign/Exceptions.hh>
#include <core/id/NamedAtomID.hh>
// Project Headers
#include <core/chemical/AA.hh>

// Utility headers
#include <ObjexxFCL/format.hh>

// #include <utility/exit.hh>
// #include <utility/excn/Exceptions.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
#include <core/util/Tracer.hh>
// #include <core/options/option.hh>
// #include <core/options/keys/abinitio.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
//#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <deque>


static core::util::Tracer tr("devel.NoesyAssign");

using core::Real;
using namespace core;
using namespace util;
//using namespace core::options;
//using namespace OptionKeys;

namespace protocols {
namespace NoesyAssign {


ResonanceList::ResonanceList( std::string const& sequence ) : sequence_( sequence )
{}

core::chemical::AA ResonanceList::aa_from_resid( core::Size resi ) const {
  runtime_assert(  resi <= sequence_.size() );
  return core::chemical::aa_from_oneletter_code( sequence_[ resi-1 ] );
}

void process_last_resonances( std::deque< Resonance >& last_resonances ) {

  std::string combine_name;
  Resonance first_res = last_resonances.front();
  last_resonances.pop_front();
  bool single_char ( first_res.name().size() == 3 );
  bool double_methyl( last_resonances.size() == 6 );
  Size str_cmp_length( 1 );
  std::string pseudo_letter( "Q" );
  if ( first_res.name().substr(0,1)=="C" ) pseudo_letter = "C";

	//if the two protons stuffed into the ambiguity queue are, .e.g, QG1 + QG2 --> QQG
	if ( first_res.name().substr(0,1)=="Q" ) pseudo_letter = "QQ";

	//things like HB1+HB2 or QG1+QG2
	if ( single_char ) {
    combine_name=pseudo_letter+first_res.name().substr(1,1);
  } else if ( double_methyl ) { //-->QQX
    combine_name=pseudo_letter+pseudo_letter+first_res.name().substr(1,1);
  } else { // HG11+HG12+HG13 --> QG1
    combine_name=pseudo_letter+first_res.name().substr(1,2);
    str_cmp_length=2;
  }
  while( last_resonances.size() > 1 ) {//check others
    if ( first_res.name().substr( 1, str_cmp_length ) == last_resonances.front().name().substr( 1, str_cmp_length ) ) {
      //throw EXCN_UnknownAtomname( "failed to combine "
			//				+first_res.name()+ " and " +last_resonances.front().name() + " into methyl group for residues " + ObjexxFCL::string_of( first_res.resid() ) + " " + ObjexxFCL::string_of( last_resonances.front().resid() )  );
			last_resonances.pop_front();
		} else {
			combine_name = first_res.name();
			break;
		}
  }
  last_resonances.push_front( Resonance( first_res.label(), first_res.freq(), first_res.error(), core::id::NamedAtomID( combine_name, first_res.resid() ) ) );
}

void ResonanceList::read_from_stream( std::istream& is ) {
  using namespace core::chemical; //for AA
  std::string line;
  std::deque< Resonance > last_resonances;
  while( getline( is, line ) ) {
    core::Size label;
    core::Real freq;
    core::Real error;
    std::string name;
    core::Size resn;
    std::istringstream line_stream( line );
    line_stream >> label >> freq >> error >> name;
		if ( !line_stream.good() ) {
			tr.Info << "ignore line : " << line << std::endl;
			continue; //ignore weird lines
		}
		if ( name=="" ) break;//got an empty line at end of file ?
		line_stream >> resn;
    std::string aa_name;
    line_stream >> aa_name;
    AA aa;

    if ( line_stream.good() ) {
      if ( aa_name.size() == 1 ) {
				aa = aa_from_oneletter_code( aa_name[ 0 ] );
      } else if ( aa_name.size() == 3 ) {
				aa = aa_from_name( aa_name );
      } else {
				throw utility::excn::EXCN_BadInput( "did not recognize aminoacid: " + aa_name);
      }
      if ( sequence_.size() < resn ) {
				while ( sequence_.size() < resn-1 ) sequence_.push_back('X');
				sequence_[ resn-1 ]=oneletter_code_from_aa( aa );
      } else if ( sequence_[ resn-1 ]!=oneletter_code_from_aa( aa ) ) {
				tr.Warning << "sequence mismatch!! check your data: found " << name_from_aa( aa ) << " in line " << line
									 << " which does not match " << sequence_[ resn-1 ] << "\n" << sequence_ << std::endl;
      }
    } else {
		  aa = aa_from_resid( resn );
		}

		if ( name == "HN" ) name ="H";
		if ( aa == aa_leu && name == "HD1" ) name = "QD1";
		if ( aa == aa_leu && name == "HD2" ) name = "QD2";
		if ( aa == aa_val && name == "HG1" ) name = "QG1";
		if ( aa == aa_val && name == "HG2" ) name = "QG2";
		if ( aa == aa_thr && name == "HG2" ) name = "QG2";
		if ( aa == aa_ala && name == "HB"  ) name = "QB";
		if ( aa == aa_gly && name == "HA"  ) name = "QA";
		if ( aa == aa_ile && name == "HG1" ) name = "QG1";
		if ( aa == aa_ile && name == "HG2" ) name = "QG2";
		if ( aa == aa_ile && name == "HD1" ) name = "QD1";

    last_resonances.push_back( Resonance( label, freq, error, core::id::NamedAtomID( name, resn ) ) );
    if ( freq != last_resonances.front().freq() ) {
      if ( last_resonances.size() > 2 ) process_last_resonances( last_resonances ); //just 2 --> 1 old freq and 1 new freq
      map_[ last_resonances.front().label() ]= last_resonances.front();
      last_resonances.pop_front();
    }
  }
  if ( last_resonances.size() > 1 ) process_last_resonances( last_resonances );
  map_[ last_resonances.front().label() ]= last_resonances.front();
  runtime_assert( last_resonances.size() == 1 );
  if ( is.fail() && is.eof() && is.bad() ) {
    tr.Error << "[ERROR WHILE READING]" << std::endl;
  }
	update_residue_map();
}

void ResonanceList::write_to_stream( std::ostream& os   ) const {
  for ( ResonanceIDs::const_iterator it = map_.begin(); it != map_.end(); ++it ) {
    runtime_assert( it->first == it->second.label() );
    it->second.write_to_stream( os );
    if ( sequence_.size() ) {
      using namespace core::chemical; //for AA
			AA aa( aa_from_resid( it->second.resid() ) );
      os << " " << name_from_aa( aa ) << " " << oneletter_code_from_aa( aa );
    }
    os << std::endl;
  }
}

Resonance const& ResonanceList::operator[] ( core::id::NamedAtomID const& atom ) const {
	ResidueMap::const_iterator it_res( by_resid_.find( atom.rsd() ) );
	if ( it_res != by_resid_.end() ) {
		Resonances const& reso_list( it_res->second );
		for ( Resonances::const_iterator it = reso_list.begin(); it != reso_list.end(); ++it ) {
			if ( it->atom() == atom ) return *it;
		}
	}
	throw EXCN_UnknownResonance( atom, "can't find atom ");

//   for ( ResonanceIDs::const_iterator it = map_.begin(); it != map_.end(); ++it ) {
//     runtime_assert( it->first == it->second.label() );
//     if ( it->second.atom() == atom ) return it->second;
//   }
  //tr.Warning << "can't find " << atom << " in resonance assignment" << std::endl;
  return map_.begin()->second; //to make compiler happy
}

Resonance const& ResonanceList::operator[] ( core::Size key ) const {
  ResonanceIDs::const_iterator iter = map_.find( key );
  if ( iter == map_.end() ) {
    throw EXCN_UnknownResonance( id::BOGUS_NAMED_ATOM_ID, "can't find resonance " + ObjexxFCL::string_of( key ) );
  }
  return iter->second;
}

void ResonanceList::update_residue_map() {
	by_resid_.clear();
	for ( ResonanceIDs::const_iterator it = map_.begin(); it != map_.end(); ++it ) {
    runtime_assert( it->first == it->second.label() );
		by_resid_[ it->second.resid() ].push_back( it->second );
	}
}

ResonanceList::Resonances const& ResonanceList::resonances_at_residue( core::Size resid ) const {
	ResidueMap::const_iterator it_res( by_resid_.find( resid ) );
	if ( it_res != by_resid_.end() ) {
		return it_res->second;
	}
	throw EXCN_UnknownResonance( id::BOGUS_NAMED_ATOM_ID, "can't find resonance with residue " + ObjexxFCL::string_of( resid ) );
	return by_resid_.begin()->second; //to make compile happy
}


bool ResonanceList::has_residue( core::Size resid ) const {
	return by_resid_.find( resid ) != by_resid_.end();
}

}
}
