// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Frank DiMaio
/// @author Srivatsan Raman

#include <protocols/RBSegmentMoves/RBSegmentMover.hh>
#include <protocols/RBSegmentMoves/util.hh>
#include <protocols/loops/Loops.hh>

// Rosetta Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueFactory.hh>

#include <core/kinematics/FoldTree.hh>

#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/BoundConstraint.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/AmbiguousConstraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/util.hh>
#include <core/chemical/VariantType.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/util/basic.hh>
#include <core/util/Tracer.hh>

// Random number generator
#include <numeric/xyzVector.io.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/random/random.hh>
#include <ObjexxFCL/FArray1D.hh>

//
#include <string>

//Auto Headers
#include <core/id/NamedAtomID.hh>
#include <core/sequence/SequenceMapping.hh>


namespace protocols {
namespace RBSegment {

using namespace core;

static numeric::random::RandomGenerator rbseg_RG(186331);
static core::util::Tracer TR("protocols::moves::RBSegmentMover");


//
///@brief set up constraints from RB segs
///       currently uses bounded constraints on each CA ... make CST type flag-selectable????
void set_rb_constraints(
	core::pose::Pose & pose,
	core::pose::Pose const &cst_pose,
	utility::vector1< protocols::RBSegment::RBSegment > const & rbsegs ,
	core::sequence::SequenceMapping const & resmap,
	core::Real cst_width,
	core::Real cst_stdev,
	core::Size cst_seqwidth
                   ) {
	core::scoring::constraints::ConstraintSetOP new_csts =	new core::scoring::constraints::ConstraintSet;

	for (int i =  1; i <= (int)rbsegs.size(); ++i) {
		for (core::Size j=1; j<=rbsegs[i].nContinuousSegments(); ++j) {
			int rbstart = rbsegs[i][j].start(), rbend = rbsegs[i][j].end();

			for (int cst_i=rbstart; cst_i<=rbend; ++cst_i) {

				// make ambiguous cst from ( cst_i-cst_seqwidth , cst_i+cst_seqwidth )
				core::scoring::constraints::ConstraintCOPs CSTs_i;

				//
				for (int cst_ij = std::max(1,cst_i-(int)cst_seqwidth), cst_ij_end=std::min(cst_pose.total_residue(),cst_i+cst_seqwidth);
				     cst_ij<=cst_ij_end; ++cst_ij) {
					// only constrain protein residues
					if (!cst_pose.residue(cst_ij).is_protein()) continue;

					core::scoring::constraints::ConstraintOP newcst_ij = new core::scoring::constraints::CoordinateConstraint(
											 core::id::AtomID( core::id::NamedAtomID( "CA"   , resmap[cst_i] ), pose ),
											 core::id::AtomID( core::id::NamedAtomID( "ORIG" , pose.total_residue() ), pose ),
											 cst_pose.residue(cst_ij).xyz( "CA" ),
											 new core::scoring::constraints::BoundFunc(0,cst_width,cst_stdev,"xyz") );


					// make sure not randomized <<<< kind of hacky
					core::Real len = (cst_pose.residue(cst_ij).xyz( "CA" )).length();
					if ( len < 500 )
						CSTs_i.push_back( newcst_ij );
				}

				//
				if ( CSTs_i.size() > 0 ) {
					TR << "Adding " << CSTs_i.size() << " ambiguous constraints for res " << resmap[cst_i]
					   << " (= input " << cst_i << ")" << std::endl;
					new_csts->add_constraint( new core::scoring::constraints::AmbiguousConstraint( CSTs_i ) );
				}

			}
		}
	}
	pose.constraint_set( new_csts );
}


///@brief set up constraints on complete pose (not just RB segments)
///       currently uses bounded constraints ... make CST type flag-selectable????
void set_constraints(
	core::pose::Pose & pose,
	core::pose::Pose const &cst_pose,
	core::Real cst_width,
	core::Real cst_stdev,
	core::Size cst_seqwidth
                   ) {
	if (pose.total_residue() != cst_pose.total_residue()) {
		TR.Warning << "set_constraints() error: #res in cst_pose (" << cst_pose.total_residue()
		            << ") != #res in pose (" << pose.total_residue() << ").  Continuing..." << std::endl;
	}
	int nres = (int)std::min( pose.total_residue() , cst_pose.total_residue() );

	core::scoring::constraints::ConstraintSetOP new_csts =	new core::scoring::constraints::ConstraintSet;
	for (int cst_i =  1; cst_i < nres; ++cst_i) {
		// make ambiguous cst from ( cst_i-cst_seqwidth , cst_i+cst_seqwidth )
		core::scoring::constraints::ConstraintCOPs CSTs_i;

		for (int cst_ij=std::max(1,cst_i-(int)cst_seqwidth), cst_ij_end=std::min(nres,cst_i+(int)cst_seqwidth);
				 cst_ij<=cst_ij_end; ++cst_ij) {
			// only constrain protein residues
			if (!cst_pose.residue(cst_ij).is_protein()) continue;

			core::scoring::constraints::ConstraintOP newcst_ij = new core::scoring::constraints::CoordinateConstraint(
									 core::id::AtomID( core::id::NamedAtomID( "CA"   , cst_i ), pose ),
									 core::id::AtomID( core::id::NamedAtomID( "ORIG" , pose.total_residue() ), pose ),
									 cst_pose.residue(cst_ij).xyz( "CA" ),
									 new core::scoring::constraints::BoundFunc(0,cst_width,cst_stdev,"xyz") );

			// make sure not randomized <<<< kind of hacky
			core::Real len = (cst_pose.residue(cst_ij).xyz( "CA" )).length();
			if ( len < 500 )
				CSTs_i.push_back( newcst_ij );
		}

		//
		if ( CSTs_i.size() > 0 ) {
			TR << "Adding " << CSTs_i.size() << " ambiguous constraints for res " << cst_i << std::endl;
			new_csts->add_constraint( new core::scoring::constraints::AmbiguousConstraint( CSTs_i ) );
		}
	}
	pose.constraint_set( new_csts );
}


///
///@brief Helper function to set up a pose; unlike alt version keep loops (use cutpoint variants)
///   unlike version in loops_main, this uses RBSegment structure to build multi-level topology
/// returns jump residues
utility::vector1< core::Size > setup_pose_rbsegs_keep_loops(
              core::pose::Pose &pose,
              utility::vector1< protocols::RBSegment::RBSegment > const &rbsegs,
              loops::Loops const &loops,
              core::kinematics::MoveMapOP mm) {
 	using namespace core::kinematics;

 	core::Size nres( pose.total_residue()-1 );
 	core::Size vrtid = nres+1;
 	core::Size nrbsegs( rbsegs.size() );

 	// "star" topology fold tree
	utility::vector1< core::Size > cuts, jump_res;
	utility::vector1< std::pair<core::Size,core::Size> > jumps;
	for( loops::Loops::const_iterator it=loops.begin(), it_end=loops.end(); it != it_end; ++it ) {
		if (it->start() == 1 || it->stop() == nres) continue;
		cuts.push_back( it->cut() );
	}
	cuts.push_back( nres );
	//cuts.push_back( nres+1 ); //?
	for (int i=1; i <= (int)nrbsegs; ++i ) {
		// jump from vrt to midpt of 1st seg
		core::Size midpt = (rbsegs[i][1].start()+rbsegs[i][1].end()) / 2;
		jumps.push_back (std::pair<core::Size,core::Size>( vrtid, midpt ) );
		jump_res.push_back ( midpt );
	}
	for (int i=1; i <= (int)nrbsegs; ++i ) {
		for (int j=2; j<= (int)rbsegs[i].nContinuousSegments() ; ++j ) {
			// jump from midpt of 1st seg here
			core::Size midpt1 = (rbsegs[i][1].start()+rbsegs[i][1].end()) / 2;
			core::Size midpt2 = (rbsegs[i][j].start()+rbsegs[i][j].end()) / 2;
			jumps.push_back (std::pair<core::Size,core::Size>( midpt1, midpt2 ) );
			jump_res.push_back ( midpt2 );
		}
	}

	ObjexxFCL::FArray2D_int fjumps( 2, jumps.size() );
	ObjexxFCL::FArray1D_int fcuts ( cuts.size() );
	for ( Size i=1; i<=jumps.size(); ++i ) {
		fjumps(1,i) = std::min( jumps[i].first , jumps[i].second );
		fjumps(2,i) = std::max( jumps[i].first , jumps[i].second );
	}
	for ( Size i=1; i<=cuts.size(); ++i ) {
		fcuts(i) = cuts[i];
	}
	kinematics::FoldTree f;
	bool valid_tree = f.tree_from_jumps_and_cuts( nres+1, jumps.size(), fjumps, fcuts );
	runtime_assert( valid_tree );
	f.reorder( vrtid );
	TR << "New fold tree: " << f << std::endl;

	pose.fold_tree( f );

	// movemap
	mm->clear();
	for( loops::Loops::const_iterator it=loops.begin(), it_end=loops.end(); it != it_end; ++it )
		for ( int j=(int)(it->start()) ; j<=(int)(it->stop()); ++j )
			mm->set_bb(j,true);
	for (int i=1; i <= (int)nrbsegs; ++i )
		mm->set_jump(i,true);

	// cb variants
	for( protocols::loops::Loops::const_iterator it=loops.begin(), it_end=loops.end(); it != it_end; ++it ) {
	 	Size const loop_cut(it->cut());
	 	if ( !pose.residue(loop_cut).is_upper_terminus() ) {
	 		if ( ! pose.residue(loop_cut).has_variant_type(core::chemical::CUTPOINT_LOWER) )
	 			core::chemical::add_variant_type_to_pose_residue( pose, core::chemical::CUTPOINT_LOWER, loop_cut );
	 		if ( ! pose.residue(loop_cut+1).has_variant_type(core::chemical::CUTPOINT_UPPER) )
	 			core::chemical::add_variant_type_to_pose_residue( pose, core::chemical::CUTPOINT_UPPER, loop_cut+1 );
	 	}
	}

	return jump_res;
}


///
///@brief Helper function to set up a loop-removed pose
void setup_pose_from_rbsegs(
             utility::vector1< protocols::RBSegment::RBSegment > const &rbsegs ,
             core::pose::Pose const &pose_in ,
             core::pose::Pose &pose_out ,
             core::sequence::SequenceMapping &resmap,
             core::kinematics::MoveMap &mm,
             bool fix_ligand  ) {
	using namespace core::kinematics;

	core::Size nres( pose_in.total_residue() );
	core::Size nres_rb = 0;

	// each ligand is its own rb-segment (for the purposes of fold-tree generation)
	utility::vector1< protocols::RBSegment::RBSegment > rbsegs_with_ligands = rbsegs;
	core::Size last_peptide_res = nres;
	while ( !pose_in.residue( last_peptide_res ).is_protein() )
		last_peptide_res--;
	for (core::Size i=last_peptide_res+1; i<=nres; ++i) {
		if ( pose_in.residue( i ).aa() != core::chemical::aa_vrt ) {
			rbsegs_with_ligands.push_back( protocols::RBSegment::RBSegment( i, i, 'X' ) );
			TR << "setup_pose_from_rbsegs: Ligand at " << i << std::endl;
		}
	}

	// count rb reses
	for ( core::Size i=1; i <= rbsegs_with_ligands.size(); ++i )
		for (core::Size j=1; j<=rbsegs_with_ligands[i].nContinuousSegments(); ++j)
			nres_rb += rbsegs_with_ligands[i][j].end() - rbsegs_with_ligands[i][j].start() + 1;

	resmap.resize( nres, nres_rb );

	pose_out.clear();
	int rb_ctr = 0;
	for ( core::Size i=1; i <= rbsegs_with_ligands.size(); ++i ) {

		for (core::Size j=1; j<=rbsegs_with_ligands[i].nContinuousSegments(); ++j) {
			core::Size rb_start  = rbsegs_with_ligands[i][j].start() ,
			           rb_end    = rbsegs_with_ligands[i][j].end()   ;
			int nsegment_reses = rb_end - rb_start + 1;
			rb_ctr++;

			char secstruct = rbsegs_with_ligands[i][j].char_type();

			if (nsegment_reses > 0) {
				core::conformation::Residue new_res_0 = pose_in.residue( rb_start );
				pose_out.append_residue_by_jump( new_res_0 , pose_out.total_residue(), "" , "" , true );
				resmap[rb_start] = pose_out.total_residue();

				for (int k=1; k<nsegment_reses; ++k) {
					core::conformation::Residue new_res_k = pose_in.residue( rb_start+k );
					pose_out.append_residue_by_bond( new_res_k );
					resmap[rb_start + k] = pose_out.total_residue();

					// set secstruct
					if ( (secstruct == 'H' || secstruct == 'E') && k!=nsegment_reses-1 )
						pose_out.set_secstruct( pose_out.total_residue(), secstruct );
				}
			}
		}
	}

	// virtual res as root
	core::chemical::ResidueTypeSetCAP const &residue_set(
									core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD )  );
	core::chemical::ResidueTypeCAPs const & rsd_type_list( residue_set->name3_map("VRT") );
	core::conformation::ResidueOP new_res( core::conformation::ResidueFactory::create_residue( *rsd_type_list[1] ) );
	pose_out.append_residue_by_jump( *new_res , 1 );

	// "star" topology fold tree
	core::kinematics::FoldTree newF;
	core::Size nstart = 1, njump = 1;
	for ( core::Size i=1; i <= rbsegs_with_ligands.size(); ++i ) {
		core::Size in_start  = rbsegs_with_ligands[i][1].start() ,
				       in_end    = rbsegs_with_ligands[i][1].end();
		core::Size out_start  = nstart,
				       out_end    = nstart + (in_end - in_start);
		core::Size out_midpt  = (out_end + out_start)/2;

		newF.add_edge( nres_rb+1, out_midpt, njump );
		newF.add_edge( out_midpt, out_start,    Edge::PEPTIDE );
		newF.add_edge( out_midpt, out_end  , Edge::PEPTIDE );
		nstart = out_end+1;

		// let all jumps except those to ligands move
		if ( !fix_ligand || i <= rbsegs.size() )
			mm.set_jump( njump , true );

		njump++;

		// in strands let bb move
		if (rbsegs_with_ligands[i].isSheet()) {
			for ( core::Size j=out_start; j<=out_end; ++j) {
				mm.set_bb( j, true );
			}
		}

		core::Size in_midpt = out_midpt;
		for (core::Size j=2; j<=rbsegs_with_ligands[i].nContinuousSegments(); ++j) {
			in_start  = rbsegs_with_ligands[i][j].start();
			in_end    = rbsegs_with_ligands[i][j].end();
			out_start  = nstart;
			out_end    = nstart + (in_end - in_start);
			out_midpt  = (out_end + out_start)/2;

			newF.add_edge( in_midpt, out_midpt, njump );
			newF.add_edge( out_midpt, out_start, Edge::PEPTIDE );
			newF.add_edge( out_midpt, out_end  , Edge::PEPTIDE );
			nstart = out_end+1;
			mm.set_jump( njump , false );
			njump++;
		}
	}

	newF.reorder( nres_rb+1 );
	pose_out.fold_tree( newF );

	TR << "New fold tree: " << newF << std::endl;
}



///@brief Helper function to restore a fully-connected pose
void restore_pose_from_rbsegs(
             utility::vector1< protocols::RBSegment::RBSegment > const &rbsegs ,
             core::pose::Pose const &pose_in ,
             core::pose::Pose &pose_out /* input/output */ )
{
	using namespace core::kinematics;

	// each ligand is its own rb-segment (for the purposes of fold-tree generation)
	utility::vector1< protocols::RBSegment::RBSegment > rbsegs_with_ligands = rbsegs;
	/*
	int nres( pose_out.total_residue() );
	// keep ligand confs from starting pose
	int last_peptide_res = nres;
	while ( !pose_out.residue( last_peptide_res ).is_protein() )
		last_peptide_res--;
	for (core::Size i=last_peptide_res+1; i<=nres; ++i) {
		if ( pose_out.residue( i ).aa() != core::chemical::aa_vrt ) {
			rbsegs_with_ligands.push_back( protocols::RBSegment::RBSegment( i, i, 'X' ) );
			TZ << "restore_pose_from_rbsegs: Ligand at " << i << std::endl;
		}
	}
	*/

	int res_rb_counter = 1;
	for ( core::Size i=1; i <= rbsegs_with_ligands.size(); ++i ) {

		for (core::Size j=1; j<=rbsegs_with_ligands[i].nContinuousSegments(); ++j) {

			core::Size rb_start  = rbsegs_with_ligands[i][j].start() ,
								 rb_end    = rbsegs_with_ligands[i][j].end()   ;
			int nsegment_reses = rb_end - rb_start + 1;

			// xyz copy??
			if (nsegment_reses > 0) {
					pose_out.copy_segment( nsegment_reses, pose_in,  rb_start, res_rb_counter  );
			}
			// copy secstruct
			for (int k=(int)rb_start; k<=(int)rb_end; ++k)
				pose_out.set_secstruct( k, pose_in.secstruct( res_rb_counter+k-rb_start ) );
			res_rb_counter += nsegment_reses;
		}

	}

	// virtual res as root
	core::chemical::ResidueTypeSetCAP const &residue_set(
									core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD )  );
	core::chemical::ResidueTypeCAPs const & rsd_type_list( residue_set->name3_map("VRT") );
	core::conformation::ResidueOP new_res( core::conformation::ResidueFactory::create_residue( *rsd_type_list[1] ) );
	pose_out.append_residue_by_jump( *new_res , pose_out.total_residue()/2 );

	// make the virt atom the root
	core::kinematics::FoldTree newF(pose_out.fold_tree());
	newF.reorder( pose_out.total_residue() );
	pose_out.fold_tree( newF );

	TR << "New fold tree: " << newF << std::endl;
}

//// remap all segs
void remap_rb_segments(
                 utility::vector1< RBSegment > const &rbsegs,
                 utility::vector1< RBSegment > &rbsegs_remap,
                 core::sequence::SequenceMapping const &resmap ) {
	for ( RBConsIt it_seg = rbsegs.begin(), it_seg_end = rbsegs.end(); it_seg != it_seg_end; ++it_seg ) {
		RBSegment it_remap = it_seg->remap( resmap );
		rbsegs_remap.push_back( it_remap );
	}
}



}
}
