// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file DockMCMProtocol
/// @brief protocols that are specific to high resolution docking
/// @detailed
///		This contains the functions that create initial positions for docking
///		You can either randomize partner 1 or partner 2, spin partner 2, or
///		perform a simple perturbation.
/// 	Also contains docking mcm protocol
/// @author Monica Berrondo
/// @author Modified by Sergey Lyskov
/// @author Modified by Sid Chaudhury
/// @author Modified by Jacob Corn

// Unit Headers
#include <protocols/docking/DockMCMProtocol.hh>

// Package Headers
#include <protocols/docking/DockFilters.hh>
#include <protocols/docking/DockMCMCycle.hh>

// Project Headers
#include <core/scoring/ScoreFunction.hh>
#include <core/pack/task/TaskFactory.hh>
#include <protocols/moves/MonteCarlo.fwd.hh>


// Utility Headers
#include <core/util/Tracer.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockMCMProtocol");

//     originally from dock_structure.cc Jeff Gray April 2001

using namespace core;

namespace protocols {
namespace docking {

// default constructor
DockMCMProtocol::DockMCMProtocol() : DockingHighRes( 1 )
{
	init();
}

// constructor with arguments
// only one movable jump
DockMCMProtocol::DockMCMProtocol(
	core::Size const rb_jump
) : DockingHighRes(rb_jump)
{
	init();
}

// constructor with arguments
// only one movable jump, scoring and packing defined
DockMCMProtocol::DockMCMProtocol(
	core::Size const rb_jump,
	core::scoring::ScoreFunctionCOP scorefxn,
	core::scoring::ScoreFunctionCOP scorefxn_pack
) : DockingHighRes(rb_jump, scorefxn, scorefxn_pack)
{
	init();
}

// constructor with arguments
// only one movable jump, scoring and packing defined
DockMCMProtocol::DockMCMProtocol(
	DockJumps const movable_jumps,
	core::scoring::ScoreFunctionCOP scorefxn
) : DockingHighRes(movable_jumps[1], scorefxn)
{
	init();
}

// constructor with arguments
// only one movable jump, scoring and packing defined
DockMCMProtocol::DockMCMProtocol(
	DockJumps const movable_jumps,
	core::scoring::ScoreFunctionCOP scorefxn,
	core::scoring::ScoreFunctionCOP scorefxn_pack
) : DockingHighRes(movable_jumps, scorefxn, scorefxn_pack)
{
	init();
}

//destructor
DockMCMProtocol::~DockMCMProtocol() {}

//clone
protocols::moves::MoverOP DockMCMProtocol::clone() const {
	return new DockMCMProtocol(*this);
}

void
DockMCMProtocol::init()
{
	moves::Mover::type( "DockMCMProtocol" );
	filter_ = new DockingHighResFilter();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking high resolution apply function
/// @brief
/// @detailed
///		decides what to call according to options
void DockMCMProtocol::apply( core::pose::Pose & pose )
{
	using namespace scoring;
	
	TR << "in DockMCMProtocol.apply" << std::endl;

	tf2()->create_and_attach_task_factory( this, pose );

	DockMCMCycleOP dock_mcm = new DockMCMCycle( movable_jumps(), scorefxn(), scorefxn_pack() );
	dock_mcm->set_task_factory( task_factory() );

	dock_mcm->apply( pose );
	dock_mcm->get_mc()->show_scores();

	filter_->set_score_margin( 10.0 );
	if ( filter_->apply( pose ) )
	{
		for ( Size i=1; i<=4; ++i ) { dock_mcm->apply( pose );
		dock_mcm->get_mc()->show_scores();}

		filter_->set_score_margin( 5.0 );
		if ( filter_->apply( pose ) )
		{
			for ( Size i=1; i<=45; ++i ) { dock_mcm->apply( pose );
			dock_mcm->get_mc()->show_scores();}
		}
	}
	filter_->set_score_margin( 0.0 );
	// add minimize to strict tolerance 0.02
}

std::string
DockMCMProtocol::get_name() const {
	return "DockMCMProtocol";
}


} // namespace docking
} // namespace protocols
