// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/enzdes/EnzFilters.cc
/// @brief
/// @author Sagar Khare (khares@uw.edu)

// unit headers
#include <protocols/enzdes/EnzFilters.hh>
#include <protocols/enzdes/EnzFilterCreators.hh>

// package headers
#include <protocols/toolbox/match_enzdes_util/EnzConstraintIO.hh>
#include <protocols/enzdes/EnzdesCacheableObserver.hh>
#include <protocols/toolbox/match_enzdes_util/EnzdesCstCache.hh>
#include <protocols/enzdes/EnzdesMovers.hh>
#include <protocols/enzdes/EnzdesSeqRecoveryCache.hh>
#include <protocols/enzdes/enzdes_util.hh>

// Project Headers
#include <core/conformation/Conformation.hh>
#include <core/conformation/Interface.hh>
#include <core/id/AtomID.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/options/option.hh>
#include <core/options/keys/enzdes.OptionKeys.gen.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/PDBPoseMap.hh> //for PDB-info-to-resid functionality
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableObserverType.hh>
#include <core/pose/datacache/ObserverCache.hh>
#include <core/pose/metrics/CalculatorFactory.hh>
#include <core/pose/metrics/PoseMetricCalculatorBase.fwd.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.tmpl.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreTypeManager.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/types.hh>
#include <core/util/datacache/cacheable_observers.hh>
#include <core/util/MetricValue.hh>

#include <protocols/ligand_docking/LigandBaseProtocol.hh>
#include <protocols/moves/DataMap.hh>
#include <protocols/RosettaScripts/util.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceSasaDefinitionCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceNeighborDefinitionCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceDeltaEnergeticsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NonlocalContactsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NumberHBondsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/BuriedUnsatisfiedPolarsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/PackstatCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/SurfaceCalculator.hh>

#include <numeric/random/random.hh>

#include <utility/Tag/Tag.hh>


//Objectxxxx header
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/format.hh>

using namespace core;
using namespace core::scoring;
using namespace ObjexxFCL::fmt;

static core::util::Tracer TR( "protocols.enzdes.EnzFilters" );
static numeric::random::RandomGenerator RG( 14084760 ); // <- Magic number, do not change it!!!

namespace protocols {
namespace enzdes {

using namespace protocols::filters;
using namespace protocols::moves;
using namespace utility::Tag;

using core::pose::Pose;

LigDSasaFilter::LigDSasaFilter( core::Real const lower_threshold, core::Real const upper_threshold ) : Filter( "DSasa" ), lower_threshold_( lower_threshold ), upper_threshold_(upper_threshold) {}

bool
LigDSasaFilter::apply( core::pose::Pose const & pose ) const {
	core::Real const dsasa( compute( pose ) );

	TR<<"dsasa is "<<dsasa<<". ";
	if( dsasa >= lower_threshold_ && dsasa <= upper_threshold_){
		TR<<"passing." <<std::endl;
		return true;
	}
	else {
		TR<<"failing."<<std::endl;
		return false;
	}
}

void
LigDSasaFilter::report( std::ostream & out, core::pose::Pose const & pose ) const {
	core::Real const dsasa( compute( pose ));
	out<<"DSasa= "<< dsasa<<'\n';
}

core::Real
LigDSasaFilter::report_sm( core::pose::Pose const & pose ) const {
	core::Real const dsasa( compute( pose ));
	return( dsasa );
}

core::Real
LigDSasaFilter::compute( core::pose::Pose const & pose ) const {
	core::util::MetricValue< core::Real > mv_sasa;
	core::Real sasa (0.0);
	core::Size lig_chain(2), prot_chain;
  for( core::Size i = 1;  i<= pose.total_residue(); ++i){
		if (pose.residue_type( i ).is_ligand()) lig_chain = pose.chain( i );
	}
  prot_chain = pose.chain( 1 );   //we're making the not so wild assumption that the the first residue in the pose belongs to the protein
  std::string lig_ch_string = utility::to_string( lig_chain );
  std::string prot_ch_string = utility::to_string( prot_chain );
  if( lig_chain == prot_chain ) { utility_exit_with_message( "WTF?!? ligand and residue 1 are on the same chain... " );}
	else{
	TR<<"Now calculating SaSa"<<std::endl;
	pose.metric( "sasa_interface", "frac_ch2_dsasa", mv_sasa); // if this seems like its come out of nowhere see declaration in protocols/jd2/DockDesignParser.cc
	sasa =  mv_sasa.value() ;
	}
	return( sasa );
}

void
LigDSasaFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
  lower_threshold_ = tag->getOption<core::Real>( "lower_threshold", 0 );
  upper_threshold_ = tag->getOption<core::Real>( "upper_threshold", 1 );

  TR<<"LigDSasaFilter with lower threshold of "<<lower_threshold_<<" and upper threshold of "<< upper_threshold_ <<std::endl;
}

LigDSasaFilter::~LigDSasaFilter() {}

DiffAtomSasaFilter::DiffAtomSasaFilter( core::Size resid1, core::Size resid2, std::string atomname1, std::string atomname2, std::string sample_type ) : Filter( "DiffAtomBurial" ), resid1_( resid1 ), resid2_( resid2), aname1_( atomname1 ), aname2_( atomname2 ), sample_type_( sample_type ) {}

bool
DiffAtomSasaFilter::apply( core::pose::Pose const & pose ) const {
  bool const dsasa( compute( pose ) );

  if( dsasa ){
    TR<<"Diff Atom Sasa filter passing." <<std::endl;
    return true;
  }
  else {
    TR<<"Diff Atom Sasa filter failing."<<std::endl;
    return false;
  }
}

void
DiffAtomSasaFilter::report( std::ostream & out, core::pose::Pose const & pose ) const {
  bool const dsasa( compute( pose ));
  out<<"Diff Sasa= "<< dsasa<<'\n';
}

core::Real
DiffAtomSasaFilter::report_sm( core::pose::Pose const & pose ) const {
  bool const dsasa( compute( pose ));
	if( dsasa ) return 1;
	else return 0;
}

bool
DiffAtomSasaFilter::compute( core::pose::Pose const & pose ) const {
	core::util::MetricValue< id::AtomID_Map< core::Real > > atom_sasa;
	bool setting (false);
  pose.metric( "sasa_interface", "delta_atom_sasa", atom_sasa );
	core::id::AtomID const atomid1 (core::id::AtomID( pose.residue( resid1_ ).atom_index(aname1_), resid1_ ));
  core::id::AtomID const atomid2 (core::id::AtomID( pose.residue( resid2_ ).atom_index(aname2_), resid2_ ));
  core::Real const atom1_delta_sasa( atom_sasa.value()( atomid1 ) );
  core::Real const atom2_delta_sasa( atom_sasa.value()( atomid2 ) );
	TR<<"Atom1: Dsasa is "<< atom1_delta_sasa <<" and Atom2 Dsasa is "<< atom2_delta_sasa << std::endl;
	if (sample_type_ == "more") {
		if (atom1_delta_sasa > atom2_delta_sasa) setting=true;
	}
	else if (atom1_delta_sasa < atom2_delta_sasa) setting=true; //sample_type == less i.e. see if a1 is less buried than a2 upon binding
	return setting;
}

void
DiffAtomSasaFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const &pose )
{
	 if ( tag->hasOption("res1_res_num") ) resid1_ =  tag->getOption<core::Size>( "res1_res_num", 0 );
	 if ( tag->hasOption("res2_res_num") ) resid2_ =  tag->getOption<core::Size>( "res2_res_num", 0 );
	 if ( tag->hasOption("res1_pdb_num") ) resid1_ = protocols::RosettaScripts::get_resnum(tag, pose, "res1_");
	 if ( tag->hasOption("res2_pdb_num") ) resid2_ = protocols::RosettaScripts::get_resnum(tag, pose, "res2_");
	if (resid1_==0 || resid2_==0){//ligand
    protocols::ligand_docking::LigandBaseProtocol ligdock;
    if (resid1_==0) resid1_ = ligdock.get_ligand_id(pose);
    if (resid2_==0) resid2_ = ligdock.get_ligand_id(pose);
  }
	aname1_  =  tag->getOption<std::string>("atomname1", "CA" );
	aname2_  =  tag->getOption<std::string>("atomname2", "CA" );
	sample_type_  =  tag->getOption<std::string>("sample_type", "more" ); //a1 is more buried than a2 i.e. dsasa is more
	runtime_assert(resid1_>0 && resid1_<=pose.total_residue() );
	runtime_assert(resid2_>0 && resid2_<=pose.total_residue() );
	runtime_assert (pose.residue( resid1_ ).has( aname1_ ));
	runtime_assert (pose.residue( resid2_ ).has( aname2_ ));
 	runtime_assert( sample_type_=="more" || sample_type_ == "less" );
  TR<<" Defined LigDSasaFilter "<< std::endl;
}

DiffAtomSasaFilter::~DiffAtomSasaFilter() {}

bool
LigBurialFilter::apply( core::pose::Pose const & pose ) const {
	core::Size const count_neighbors( compute( pose ) );

	TR<<"Number of interface neighbors of ligand is "<<count_neighbors<<std::endl;
	return( count_neighbors >= neighbors_ );
}

void
LigBurialFilter::report( std::ostream & out, core::pose::Pose const & pose ) const {
	core::Size const count_neighbors( compute( pose ) );

	out<<"Number of interface neighbors of residue is "<<count_neighbors<<'\n';
}

core::Real
LigBurialFilter::report_sm( core::pose::Pose const & pose ) const {
	core::Size const count_neighbors( compute( pose ) );

	return( count_neighbors );
}

/// @details counts the number of residues contacting the ligand
core::Size
LigBurialFilter::compute( core::pose::Pose const & pose ) const {

	core::Size real_lig_id (lig_id_);
  if (real_lig_id==0) {
  	protocols::ligand_docking::LigandBaseProtocol ligdock;
		real_lig_id = ligdock.get_ligand_id(pose);
		TR<<"Calculating neighbors of Ligand resid is " << real_lig_id << std::endl;
	}
	core::Size count_neighbors( 0 );
	core::conformation::Residue const res_target( pose.conformation().residue( real_lig_id ) );
	for( core::Size i=1; i<=pose.total_residue(); ++i ){
		core::conformation::Residue const resi( pose.residue( i ) );
		core::Real const distance( resi.xyz( resi.nbr_atom() ).distance( res_target.xyz( res_target.nbr_atom() ) ) );
		if( distance <= distance_threshold_ ) ++count_neighbors;
	}
	return( count_neighbors);
}

/// @details: this filter basically works exactly as ResidueBurialFilter, but with the advantage that it has the capability to
/// @details: figure out resid of the ligand
void
LigBurialFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const &)
{
	lig_id_ =  tag->getOption<core::Size>( "lig_id", 0 );
  distance_threshold_ = tag->getOption<core::Real>( "distance", 8.0 );
  neighbors_ = tag->getOption<core::Size>( "neighbors", 1 );

  TR<<"LigBurialFilter with distance threshold of "<<distance_threshold_<<" around residue "<<lig_id_<<" with "<<neighbors_<<" neighbors."<<std::endl;
	TR.flush();
}

LigBurialFilter::~LigBurialFilter() {}

LigInterfaceEnergyFilter::LigInterfaceEnergyFilter( core::scoring::ScoreFunctionOP scorefxn, core::Real const threshold, bool const include_cstE, core::Size const rb_jump, core::Real const interface_distance_cutoff ) : Filter( "LigInterfaceEnergy" ), threshold_( threshold ), 	include_cstE_ ( include_cstE ), rb_jump_ ( rb_jump ), interface_distance_cutoff_ ( interface_distance_cutoff )  {

		using namespace core::scoring;

		if( scorefxn ) scorefxn_ = new core::scoring::ScoreFunction( *scorefxn );
		if (!include_cstE) enzutil::enable_constraint_scoreterms( scorefxn_);
	}

	LigInterfaceEnergyFilter::LigInterfaceEnergyFilter( LigInterfaceEnergyFilter const &init ) :
	Filter( init ), threshold_( init.threshold_ ),
	include_cstE_ (init.include_cstE_), rb_jump_ (init.rb_jump_), interface_distance_cutoff_ ( init.interface_distance_cutoff_){
		using namespace core::scoring;
		if( init.scorefxn_ ) scorefxn_ = new core::scoring::ScoreFunction( *init.scorefxn_ );
	}


bool
LigInterfaceEnergyFilter::apply( core::pose::Pose const & pose ) const
{
	using namespace core::scoring;

	if ( pose.conformation().num_chains() < 2 ) {
			TR << "pose must contain at least two chains!" << std::endl;
			return false;
	}
		else {
			TR<<" \n \t --------- computing  ---------- \n \t-------- ligand interface energies   --------\n \t \t------------------- \n" << std::endl;
			core::Real interf_E (compute(pose));
			return (interf_E < threshold_);
		}
}


void
LigInterfaceEnergyFilter::report( std::ostream & out, core::pose::Pose const & pose ) const

{
	using namespace core::scoring;
	using ObjexxFCL::FArray1D_bool;

		core::pose::Pose in_pose = pose;
		FArray1D_bool prot_res( in_pose.total_residue(), false );
		in_pose.fold_tree().partition_by_jump( rb_jump_, prot_res);
		core::conformation::Interface interface_obj(rb_jump_);
		in_pose.update_residue_neighbors();
		interface_obj.distance( interface_distance_cutoff_ );
		interface_obj.calculate( in_pose );
		(*scorefxn_)( in_pose );

		out<<"\n"<<A(9, "chain")<<A( 9, "res")<<A( 9, "AA")<<A( 9, "total")<<A( 9, "contact")<<A( 9, "fa_atr")<<A( 9, "fa_rep")<<A( 9, "hb_bb_sc")<<A( 9, "hb_sc")<<A( 9, "fa_sol")<<A( 9, "fa_dun")<<A( 9, "fa_pair")<<"\n";
		for ( core::Size resnum_ = 1; resnum_ <= pose.total_residue(); ++resnum_) {
//			if ( !in_pose.residue(resnum_).is_protein() ) continue;
			if( interface_obj.is_interface( resnum_ ) ) { // in interface

				Real total=in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( total_score ) ];
				if (!include_cstE_) total-= constraint_energy(in_pose, resnum_ );
				Real weighted_fa_atr=( (*scorefxn_)[ ScoreType( fa_atr) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( fa_atr ) ]);
				Real weighted_fa_rep=( (*scorefxn_)[ ScoreType( fa_rep) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( fa_rep) ]);
				Real weighted_hbond_bb_sc=( (*scorefxn_)[ ScoreType( hbond_bb_sc) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( hbond_bb_sc ) ]);
				Real weighted_hbond_sc=( (*scorefxn_)[ ScoreType( hbond_sc) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( hbond_sc ) ]);
				Real weighted_fa_sol=( (*scorefxn_)[ ScoreType( fa_sol) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( fa_sol ) ]);
				Real weighted_contact_score = weighted_fa_atr + weighted_fa_rep + weighted_hbond_bb_sc + weighted_hbond_sc + weighted_fa_sol;

				Real weighted_fa_dun=( (*scorefxn_)[ ScoreType( fa_dun) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( fa_dun ) ]);
				Real weighted_fa_pair=( (*scorefxn_)[ ScoreType( fa_pair ) ] ) * ( in_pose.energies().residue_total_energies( resnum_ )[ ScoreType( fa_pair ) ]);

				out<<A (9 , in_pose.pdb_info()->chain( resnum_) )<< I(9,0, in_pose.pdb_info()->number(resnum_))<<A (9,in_pose.residue( resnum_).name3())
				<<F (9 , 3, total) <<" "
				<<F (9 , 3, weighted_contact_score)<<" "
				<<F (9 , 3, weighted_fa_atr) <<" "
				<<F (9 , 3, weighted_fa_rep) <<" "
				<<F (9 , 3, weighted_hbond_bb_sc )<<" "
				<<F (9 , 3, weighted_hbond_sc )<<" "
				<<F (9 , 3, weighted_fa_sol )<<" "
				<<F (9 , 3, weighted_fa_dun )<<" "
				<<F (9 , 3, weighted_fa_pair )<<"\n";


			}
		}

}


core::Real
LigInterfaceEnergyFilter::report_sm( core::pose::Pose const & pose ) const
{
	using namespace core::scoring;

	core::Real const energy( compute( pose ) );
	return( energy );
}

core::Real
LigInterfaceEnergyFilter::compute( core::pose::Pose const & pose ) const
{
	using namespace core::scoring;
	using namespace core::pose::metrics;
	core::pose::Pose in_pose = pose;
	in_pose.update_residue_neighbors();
	(*scorefxn_)( in_pose );
	protocols::ligand_docking::LigandBaseProtocol ligdock;
  core::Size const lig_id = ligdock.get_ligand_id(pose);
	enzutil::enable_constraint_scoreterms( scorefxn_);
	core::util::MetricValue< core::Real > mv_interfE;
	in_pose.metric( "liginterfE","weighted_total", mv_interfE);  // if this seems like its come out of nowhere see declaration in protocols/jd2/DockDesignParser.cc
	core::Real weighted_score = mv_interfE.value();
	TR<<"Calculated Interface Energy is before cst correction is:"<<weighted_score<<std::endl;
	if (!include_cstE_) weighted_score -= 2*constraint_energy(in_pose, lig_id);
	TR<<"Calculated Interface Energy is "<<weighted_score<<std::endl;
	return( weighted_score );
}

core::Real
LigInterfaceEnergyFilter::constraint_energy( core::pose::Pose const & in_pose , int which_res ) const
{
 using namespace core::scoring;
  core::pose::Pose pose (in_pose);
  (*scorefxn_)(pose);
  EnergyMap all_weights = pose.energies().weights();
  EnergyMap scores;

  if( which_res == -1){ //means we want to know stuff for the whole pose
    scores = pose.energies().total_energies();
  }
  else{ scores = pose.energies().residue_total_energies( which_res ); }

  return scores[ coordinate_constraint ] * all_weights[ coordinate_constraint ] + scores[atom_pair_constraint] * all_weights[ atom_pair_constraint] +
    scores[ angle_constraint ] * all_weights[ angle_constraint ] + scores[ dihedral_constraint ] * all_weights[ dihedral_constraint ];

}
void
LigInterfaceEnergyFilter::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
  using namespace core::scoring;

  std::string const scorefxn_name( tag->getOption<std::string>( "scorefxn", "score12" ) );
  scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name ) ));
  threshold_ = tag->getOption<core::Real>( "energy_cutoff", 0.0 );
  include_cstE_ = tag->getOption<bool>( "include_cstE" , 0 );
  rb_jump_ = tag->getOption<core::Size>( "jump_number", pose.num_jump() ); //assume ligand is connected by last jump
  interface_distance_cutoff_ = tag->getOption<core::Real>( "interface_distance_cutoff" , 8.0 );

}

LigInterfaceEnergyFilter::~LigInterfaceEnergyFilter() {}

EnzScoreFilter::EnzScoreFilter( core::Size const resnum, std::string const cstid, core::scoring::ScoreFunctionOP scorefxn, core::scoring::ScoreType const score_type, core::Real const threshold, bool const whole_pose, bool const is_cstE ) : Filter( "EnzScore" ), resnum_( resnum ), cstid_( cstid ), score_type_( score_type ), threshold_( threshold ),  whole_pose_ ( whole_pose ), is_cstE_ ( is_cstE ) {

    using namespace core::scoring;

    if( scorefxn ) scorefxn_ = new core::scoring::ScoreFunction( *scorefxn );
		//if(score_type_ == total_score || is_cstE_ ) enzutil::enable_constraint_scoreterms(scorefxn_);
    if( score_type_ != total_score ) {
      core::Real const old_weight( scorefxn_->get_weight( score_type_ ) );
      scorefxn_->reset();
      scorefxn_->set_weight( score_type_, old_weight );
    }
  }

  EnzScoreFilter::EnzScoreFilter( EnzScoreFilter const &init ) :
  Filter( init ), resnum_( init.resnum_ ), cstid_( init.cstid_ ),score_type_( init.score_type_ ), threshold_( init.threshold_ ),
  whole_pose_ (init.whole_pose_), is_cstE_ ( init.is_cstE_ ) {
    using namespace core::scoring;
    if( init.scorefxn_ ) scorefxn_ = new core::scoring::ScoreFunction( *init.scorefxn_ );
  }

bool
EnzScoreFilter::apply( core::pose::Pose const & pose ) const
{
  using namespace core::scoring;
	core::Real energy (compute(pose));
	TR<<"Score found as "<<energy <<" and threshold is " << threshold_<<std::endl;
  return (energy<threshold_);
}

void
EnzScoreFilter::report( std::ostream & out, core::pose::Pose const & pose ) const

{
  using namespace core::scoring;
	out<<"Weighted score "<<compute( pose )<<'\n';
}

core::Real
EnzScoreFilter::report_sm( core::pose::Pose const & pose ) const {
  return( compute( pose ) );
}

core::Real
EnzScoreFilter::compute( core::pose::Pose const & pose ) const {
	using namespace core::pose;
  using namespace core::scoring;
	PoseOP in_pose = new Pose( pose );
	core::Size resnum = resnum_;
	if (resnum==0){//means we want ligand scores
		protocols::ligand_docking::LigandBaseProtocol ligdock;
  	resnum = ligdock.get_ligand_id(pose);
	}
	core::Real weight(0.0), score(0.0);
	if (whole_pose_){
		(*scorefxn_)( *in_pose );
  	if (!is_cstE_ ) {
			weight = ( (*scorefxn_)[ ScoreType( score_type_ ) ] );
  		score = ( in_pose->energies().total_energies()[ ScoreType( score_type_ ) ]);
  		TR<<"Raw Score is: " << score << std::endl;
			}
		if( score_type_  == total_score ) {
			if (pose.constraint_set()->has_constraints()) {//check if constraints exist, and remove their score
				enzutil::enable_constraint_scoreterms(scorefxn_);
				 (*scorefxn_)( *in_pose );
				 LigInterfaceEnergyFilter liginter( scorefxn_,0.0,false, 1, 0.0 );
				 core::Real all_cstE = liginter.constraint_energy(*in_pose, -1);// for whole pose;
				score -= all_cstE;
				}
			return( score );
			}
		else if (is_cstE_){
				TR<<"Evaluating constraint score for whole pose..."<<std::endl;
				enzutil::enable_constraint_scoreterms(scorefxn_);
				 LigInterfaceEnergyFilter liginter( scorefxn_,0.0,false, 1, 0.0 );
        core::Real all_cstE = liginter.constraint_energy(*in_pose, -1); // for whole pose;
				TR<<"And constraint energy is..."<< all_cstE << std::endl;
				return( all_cstE ) ;
		}
		else {
			core::Real const weighted_score( weight * score );
  	  TR<<"Weighted Score is: " << weighted_score << std::endl;
  		return( weighted_score );
		}
	}
	else{ //score for a particular residue/cstid
  	if (! (cstid_ =="")) resnum = enzutil::get_resnum_from_cstid( cstid_, *in_pose );
    (*scorefxn_)( *in_pose );
		TR<<"For Resid:"<< resnum << std::endl;
		runtime_assert(resnum>0);
		if (is_cstE_ || score_type_ == total_score ){
		 	enzutil::enable_constraint_scoreterms(scorefxn_);
      (*scorefxn_)( *in_pose );
       LigInterfaceEnergyFilter liginter( scorefxn_,0.0,false, 1, 0.0 );
       core::Real resnum_cstE = liginter.constraint_energy(*in_pose, resnum);
       if (is_cstE_) return resnum_cstE;
			 else return (in_pose->energies().residue_total_energies( resnum )[ ScoreType( score_type_ )] - resnum_cstE );
		}
		else return( ( (*scorefxn_)[ ScoreType( score_type_ ) ] ) * (in_pose->energies().residue_total_energies( resnum )[ ScoreType( score_type_ )]) );
	}
	return 0.0;
}

void
EnzScoreFilter::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &, Movers_map const &, core::pose::Pose const &pose)
{
  using namespace core::scoring;
	is_cstE_ = false;
	if (tag->hasOption( "pdb_num" )) resnum_ = protocols::RosettaScripts::get_resnum(tag, pose);
	else if (tag->hasOption( "res_num" )) resnum_ =  tag->getOption<core::Size>( "res_num", 0 );
	cstid_ = tag->getOption<std::string>( "cstid", "" );

	std::string const scorefxn_name( tag->getOption<std::string>( "scorefxn", "score12" ) );
  scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name ) ));
	std::string sco_name = tag->getOption<std::string>( "score_type", "total_score" );
	if (sco_name == "cstE") {
		is_cstE_ = true;
		score_type_=atom_pair_constraint; //dummy assignment, never actually used.
	}
	else score_type_ = core::scoring::score_type_from_name( sco_name );
  threshold_ = tag->getOption<core::Real>( "energy_cutoff", 0.0 );
  whole_pose_ = tag->getOption<bool>( "whole_pose" , 0 );

	//check to make sure one and only one of resnum, cstid and whole_pose are specified
	runtime_assert(tag->hasOption( "res_num" )|| tag->hasOption( "pdb_num" ) || tag->hasOption( "cstid" ) || whole_pose_==1 );
	runtime_assert(!( (tag->hasOption( "res_num" )|| tag->hasOption( "pdb_num" )) &&  tag->hasOption( "cstid" )));
	runtime_assert(!( (tag->hasOption( "res_num" )|| tag->hasOption( "pdb_num" )) &&  whole_pose_==1));
	runtime_assert(!(tag->hasOption( "cstid" ) &&  whole_pose_==1));

	if (whole_pose_==1 ) {
    TR<<"energies for whole pose will be calculated "
    << "\n and scorefxn " <<scorefxn_name <<" will be used" <<std::endl;
  }
  else {
    TR<<"EnzScoreFilter for residue or cstid with cutoff "<<threshold_<<std::endl;
    }
}
EnzScoreFilter::~EnzScoreFilter() {}

RepackWithoutLigandFilter::RepackWithoutLigandFilter( core::scoring::ScoreFunctionOP scorefxn, core::Real rms_thresh, core::Real energy_thresh, utility::vector1< core::Size > rms_target_res  ) : Filter( "RepackWithoutLigand" ), scorefxn_(scorefxn), rms_threshold_( rms_thresh ), energy_threshold_( energy_thresh ), target_res_( rms_target_res ) {}

bool
RepackWithoutLigandFilter::apply( core::pose::Pose const & pose ) const
{

	core::Real const value( compute( pose ) );

	if (calc_dE_) {
		if (value > energy_threshold_) return false;
		else return true;
		}
	else if (calc_rms_){
		if (value > rms_threshold_) return false;
		else return true;
    }
	return false;
}

void
RepackWithoutLigandFilter::report( std::ostream & out, core::pose::Pose const & pose ) const
{
 //Output to tracer triggers the compute function separately. Since this is an expensive filter, we don't want to unncessarily compute, hence disabled.
 // Dummy use of pose to avoid compiler warning.
//  core::Real value( compute( pose ));
//  if (calc_rms_) out<<"RMS= "<< value<<'\n';
//  else out<<"dEnergy = "<< value<<'\n';
	out<<"Ligand to take out is: "<< pose.total_residue();
	out<<'\n';
}

core::Real
RepackWithoutLigandFilter::report_sm( core::pose::Pose const & pose ) const
{
  return ( compute( pose ));
}

core::Real
RepackWithoutLigandFilter::compute( core::pose::Pose const & pose ) const
{
	core::Real value(10000.);
	core::pose::Pose rnl_pose = pose;
	enzutil::disable_constraint_scoreterms(scorefxn_);
	(*scorefxn_)(rnl_pose) ;
	core::Real wl_score = rnl_pose.energies().total_energies()[total_score];

	enzutil::remove_all_enzdes_constraints( rnl_pose );
	protocols::enzdes::RepackLigandSiteWithoutLigandMover rnl( scorefxn_, false );
	rnl.apply( rnl_pose );
	(*scorefxn_)(rnl_pose);
	core::Real nl_score = rnl_pose.energies().total_energies()[total_score];

	if (calc_dE_) {
		TR<<"Total energy with ligand is: "<<wl_score<<" and total energy without ligand is "<<nl_score<<std::endl;
		return (wl_score - nl_score);
	}
	else if (calc_rms_){
		ObjexxFCL::FArray1D_bool rms_seqpos( pose.total_residue(), false );
		utility::vector1< core::Size > trg_res;
		if (rms_all_rpked_) {
		TR<<"Getting identities of all pack residues... "<< std::endl;
			core::pack::task::PackerTaskCOP rnl_ptask = rnl.get_ptask();
			for( core::Size i = 1; i <= rnl_ptask->total_residue(); ++i ){
      	if( rnl_ptask->residue_task( i ).being_packed() && pose.residue( i ).is_protein() ) {
					trg_res.push_back( i );
				}
    	}
		}
		else if( use_cstids_ ) {
			enzutil::get_resnum_from_cstid_list(cstid_list_, pose, trg_res);
		}
		trg_res.insert( trg_res.begin(), target_res_.begin(), target_res_.end() );
  	std::unique( trg_res.begin(), trg_res.end() );

		TR<<"Calculating RMS for residues ";
		for (core::Size i=1; i<=pose.total_residue(); ++i){
			if (! (pose.residue_type( i ).is_ligand()) ){
				utility::vector1< core::Size >::const_iterator resfind = find( trg_res.begin(), trg_res.end(), i );
				if (resfind != trg_res.end()) {
					rms_seqpos( i ) =true;
					TR<< i<< ", ";
				}
			}
		}
	  TR<< std::endl;
		core::Real rmsd( core::scoring::rmsd_no_super_subset( pose, rnl_pose, rms_seqpos, core::scoring::is_protein_sidechain_heavyatom ) );
		TR<<"Total rms of requested region is: "<< rmsd <<std::endl;
		return rmsd;
	}
	return value; //should not get here
}

void
RepackWithoutLigandFilter::parse_my_tag( TagPtr const tag, DataMap &data, Filters_map const &, Movers_map const &, core::pose::Pose const &pose )
{
	TR<<" Defining RepackWithoutLigandFilter "<< std::endl;
	std::string const scorefxn_name( tag->getOption<std::string>( "scorefxn", "score12" ) );
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name ) ));

	calc_rms_ = false; calc_dE_ = false; rms_all_rpked_ = false; use_cstids_ = false;
	runtime_assert( tag->hasOption("energy_threshold") ||  tag->hasOption("rms_threshold") );
	if (tag->hasOption("rms_threshold")) {
		rms_threshold_ =  tag->getOption<core::Real>("rms_threshold", 0.5 );
		calc_rms_ = true;
		if( tag->hasOption("target_res") ) {
			std::string target_res = tag->getOption<std::string>("target_res", "" );
			if (target_res=="all_repacked") rms_all_rpked_ = true;
			else target_res_ = protocols::RosettaScripts::get_resnum_list(tag, "target_res",pose);
		}
		if( tag->hasOption("target_cstids") ) {
			cstid_list_ = tag->getOption<std::string>("target_cstids", "" );
			use_cstids_ = true;
		}
	}
	else if (tag->hasOption("energy_threshold")) {
		energy_threshold_  =  tag->getOption<core::Real>("energy_threshold", 0.0 );
		calc_dE_ = true;
	}
	TR<<" Defined RepackWithoutLigandFilter "<< std::endl;
}

RepackWithoutLigandFilter::~RepackWithoutLigandFilter() {}

EnzdesScorefileFilter::EnzdesScorefileFilter()
	: Filter(),
	no_packstat_calc_( core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ),
	native_comparison_(core::options::option[core::options::OptionKeys::enzdes::compare_native].user() ),
	repack_no_lig_(core::options::option[core::options::OptionKeys::enzdes::final_repack_without_ligand]),
	keep_rnl_pose_(core::options::option[core::options::OptionKeys::enzdes::dump_final_repack_without_ligand_pdb]),
	rnl_pose_(NULL),
	sfxn_(core::scoring::ScoreFunctionFactory::create_score_function("enzdes") ),
	enzcst_io_(NULL),
	native_comp_(NULL)
{
	if( native_comparison_ ) native_comp_ = new protocols::enzdes::DesignVsNativeComparison();
	residue_calculators_.clear();
	native_compare_calculators_.clear();
	silent_Es_.clear();
	relevant_scoreterms_.clear();
	spec_segments_.clear();

	relevant_scoreterms_.push_back( "total_score" );
	relevant_scoreterms_.push_back( "fa_rep" );
	relevant_scoreterms_.push_back( "hbond_sc" );
	relevant_scoreterms_.push_back( "all_cst" );
}

EnzdesScorefileFilter::EnzdesScorefileFilter( EnzdesScorefileFilter const & other )
: Filter( other ),
	no_packstat_calc_( other.no_packstat_calc_ ),
	native_comparison_(other.native_comparison_ ),
	repack_no_lig_( other.repack_no_lig_),
	keep_rnl_pose_( other.keep_rnl_pose_),
	rnl_pose_( other.rnl_pose_),
	sfxn_( other.sfxn_->clone() ),
	enzcst_io_( other.enzcst_io_),
	residue_calculators_(other.residue_calculators_),
	native_compare_calculators_(other.native_compare_calculators_),
	native_comp_(other.native_comp_),
	silent_Es_(other.silent_Es_),
	relevant_scoreterms_(other.relevant_scoreterms_),
	spec_segments_(other.spec_segments_)
{}


EnzdesScorefileFilter::~EnzdesScorefileFilter(){}

/// @details not implemented yet
bool
EnzdesScorefileFilter::apply( core::pose::Pose const & ) const{
	return true;
}

/// @details not implemented yet
void
EnzdesScorefileFilter::parse_my_tag( utility::Tag::TagPtr const , protocols::moves::DataMap &, Filters_map const &, protocols::moves::Movers_map const &, core::pose::Pose const & )
{
}

void
EnzdesScorefileFilter::set_cstio(
	toolbox::match_enzdes_util::EnzConstraintIOCOP enzcst_io )
{
	enzcst_io_ = enzcst_io;
}

void
EnzdesScorefileFilter::examine_pose(
	core::pose::Pose const & pose )
{
	using namespace core::io::silent;

	silent_Es_.clear();
	//flo jan 2010 with the new cst cache, some calcs that take the
	//ligand out of the pose will crash. for now, let's make a copy
	//of the pose and remove the cst cache from it
	core::pose::Pose calc_pose = pose;
	get_enzdes_observer( calc_pose )->set_cst_cache( NULL );
	(*sfxn_)( calc_pose );

	if( pose.observer_cache().has( core::pose::datacache::CacheableObserverType::SPECIAL_SEGMENTS_OBSERVER) ){
		spec_segments_ = utility::pointer::static_pointer_cast< core::util::datacache::SpecialSegmentsObserver const >(pose.observer_cache().get_const_ptr( core::pose::datacache::CacheableObserverType::SPECIAL_SEGMENTS_OBSERVER ) )->segments();
	}

	setup_pose_metric_calculators( pose );

	bool separate_out_constraints = false;
	utility::vector1< std::string >::const_iterator cstfind = find( relevant_scoreterms_.begin(), relevant_scoreterms_.end(),"all_cst");
	if( cstfind != relevant_scoreterms_.end() ) separate_out_constraints = true;

  //first write out the relevant score terms for the pose total
  for( utility::vector1< std::string >::const_iterator sco_it = relevant_scoreterms_.begin();
       sco_it != relevant_scoreterms_.end(); ++sco_it ){
    std::string sco_name = *sco_it;
    int width = std::max( 10, (int) sco_name.length() + 3 );

    SilentEnergy new_se;
    if( *sco_it == "all_cst" ) {
      new_se = SilentEnergy ( sco_name, enzutil::sum_constraint_scoreterms(pose, -1 ), 1, width);
    }
		else if( separate_out_constraints && ( *sco_it == "total_score" ) ){
			core::Real desired_value = pose.energies().total_energies()[ core::scoring::score_type_from_name( *sco_it ) ] - enzutil::sum_constraint_scoreterms(pose, -1 );
			new_se = SilentEnergy(sco_name, desired_value, 1, width);
		}
    else{
      new_se = SilentEnergy ( sco_name, pose.energies().total_energies()[ core::scoring::score_type_from_name( *sco_it ) ] * pose.energies().weights()[ core::scoring::score_type_from_name( *sco_it ) ], 1 ,width);
    }
    silent_Es_.push_back( new_se );

  }
	//pose metric calculators for pose total
	std::map< Size, utility::vector1< std::pair< std::string, std::string > > >::const_iterator totcalc_it = residue_calculators_.find( 0 );
	if( totcalc_it != residue_calculators_.end() ){

    utility::vector1< std::pair< std::string, std::string > > const & tot_calculators = totcalc_it->second;
		for( utility::vector1< std::pair< std::string, std::string > >::const_iterator calc_it = tot_calculators.begin();
				 calc_it != tot_calculators.end(); ++calc_it){

			std::string calc_name = "tot_" + calc_it->first;
			int width = std::max( 10, (int) calc_name.length() + 3 );

			core::Real calc_value;

			//following lines fairly hacky, but don't know a better solution at the moment
			if( calc_it->first == "hbond_pm" || calc_it->first == "burunsat_pm" || calc_it->first == "NLconts_pm" ){
				core::util::MetricValue< core::Size > mval_size;
				calc_pose.metric( calc_it->first, calc_it->second, mval_size );
				calc_value = mval_size.value();
			} else if (calc_it->first == "seq_recovery") {
				if ( get_enzdes_observer( calc_pose ) -> get_seq_recovery_cache() ) {
					calc_value = get_enzdes_observer( calc_pose ) -> get_seq_recovery_cache() -> sequence_recovery( calc_pose );
				} else {
					calc_value= 0.0;
				}
			}	else {
				core::util::MetricValue< core::Real > mval_real;
				calc_pose.metric( calc_it->first, calc_it->second, mval_real );
				calc_value = mval_real.value();
			}

			SilentEnergy new_se( calc_name, calc_value, 1, width);
			silent_Es_.push_back( new_se );
		}

	}

  //then write out the relevant scoreterms (and potentially pose metrics) for each of the special residues
	Size spec_res_counter(0);
	utility::vector1< Size > special_res = enzutil::catalytic_res( pose );

  for( utility::vector1<Size>::const_iterator res_it = special_res.begin(); res_it != special_res.end(); res_it++ ){

    spec_res_counter++;
    //for convenience, the sequence number of the residue will be written out
    std::stringstream temp;
    temp << spec_res_counter;
    std::string spec_res_name = "SR_" + temp.str();
    //std::cerr << "name for res " << *res_it << " is " << spec_res_name ;
    SilentEnergy res_name(spec_res_name, *res_it, 1, 10);
    silent_Es_.push_back( res_name );

		utility::vector1< core::Size > dummy_vec;
		dummy_vec.push_back( *res_it );
		compute_metrics_for_residue_subset( spec_res_name, separate_out_constraints, calc_pose, dummy_vec);
	}

	for( core::Size specseg(1); specseg <= spec_segments_.size(); ++specseg ){
		std::string segname( "Seg_" + utility::to_string( specseg ) );
		//SilentEnergy segname_e( segname, specseg, 1, 10 );
		//silent_Es_.push_back( segname_e );
		utility::vector1< core::Size > segment;
		for( core::Size i = spec_segments_[specseg].first; i <= spec_segments_[specseg].second; ++i ) segment.push_back( i );
		compute_metrics_for_residue_subset( segname, separate_out_constraints, calc_pose, segment );
	}

	//if comparison to native is requested
	if( native_comparison_ ){
		native_comp_->compare_to_native( calc_pose, native_compare_calculators_, sfxn_, silent_Es_ );
	}

	//if repack without lig requested
	if( repack_no_lig_ ){

		core::pose::PoseOP rnlpose = new core::pose::Pose( pose );
		protocols::enzdes::RepackLigandSiteWithoutLigandMover rnl_mov( sfxn_, true );
		if( enzcst_io_ ) rnl_mov.set_cstio( enzcst_io_ );
		rnl_mov.apply( *rnlpose );
		for( core::Size i =1; i<= rnl_mov.silent_Es().size(); ++i ) silent_Es_.push_back( rnl_mov.silent_Es()[i] );

		if( keep_rnl_pose_ ) rnl_pose_ = rnlpose;
	}
} //examine pose


void
EnzdesScorefileFilter::compute_metrics_for_residue_subset(
	std::string sub_name,
	bool separate_out_constraints,
	core::pose::Pose const & calc_pose,
	utility::vector1< core::Size > const & res_subset )
{

	using namespace core::io::silent;

	for( utility::vector1< std::string >::const_iterator sco_it = relevant_scoreterms_.begin();
       sco_it != relevant_scoreterms_.end(); ++sco_it ){

		std::string sco_name = sub_name + "_" +  *sco_it ;
		int width = std::max( 10, (int) sco_name.length() + 3 );

		SilentEnergy new_se;
		if( *sco_it == "all_cst" ) {
			core::Real value(0.0);
			for( core::Size ii =1; ii <= res_subset.size(); ++ii ) value += enzutil::sum_constraint_scoreterms(calc_pose, res_subset[ii]);
			new_se = SilentEnergy ( sco_name, value , 1, width);
		}
		else if( separate_out_constraints && ( *sco_it == "total_score" ) ){
			core::Real desired_value(0.0);
			for( core::Size ii =1; ii <= res_subset.size(); ++ii ) desired_value +=	calc_pose.energies().residue_total_energies( res_subset[ii] )[ core::scoring::score_type_from_name( *sco_it ) ] - enzutil::sum_constraint_scoreterms(calc_pose, res_subset[ii] );
			new_se = SilentEnergy(sco_name, desired_value, 1, width);
		}
		else{
			core::Real value(0.0);
			for( core::Size ii =1; ii <= res_subset.size(); ++ii ) value += calc_pose.energies().residue_total_energies( res_subset[ii] )[ core::scoring::score_type_from_name( *sco_it ) ];
			value *= calc_pose.energies().weights()[ core::scoring::score_type_from_name( *sco_it ) ];
			new_se = SilentEnergy ( sco_name, value, 1 ,width);
		}

		silent_Es_.push_back( new_se );
	}//loop over relevant scoreterms


	//if there are calculators that need to be evaluated for this residue, let's do that now
	//note: section still under development, right now only calculators that return reals are supported
	std::map< Size, utility::vector1< std::pair< std::string, std::string > > >::const_iterator res_calc_it = residue_calculators_.find( res_subset[1] );
	if( res_calc_it != residue_calculators_.end() ){

		utility::vector1< std::pair< std::string, std::string > > calculators_this_res = res_calc_it->second;
		for( utility::vector1< std::pair< std::string, std::string > >::iterator calc_it = calculators_this_res.begin();
				 calc_it != calculators_this_res.end(); calc_it++ ){

			std::string res_calc_name = sub_name + "_" + calc_it->first;
			int width = std::max( 10, (int) res_calc_name.length() + 3 );

			core::Real calc_value(0.0);

			core::util::MetricValue< core::Real > mval_real;
			core::util::MetricValue< utility::vector1< core::Size > >mval_sizevec;
			core::util::MetricValue< utility::vector1< core::Real > >mval_realvec;

			//following lines fairly hacky, but don't know a better solution at the moment
			if( ( calc_it->first == "hbond_pm") || ( calc_it->first == "burunsat_pm") || ( calc_it->first == "NLconts_pm" ) ){
				calc_pose.metric( calc_it->first, calc_it->second, mval_sizevec );
				for( core::Size ii =1; ii <= res_subset.size(); ++ii ) calc_value += mval_sizevec.value()[ res_subset[ii] ];
			}
			else if( calc_it->first == "nlsurfaceE_pm" ){
				calc_pose.metric( calc_it->first, calc_it->second, mval_realvec );
				for( core::Size ii =1; ii <= res_subset.size(); ++ii ) calc_value += mval_realvec.value()[ res_subset[ii] ];
			}
			else if( (calc_it->first == "pstat_pm") || (calc_it->first == "nlpstat_pm" ) ) {
				calc_pose.metric( calc_it->first, calc_it->second, mval_realvec );
				core::Real pstat_sum(0.0);
				for( core::Size ii =1; ii <= res_subset.size(); ++ii ) pstat_sum += mval_realvec.value()[ res_subset[ii] ];
				calc_value = pstat_sum / res_subset.size();
			}
			else {
				calc_pose.metric( calc_it->first, calc_it->second, mval_real );
				//for( core::Size ii =1; ii <= res_subset.size(); ++ii ) calc_value += mval_realvec.value()[ res_subset[ii] ];
				calc_value = mval_real.value();
			}
			//std::cerr << " hehe, just executed pose metric for " << calc_it->first << " calculator   ";


			//special case: if this is an interface calculation, we do not want to include constraint terms
			//hacky at the moment, haven't figured out yet how to do this really clean
			if( separate_out_constraints && ( calc_it->first == "interf_E_1_2") ){
				core::Real cst_val(0.0);
				for( core::Size ii =1; ii <= res_subset.size(); ++ii ) cst_val += enzutil::sum_constraint_scoreterms(calc_pose, res_subset[ii]);
				calc_value = calc_value - ( 2 * cst_val );
			}
			SilentEnergy new_se( res_calc_name, calc_value, 1, width);
			silent_Es_.push_back( new_se );

		} // for calculators this res

	}// if calculators for this res
    //else std::cerr << "resi " << *res_it << " has no calcs." << std::endl;
} //compute_metrics_for_residue_subset


/// @brief function to setup residue specific calculators for all the catalytic residues and the ligand
void
EnzdesScorefileFilter::setup_pose_metric_calculators( core::pose::Pose const & pose )
{

	using namespace core::pose::metrics;
	residue_calculators_.clear();
	native_compare_calculators_.clear();

	//calculators setup by this function ( under development )
	//1. packstat for the complete pose (with and without ligand)
	//2. SASA and interface delta for ligand
	//3. number of Hbonds for catalytic residues + ligand
	//4. number of buried polars for catalytic residues + ligand
	//5. packstat for every catalytic residue
	//6. further down the road maybe some stuff to compare native and designed structure

	std::string hbond_calc_name = "hbond_pm";
	std::string burunsat_calc_name = "burunsat_pm";
	std::string packstat_calc_name = "pstat_pm";
	std::string noligpackstat_calc_name = "nlpstat_pm";
	std::string nonlocalcontacts_calc_name = "NLconts_pm";
	std::string surface_calc_name = "nlsurfaceE_pm";

	//before starting, we make sure that all necessary calculators are instantiated
	if( !CalculatorFactory::Instance().check_calculator_exists( hbond_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP hb_calc = new protocols::toolbox::PoseMetricCalculators::NumberHBondsCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( hbond_calc_name, hb_calc );
	}
	if( !CalculatorFactory::Instance().check_calculator_exists( burunsat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP burunsat_calc = new protocols::toolbox::PoseMetricCalculators::BuriedUnsatisfiedPolarsCalculator("default", hbond_calc_name);
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( burunsat_calc_name, burunsat_calc );
	}
	if( !CalculatorFactory::Instance().check_calculator_exists( packstat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP pstat_calc = new protocols::toolbox::PoseMetricCalculators::PackstatCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( packstat_calc_name, pstat_calc );
	}
	if( !CalculatorFactory::Instance().check_calculator_exists( noligpackstat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP noligpstat_calc = new protocols::toolbox::PoseMetricCalculators::PackstatCalculator( true );
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( noligpackstat_calc_name, noligpstat_calc );
	}
	if( !CalculatorFactory::Instance().check_calculator_exists( nonlocalcontacts_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP nlcontacts_calc = new protocols::toolbox::PoseMetricCalculators::NonlocalContactsCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( nonlocalcontacts_calc_name, nlcontacts_calc );
	}
	if( !CalculatorFactory::Instance().check_calculator_exists( surface_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP surface_calc = new protocols::toolbox::PoseMetricCalculators::SurfaceCalculator( true );
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( surface_calc_name, surface_calc );
	}


	//first general pose calculators

	utility::vector1< std::pair< std::string, std::string > > total_pose_calculators;

	if( !no_packstat_calc_ ){
		total_pose_calculators.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "total_packstat") );
		total_pose_calculators.push_back( std::pair< std::string, std::string > ( noligpackstat_calc_name, "total_packstat") );
	}

	total_pose_calculators.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "all_bur_unsat_polars") );
	total_pose_calculators.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "all_Hbonds") );
	total_pose_calculators.push_back( std::pair< std::string, std::string > ( nonlocalcontacts_calc_name, "total_nlcontacts") );
	total_pose_calculators.push_back( std::pair< std::string, std::string > ( surface_calc_name, "total_surface") );
  total_pose_calculators.push_back( std::pair< std::string, std::string > ( "seq_recovery", "seq_recovery") );

	residue_calculators_.insert( std::pair< Size, utility::vector1< std::pair< std::string, std::string > > > ( 0, total_pose_calculators ) );

	//if native compare is requested
	if( native_comparison_ ){
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "all_bur_unsat_polars") );
		if( !core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ){
			native_compare_calculators_.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "total_packstat") );
		}
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "all_Hbonds") );
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( nonlocalcontacts_calc_name, "total_nlcontacts") );
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( surface_calc_name, "total_surface") );
	}

	//general pose calculators done, now onto the residue specific calculators
	//note: all residue calculators will also be used to calculate stuff for the special segments
	//they'll be added to the map under the seqpos of the first residue in the
	//special segment
	std::set< core::Size > spec_seg_first_res;
	for( core::Size speccount =1; speccount <= spec_segments_.size(); ++speccount ) spec_seg_first_res.insert( spec_segments_[speccount].first );
	for( core::Size i = 1;  i<= pose.total_residue(); ++i){

		if( ! (enzutil::is_catalytic_seqpos( pose, i ) || (spec_seg_first_res.find( i ) != spec_seg_first_res.end()) ) ) continue;

		utility::vector1< std::pair< std::string, std::string > > calculators_this_res;

		//first a couple of ligand specific calculators ( interface SASA and interface energetics)
		if( pose.residue_type( i ).is_ligand() ){

			if( pose.conformation().num_chains() != 2){
				std::cerr << "WARNING: the input pose has more than two chains, interface score reported in scorefile might not be accurate." << std::endl;
			}
			Size lig_chain = pose.chain( i );
			Size prot_chain = pose.chain( 1 );   //we're making the not so wild assumption that the the first residue in the pose belongs to the protein
			std::string lig_ch_string = utility::to_string( lig_chain );
			std::string prot_ch_string = utility::to_string( prot_chain );
			if( lig_chain == prot_chain ) { utility_exit_with_message( "WTF?!? ligand and residue 1 are on the same chain... " );}

			std::string lig_interface_neighbor_calc_name = "neighbor_def_" + prot_ch_string + "_" + lig_ch_string;
			std::string lig_dsasa_calc_name = "dsasa_" + prot_ch_string + "_" + lig_ch_string;
			std::string lig_interface_e_calc_name = "interf_E_" + prot_ch_string + "_" + lig_ch_string;

			if( !CalculatorFactory::Instance().check_calculator_exists( lig_interface_neighbor_calc_name ) ){
				PoseMetricCalculatorOP lig_neighbor_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceNeighborDefinitionCalculator( prot_chain, lig_chain );
				CalculatorFactory::Instance().register_calculator( lig_interface_neighbor_calc_name, lig_neighbor_calc );
			}
			if( !CalculatorFactory::Instance().check_calculator_exists( lig_dsasa_calc_name ) ){
				PoseMetricCalculatorOP lig_dsasa_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceSasaDefinitionCalculator( prot_chain, lig_chain );
				CalculatorFactory::Instance().register_calculator( lig_dsasa_calc_name, lig_dsasa_calc );
			}
			if( !CalculatorFactory::Instance().check_calculator_exists( lig_interface_e_calc_name ) ){
				PoseMetricCalculatorOP lig_interf_E_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceDeltaEnergeticsCalculator( lig_interface_neighbor_calc_name );
				CalculatorFactory::Instance().register_calculator( lig_interface_e_calc_name, lig_interf_E_calc );
			}

			calculators_this_res.push_back( std::pair< std::string, std::string > ( lig_interface_e_calc_name, "weighted_total") );
			calculators_this_res.push_back( std::pair< std::string, std::string > ( lig_dsasa_calc_name, "frac_ch2_dsasa") );

		} //ligand specific calculators set up

		//now calculators for every residue, for starters number of Hbonds and number of buried polars
		calculators_this_res.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "residue_Hbonds") );
		calculators_this_res.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "residue_bur_unsat_polars") );

		//and for protein residues, we also want to know the packstat
		if( pose.residue_type( i ).is_protein() && ( !core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ) ){
			calculators_this_res.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "residue_packstat") );
			calculators_this_res.push_back( std::pair< std::string, std::string > ( noligpackstat_calc_name, "residue_packstat") );

			//for loop residues also non local contacts and surface E
			if(spec_seg_first_res.find( i ) != spec_seg_first_res.end()){
				calculators_this_res.push_back( std::pair< std::string, std::string > ( nonlocalcontacts_calc_name, "residue_nlcontacts") );
				calculators_this_res.push_back( std::pair< std::string, std::string > ( surface_calc_name, "residue_surface") );
			}
		}
		//debug
		//for( utility::vector1< std::pair< std::string, std::string > >::iterator calc_it = calculators_this_res.begin();
		//		 calc_it != calculators_this_res.end(); calc_it++ ){
		//	std::cerr << "calculator " << calc_it->first << " created for residue " << *res_it << std::endl;
		//}
		//debug over

		residue_calculators_.insert( std::pair< Size, utility::vector1< std::pair< std::string, std::string > > > ( i, calculators_this_res ) );

	} // loop over all (catalytic) residues
} //setup_pose_metric_calculators


core::pose::PoseOP
EnzdesScorefileFilter::rnl_pose()
{
	return rnl_pose_;
}

/// @brief clear rnl pose to save some memory
void
EnzdesScorefileFilter::clear_rnl_pose()
{
	rnl_pose_ = NULL;
}

filters::FilterOP
DiffAtomSasaFilterCreator::create_filter() const { return new DiffAtomSasaFilter; }

std::string
DiffAtomSasaFilterCreator::keyname() const { return "DiffAtomBurial"; }

filters::FilterOP
EnzScoreFilterCreator::create_filter() const { return new EnzScoreFilter; }

std::string
EnzScoreFilterCreator::keyname() const { return "EnzScore"; }

filters::FilterOP
LigBurialFilterCreator::create_filter() const { return new LigBurialFilter; }

std::string
LigBurialFilterCreator::keyname() const { return "LigBurial"; }

filters::FilterOP
LigDSasaFilterCreator::create_filter() const { return new LigDSasaFilter; }

std::string
LigDSasaFilterCreator::keyname() const { return "DSasa"; }

filters::FilterOP
LigInterfaceEnergyFilterCreator::create_filter() const { return new LigInterfaceEnergyFilter; }

std::string
LigInterfaceEnergyFilterCreator::keyname() const { return "LigInterfaceEnergy"; }

filters::FilterOP
RepackWithoutLigandFilterCreator::create_filter() const { return new RepackWithoutLigandFilter; }

std::string
RepackWithoutLigandFilterCreator::keyname() const { return "RepackWithoutLigand"; }


} // enzdes
} // protocols
