// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/enzdes/EnzdesMovers.hh
/// @brief a collection of movers that are used at different stages in enzyme design
/// @author Sinisa Bjelic sinibjelic@gmail.com, Florian Richter, floric@u.washington.edu

#include <protocols/enzdes/EnzdesMovers.hh>
#include <protocols/enzdes/EnzdesMoversCreator.hh>

// AUTO-REMOVED #include <protocols/enzdes/EnzdesFixBBProtocol.hh>
#include <protocols/enzdes/EnzdesBaseProtocol.hh>
#include <protocols/enzdes/EnzdesCacheableObserver.hh>
#include <protocols/toolbox/match_enzdes_util/EnzdesCstCache.hh>
#include <protocols/toolbox/match_enzdes_util/EnzConstraintIO.hh>
#include <protocols/enzdes/EnzdesTaskOperations.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/toolbox/pose_manipulation.hh>
#include <protocols/ligand_docking/ligand_functions.hh>
#include <protocols/moves/DataMap.hh>
#include <protocols/moves/Mover.hh>
#include <utility/Tag/Tag.hh>

#include <core/chemical/ResidueTypeSet.fwd.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>
// AUTO-REMOVED #include <core/chemical/util.hh>
// AUTO-REMOVED #include <core/kinematics/MoveMap.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Constraints.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/MultiConstraint.hh>
#include <core/scoring/constraints/MultiConstraint.fwd.hh>
#include <core/scoring/constraints/Func.fwd.hh>
#include <core/scoring/rms_util.tmpl.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
// AUTO-REMOVED #include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/Tracer.hh>
#include <core/id/AtomID.hh>

#include <utility/string_util.hh>
// AUTO-REMOVED #include <utility/io/izstream.hh>

// option key includes
#include <core/options/keys/enzdes.OptionKeys.gen.hh>

namespace protocols {
namespace enzdes {

static core::util::Tracer mv_tr("protocols.enzdes.PredesignPerturbMover");
//PredesignPerturbMoverCreator

moves::MoverOP
PredesignPerturbMoverCreator::create_mover() const
{
	return new PredesignPerturbMover;
}

std::string
PredesignPerturbMoverCreator::keyname() const
{
	return PredesignPerturbMoverCreator::mover_name();
}

std::string
PredesignPerturbMoverCreator::mover_name()
{
	return "PredesignPerturbMover";
}

//-------------PredesignPerturbMover-----------------//

PredesignPerturbMover::PredesignPerturbMover():
	protocols::moves::RigidBodyPerturbMover()
{
	trans_magnitude(core::options::option[core::options::OptionKeys::enzdes::trans_magnitude]);
	rot_magnitude(core::options::option[core::options::OptionKeys::enzdes::rot_magnitude]);
	dock_trials_ = core::options::option[core::options::OptionKeys::enzdes::dock_trials];
}

PredesignPerturbMover::~PredesignPerturbMover(){}

void
PredesignPerturbMover::set_docking_pose(
	core::pose::Pose &pose,
	core::pack::task::PackerTaskCOP task )
{
	toolbox::match_enzdes_util::EnzdesCstCacheOP cst_cache( get_enzdes_observer( pose )->cst_cache() );

	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if( task -> design_residue(i) && !cst_cache -> contains_position(i) )
			positions_to_replace_.push_back(i);
	}

	protocols::toolbox::pose_manipulation::construct_poly_ala_pose(
		pose, positions_to_replace_, true, true, true );
}

void
PredesignPerturbMover::reinstate_pose(
	core::pose::Pose &pose,
	core::pose::Pose const &old_Pose )
{
	core::Size ires;
  for(core::Size i=1, i_end=positions_to_replace_.size(); i<=i_end; ++i) {
		ires=positions_to_replace_[i];
 		pose.replace_residue( ires, old_Pose.residue(ires), true);
	}
}

void
PredesignPerturbMover::add_constrained_lig_atoms_from_multiconstraint(
  core::scoring::constraints::MultiConstraintCOP real_multi_constraint )
{
  core::scoring::constraints::ConstraintCOPs multi_constraint_members;
  multi_constraint_members=real_multi_constraint->member_constraints();

	for (core::scoring::constraints::ConstraintCOPs::const_iterator MC_it=multi_constraint_members.begin(); MC_it!=multi_constraint_members.end(); MC_it++) {

		if ( ((*MC_it)->type()) == "AtomPair") {
			( ((*MC_it)->atom(1)).rsd() == ligand_seqpos_) ? add_constrained_lig_atom( ((*MC_it)->atom(1)).atomno() ) : add_constrained_lig_atom( ((*MC_it)->atom(2)).atomno() ) ;
     // mv_tr.Info << "adding AtomPair" << std::endl;
     // mv_tr.Info << ((*MC_it)->atom(1)).atomno() << std::endl;
		} else if ( ((*MC_it)->type()) == "MultiConstraint") {
			add_constrained_lig_atoms_from_multiconstraint( (dynamic_cast <core::scoring::constraints::MultiConstraint const * > ((*MC_it)())) );
    } else if ( ((*MC_it)->type()) == "AmbiguousConstraint") {
      add_constrained_lig_atoms_from_multiconstraint( (dynamic_cast <core::scoring::constraints::MultiConstraint const * > ((*MC_it)())) );
		}
	}
}

void
PredesignPerturbMover::add_constrained_lig_atom(
	core::Size atom_no )
{
  for(utility::vector1< core::Size >::const_iterator it=constrained_lig_atoms_.begin(); it !=constrained_lig_atoms_.end(); it++){
    if ( (*it) == atom_no)
			return;
	}
	constrained_lig_atoms_.push_back(atom_no);
  mv_tr.Info << "Constrained ligand atom: " << std::endl;
  mv_tr.Info << atom_no << std::endl;
}

void
PredesignPerturbMover::find_constraints_to_ligand(
	core::pose::Pose const & pose )
{
  using namespace core::scoring::constraints;

  //Get a set of constraints for a ligand.
  //It is a vector of ResidueConstraints.
  ConstraintSetCOP constraint_set;
  constraint_set=pose.constraint_set();
  ConstraintSet::ResiduePairConstraintsIterator rpc_start=constraint_set->residue_pair_constraints_begin(ligand_seqpos_);
  ConstraintSet::ResiduePairConstraintsIterator rpc_end=constraint_set->residue_pair_constraints_end(ligand_seqpos_);

  // Each residue constraint is a map container
  // of Size and ConstraintsOP, defining a residue number
 	//and constraints object respectively.
  for (; rpc_start != rpc_end; ++rpc_start ) {
    //Constraints object is a pointer vector of constraint objects.
    // Each contraint object contains type of constraints.
    ConstraintsOP constraints;
		constraints=rpc_start->second;
		Constraints::const_iterator it_begin=constraints->begin();
  	Constraints::const_iterator it_end=constraints->end();
    for (; it_begin != it_end; ++it_begin ) {
		   mv_tr.Info <<(*it_begin)->type() << std::endl;
			  if ( ((*it_begin)->type()) == "MultiConstraint") {
  //        MultiConstraintCOP real_multi_constraint = dynamic_cast <MultiConstraint const * > ((*it_begin)());
  //        add_constrained_lig_atoms_from_multiconstraint(real_multi_constraint);
          add_constrained_lig_atoms_from_multiconstraint( dynamic_cast <MultiConstraint const * > ((*it_begin)()) );
			  } else if ( ((*it_begin)->type()) == "AmbiguousConstraint") {
          add_constrained_lig_atoms_from_multiconstraint( dynamic_cast <MultiConstraint const * > ((*it_begin)()) );
				} else if ( ((*it_begin)->type()) == "AtomPair") {
        // if ligand equals residue of atom1 then assign atom1
        // otherwise asign the other atom
      ( ((*it_begin)->atom(1)).rsd() == ligand_seqpos_ ) ? add_constrained_lig_atom( ((*it_begin)->atom(1)).atomno() ) : add_constrained_lig_atom( ((*it_begin)->atom(2)).atomno() ) ;
			  }
 		}
	}
}

core::Vector
PredesignPerturbMover::find_geometric_center_for_constrained_lig_atoms(
	core::pose::Pose const &pose)
{
  // geometric_center=1000000; //for testing
	core::Vector geometric_center;
  geometric_center = 0.0;
  for( utility::vector1< core::Size >::const_iterator it = constrained_lig_atoms_.begin();
				it != constrained_lig_atoms_.end(); ++it ){
		geometric_center+=pose.residue(ligand_seqpos_).xyz(*it);
	}

  //check if there are any atoms in constraind_lig_atoms_
  if( constrained_lig_atoms_.size() != 0 ) {
	  geometric_center /= constrained_lig_atoms_.size();
	} else {
  	std::cerr << "There are no atom pair constraints. Predocking is done relative to a constrained ligand atom!" << std::endl;
	}
  return geometric_center;
}

void
PredesignPerturbMover::apply(
	core::pose::Pose & pose)
{
  //make a poly ala of the designable
 	protocols::enzdes::EnzdesBaseProtocolOP enzprot = new protocols::enzdes::EnzdesBaseProtocol();
  core::pose::Pose org_Pose(pose);
  core::pack::task::PackerTaskCOP task
		= enzprot -> create_enzdes_pack_task( pose, true );
  set_docking_pose( pose, task );

  protocols::moves::MonteCarloOP MCpredock = new protocols::moves::MonteCarlo(
		pose,
		*(core::scoring::ScoreFunctionFactory::create_score_function( "enzdes_polyA_min" ) ),
		2.0 /* temperature, from RosettaLigand paper */);
  MCpredock->reset( pose );
  MCpredock->reset_counters();

  find_constraints_to_ligand(pose);

  //itereate through all constraints in the pose and check for the constrained atoms
	//ligand is always connected through the last jump
	mv_tr.Info << "starting predocking ... " << std::endl;
  for( core::Size i=1; i <= dock_trials_; ++i ){
    rot_center_= find_geometric_center_for_constrained_lig_atoms(pose);

    core::kinematics::Jump flexible_jump = pose.jump( pose.num_jump() );
    core::kinematics::Stub downstream_stub = pose.conformation().downstream_jump_stub( pose.num_jump() );
    flexible_jump.set_rb_center( dir_, downstream_stub, rot_center_ );

    flexible_jump.gaussian_move( dir_, trans_mag_, rot_mag_ );
    pose.set_jump( pose.num_jump(), flexible_jump );
    MCpredock->boltzmann(pose);

  }
  MCpredock->show_counters();
  MCpredock->recover_low( pose );
  mv_tr.Info << "... done predocking" << std::endl;
  //put back the old pose
  reinstate_pose( pose, org_Pose );
}


void
PredesignPerturbMover::parse_my_tag(
	utility::Tag::TagPtr const tag,
	protocols::moves::DataMap & ,
	protocols::filters::Filters_map const & ,
	protocols::moves::Movers_map const & ,
	core::pose::Pose const & pose)
{
	trans_magnitude( tag -> getOption< core::Real >( "trans_magnitude", 0.1 ) );
	rot_magnitude( tag -> getOption< core::Real >( "rot_magnitude", 2.0 ) );
	dock_trials_ = tag -> getOption< core::Size >( "dock_trials", 100 );
	ligand_seqpos_ = (core::Size) pose.fold_tree().downstream_jump_residue( pose.num_jump() );
}

protocols::moves::MoverOP
PredesignPerturbMover::clone() const
{
	return protocols::moves::MoverOP( new PredesignPerturbMover( *this ) );
}

protocols::moves::MoverOP
PredesignPerturbMover::fresh_instance() const
{
	return protocols::moves::MoverOP( new PredesignPerturbMover );
}

std::string
PredesignPerturbMover::get_name() const
{
	return "PredesignPerturbMover";
}

//-------RepackLigandSiteWithoutLigandMover----------//

RepackLigandSiteWithoutLigandMover::RepackLigandSiteWithoutLigandMover()
	: sfxn_(NULL), lig_seqpos_(0), enzcst_io_(NULL), calculate_silent_Es_(false)
{
	silent_Es_.clear();
}

RepackLigandSiteWithoutLigandMover::RepackLigandSiteWithoutLigandMover(
	core::scoring::ScoreFunctionCOP sfxn,
	bool calculate_silent_Es
	) : sfxn_(sfxn), lig_seqpos_(0), enzcst_io_(NULL), calculate_silent_Es_(calculate_silent_Es)
{
	silent_Es_.clear();
}

RepackLigandSiteWithoutLigandMover::~RepackLigandSiteWithoutLigandMover(){}

void
RepackLigandSiteWithoutLigandMover::set_sfxn(
	core::scoring::ScoreFunctionCOP sfxn )
{
	sfxn_ = sfxn;
}

void
RepackLigandSiteWithoutLigandMover::set_cstio(
	toolbox::match_enzdes_util::EnzConstraintIOCOP enzcst_io )
{
	enzcst_io_ = enzcst_io;
}

void
RepackLigandSiteWithoutLigandMover::set_calculate_silent_Es(
	bool calculate )
{
	calculate_silent_Es_ = calculate;
}

void
RepackLigandSiteWithoutLigandMover::apply(
	core::pose::Pose & pose )
{
	runtime_assert( sfxn_ );
	//tmp hack
	//the constraints can be a headache in this situation, so well completely take them out for now
	// the problem is that some constrained interactions can be covalent, and the EnzConstraintIO
	// object at the moment can't separately take out the constraints and the covalent connections
	core::scoring::ScoreFunctionOP tmpsfxn = sfxn_->clone();
	tmpsfxn->set_weight(core::scoring::coordinate_constraint, 0.0 );
	tmpsfxn->set_weight(core::scoring::atom_pair_constraint, 0.0 );
	tmpsfxn->set_weight(core::scoring::angle_constraint, 0.0 );
	tmpsfxn->set_weight(core::scoring::dihedral_constraint, 0.0 );
	sfxn_ = tmpsfxn;
	//tmp hack over

	if( lig_seqpos_ == 0 ){
		utility::vector1< core::Size > all_ligands( ligand_docking::get_ligand_seqpos( pose ) );
		if( all_ligands.size() != 1 ) utility_exit_with_message( "Pose has more or less than one ligand. This mover atm can only hadndle poses with one ligand.");
		lig_seqpos_ = all_ligands[1];
	}
	core::pose::PoseOP startpose;
	utility::vector1< core::Size > special_res;
	if( calculate_silent_Es_ ) startpose = new core::pose::Pose( pose );

	//1. if there are constraints between protein and ligand, we should take them out.
	if( enzcst_io_ ){
		if( calculate_silent_Es_ ) special_res = enzcst_io_->ordered_constrained_positions( pose );
		enzcst_io_->remove_constraints_from_pose( pose, true /* keep covalent*/, false /*fail on missing*/ );
	}
	core::Real start_score( (*sfxn_)( pose ) );
	if( enzcst_io_ ){
		enzcst_io_->remove_constraints_from_pose( pose, false /* keep covalent*/, false /*fail on missing*/ );
	}

	//2. construct the proper task
	DetectProteinLigandInterfaceOP detect_enzdes_interface = new DetectProteinLigandInterface();
  detect_enzdes_interface->set_design(false);
	core::pack::task::TaskFactory taskfactory;
  taskfactory.push_back( new core::pack::task::operation::InitializeFromCommandline() );
  taskfactory.push_back( detect_enzdes_interface);
	ptask_ = taskfactory.create_task_and_apply_taskoperations( pose );

	//3. shoot the ligand into space
	separate_protein_and_ligand( pose );
	(*sfxn_)( pose );
	//pose.dump_pdb( "rlswlm_after_rigid.pdb");

	//4. repack
	protocols::moves::PackRotamersMoverOP packer = new protocols::moves::PackRotamersMover(sfxn_, ptask_);
	packer->apply( pose );
	//pose.dump_pdb( "rlswlm_after_repack.pdb");

	//5. if requested, do more shit
	if( calculate_silent_Es_ ){
		//5a. Ediff
		core::Real end_score( (*sfxn_)( pose ) );
		//std::cerr << "start score=" << start_score <<", int score=" << int_score << ", end_score=" << end_score << std::endl;
		silent_Es_.push_back( core::io::silent::SilentEnergy("nlr_dE", end_score - start_score, 1, 12 ) );

		//5b. rmsd of repackable region
		ObjexxFCL::FArray1D_bool pack_region( ptask_->total_residue(), false );
		for( core::Size i = 1; i <= ptask_->total_residue(); ++i ){
			if( ptask_->residue_task( i ).being_packed() && pose.residue( i ).is_protein() ) pack_region( i ) = true;
		}
		core::Real pack_region_rmsd( core::scoring::rmsd_no_super_subset( *startpose, pose, pack_region, core::scoring::is_protein_sidechain_heavyatom ) );
		silent_Es_.push_back( core::io::silent::SilentEnergy("nlr_totrms", pack_region_rmsd, 1, 12 ) );

		//5c. rmsds of any eventual special residues
		for( core::Size i =1; i <= special_res.size(); ++i){
			if( pose.residue_type( special_res[i] ).is_ligand() ) continue;
			ObjexxFCL::FArray1D_bool pack_region( ptask_->total_residue(), false );
			pack_region( special_res[i] ) = true;
			core::Real spec_res_rmsd( core::scoring::rmsd_no_super_subset( *startpose, pose, pack_region, core::scoring::is_protein_sidechain_heavyatom ) );
			std::string title( "nlr_SR"+utility::to_string( i )+"_rms");
			silent_Es_.push_back( core::io::silent::SilentEnergy(title, spec_res_rmsd, 1, 14 ) );
		}
	}
	//6. finally let's remove the ligand from the pose for completeness
	if( enzcst_io_ ){
		if( enzcst_io_->contains_position( pose, lig_seqpos_ ) ){
			enzcst_io_->remove_position_from_template_res( pose, lig_seqpos_ );
			//for now we'll wipe out the cst cache
			get_enzdes_observer( pose )->set_cst_cache( NULL );
		}
	}
	pose.conformation().delete_residue_slow( lig_seqpos_ );
	(*sfxn_)( pose ); //make sure energies are up to date
	//pose.dump_pdb( "rlswlm_after_repackdel.pdb");
}

std::string
RepackLigandSiteWithoutLigandMover::get_name() const {
	return "RepackLigandSiteWithoutLigandMover";
}


void
RepackLigandSiteWithoutLigandMover::separate_protein_and_ligand(
	core::pose::Pose & pose ) const
{
	protocols::moves::RigidBodyTransMover trans_mover( pose, pose.fold_tree().get_jump_that_builds_residue( lig_seqpos_ ) );
	trans_mover.step_size( 666 );
	trans_mover.apply( pose );
}

core::pack::task::PackerTaskCOP
RepackLigandSiteWithoutLigandMover::get_ptask() const {
	return ptask_;
}

} //enzdes
} //protocols

