#include <protocols/fast_sc_mover/SimpleInteractionGraph.hh>
#include <protocols/fast_sc_mover/SimpleInteractionGraph.fwd.hh>
#include <core/graph/Graph.hh>
#include <core/graph/PointGraph.hh>
#include <core/graph/find_neighbors.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionInfo.hh>


#include <core/conformation/Residue.hh>
#include <core/pose/Pose.hh>

#include <core/scoring/TenANeighborGraph.hh>
#include <core/scoring/Energies.hh>

#include <core/util/Tracer.hh>
#include <core/util/prof.hh>


namespace protocols {
namespace fast_sc_mover {

using namespace core;
using namespace core::scoring;

static util::Tracer TR("protocols.fast_sc_mover.SimpleInteractionGraph");

SimpleNode::SimpleNode( Graph* owner, core::Size node_id ):
  Node( owner, node_id  ),
  moved_(false),
  resnum_( node_id ),
  current_one_body_energy_(0),
  alternate_one_body_energy_(0)
{
  initialize();
}

SimpleNode::~SimpleNode() {}

void
SimpleNode::initialize(){
  /**
  SimpleInteractionGraphCOP ig( dynamic_cast< SimpleInteractionGraph const * >(Node::get_owner()));
  runtime_assert( ig );
  set_current( ig->pose().residue( resnum_ ) );
  **/ //better to set up in graph initialization??
}

core::Real
SimpleNode::one_body_energy(){
  if( moved_ ){
    return alternate_one_body_energy_;
  }
  else{
    return current_one_body_energy_;
  }
}

core::Real
SimpleNode::proposed_one_body_energy(){
  return alternate_one_body_energy_;
}

core::Real
SimpleNode::current_one_body_energy(){
  return current_one_body_energy_;
}

void
SimpleNode::commit_change(){
  current_residue_ = alternate_residue_;
  moved_ = false;
  current_one_body_energy_ = alternate_one_body_energy_;
  for( EdgeListIterator edge_itr = this->edge_list_begin();
       edge_itr != this->edge_list_end();
       ++edge_itr){
    //update energies
    SimpleEdge* current_edge(dynamic_cast< SimpleEdge *>(*edge_itr));
    current_edge->commit_change();
  }
}

void
SimpleNode::reject_change(){
  moved_ = false;
}

bool
SimpleNode::moved(){
    return moved_;
}

void
SimpleNode::set_current( core::conformation::ResidueOP res){
	//TR.Debug << "setting res " << res->seqpos() << " to current: " << std::endl;
	current_residue_ = res;
	moved_ = false;
	//current_residue_->update_actcoord();
	update_current_one_body_energy();
	//also update edge energies
	for( EdgeListIterator edge_itr = this->edge_list_begin();
		  edge_itr != this->edge_list_end();
		  ++edge_itr){
		//update energies
		SimpleEdge* current_edge(dynamic_cast< SimpleEdge *>(*edge_itr));
		current_edge->update_current_energy();
	}
}


void
SimpleNode::set_alternate( core::conformation::ResidueOP res){
	//TR.Debug << "setting res " << res->seqpos() << " to alternate: " << std::endl;
        if( alternate_residue_ != 0 ) {
	  TR.Debug << "setting res " << alternate_residue_->type().name() << " to new-res: " << res->type().name() << std::endl;
        }
	alternate_residue_ = res;
	moved_ = true;
	//alternate_residue_->update_actcoord();
	update_alternate_one_body_energy();
	//also update edge energies
	for( EdgeListIterator edge_itr = this->edge_list_begin();
		  edge_itr != this->edge_list_end();
		  ++edge_itr){
		SimpleEdge* current_edge(dynamic_cast< SimpleEdge *>(*edge_itr));
		current_edge->update_proposed_energy();
	}
}

core::conformation::ResidueOP
SimpleNode::get_current(){
  return current_residue_;
}

core::conformation::ResidueOP
SimpleNode::get_alternate(){
  return alternate_residue_;
}

void
SimpleNode::update_current_one_body_energy(){
	  SimpleInteractionGraphCOP ig( dynamic_cast< SimpleInteractionGraph const * >(Node::get_owner()));
	  runtime_assert( ig );
	  EnergyMap emap;
	  ig->scorefunction().eval_ci_1b( *current_residue_, ig->pose(), emap );
	  ig->scorefunction().eval_cd_1b( *current_residue_, ig->pose(), emap );
	  current_one_body_energy_ = ig->scorefunction().weights().dot(emap);
}

void
SimpleNode::update_alternate_one_body_energy(){
	SimpleInteractionGraphCOP ig( static_cast< SimpleInteractionGraph const * >(Node::get_owner()));
	assert( dynamic_cast< SimpleInteractionGraph const * >(Node::get_owner()) );
	EnergyMap emap;
	ig->scorefunction().eval_ci_1b( *alternate_residue_, ig->pose(), emap );
	ig->scorefunction().eval_cd_1b( *alternate_residue_, ig->pose(), emap );
	alternate_one_body_energy_ =
	  ig->scorefunction().weights().dot( emap );
}



SimpleEdge::SimpleEdge( Graph* owner ):
  Edge( owner, 0, 0 ),
  current_energy_(0),
  proposed_energy_(0)
{
}


SimpleEdge::SimpleEdge( Graph* owner, core::Size res1, core::Size res2 ):
  Edge( owner, res1, res2 ),
  current_energy_(0),
  proposed_energy_(0)
{
  // compute_energy();
}

SimpleEdge::~SimpleEdge() {}

core::Real
SimpleEdge::get_current_energy(){
  return current_energy_;
}

core::Real
SimpleEdge::get_proposed_energy(){
  return proposed_energy_;
}

void
SimpleEdge::update_current_energy(){
  compute_energy( true, true );
}

void
SimpleEdge::update_proposed_energy(){
  SimpleNode * node1( dynamic_cast< SimpleNode * > (this->get_node( (platform::Size) 0 )));
  SimpleNode * node2( dynamic_cast< SimpleNode * > (this->get_node( (platform::Size) 1 )));
  bool use_current_node1 = true;
  bool use_current_node2 = true;
  if( node1->moved() ){
    use_current_node1 = false;
  }
  if( node2->moved() ){
    use_current_node2 = false;
  }
  compute_energy( use_current_node1, use_current_node2 );
}

void
SimpleEdge::commit_change(){
  current_energy_ = proposed_energy_;
}

void
SimpleEdge::compute_energy( bool use_current_node1, bool use_current_node2 ){
	//  TR.Debug << "num nodes " << (this->get_owner())->num_nodes() << std::endl;
	SimpleNode * node1( dynamic_cast< SimpleNode * > (this->get_node( (platform::Size) 0 )));
	SimpleNode * node2( dynamic_cast< SimpleNode * > (this->get_node( (platform::Size) 1 )));
	assert( node1 && node2 );
	core::conformation::ResidueOP res1;
	core::conformation::ResidueOP res2;
	//std::string node1_used = "";
	//std::string node2_used = "";

	if( !use_current_node1 ){
		res1 = node1->get_alternate();
		//node1_used= "alternate";
	}else{
		res1 = node1->get_current();
		//node1_used = "current";
	}

	if( !use_current_node2 ){
		//node2_used = "alternate";
		res2 = node2->get_alternate();
	}else{
		//node2_used = "current";
		res2 = node2->get_current();
	}

	//res1->update_actcoord();
	//res2->update_actcoord();

	// TR.Debug << res1->seqpos() << " using " << node1_used << " " << res2->seqpos() << " " << node2_used << std::endl;

	SimpleInteractionGraphCOP ig( static_cast< SimpleInteractionGraph const * >(this->get_owner()));
	assert( dynamic_cast< SimpleInteractionGraph const * >(this->get_owner()) );
	core::pose::Pose const & pose = ig->pose();
	EnergyMap twobody_emap; // APL Note: class TwoBodyEnergyMap has been deprecated / removed

	ig->scorefunction().eval_ci_2b_sc_sc( *res1, *res2, pose, twobody_emap );
	ig->scorefunction().eval_cd_2b_sc_sc( *res1, *res2, pose, twobody_emap );
	ig->scorefunction().eval_ci_2b_bb_sc( *res1, *res2, pose, twobody_emap );
	ig->scorefunction().eval_cd_2b_bb_sc( *res1, *res2, pose, twobody_emap );
	ig->scorefunction().eval_ci_2b_bb_sc( *res2, *res1, pose, twobody_emap );
	ig->scorefunction().eval_cd_2b_bb_sc( *res2, *res1, pose, twobody_emap );

	core::Real energy =
		ig->scorefunction().weights().dot( twobody_emap, ig->scorefunction().ci_2b_types() ) +
		ig->scorefunction().weights().dot( twobody_emap, ig->scorefunction().cd_2b_types() );
	if( use_current_node1 && use_current_node2 ){
		current_energy_ = energy;
	}else{
		proposed_energy_ = energy;
	}
}

SimpleInteractionGraph::SimpleInteractionGraph() :
	Graph(),
	sfxn_(),
	pose_(),
	accumulated_ediff_(0)
{}

SimpleInteractionGraph::~SimpleInteractionGraph() {}

void SimpleInteractionGraph::set_scorefunction( ScoreFunction const & sfxn )
{
	sfxn_ = new core::scoring::ScoreFunction( sfxn );
}

//set up all nodes of graph
void
SimpleInteractionGraph::initialize( core::pose::Pose const & pose){
	TR.Debug << "calling initialize on pose " << std::endl;
	pose_ = new core::pose::Pose(pose);
	//core::Real current_energy = (*sfxn_)(*pose_);	//DEBUG
	runtime_assert( pose_ );
	//Graph::delete_everything();//DEBUG
	if( Graph::num_nodes() != pose.total_residue() ){
	  Graph::set_num_nodes( pose.total_residue() );
	}
	for( core::Size res_i = 1; res_i <= pose_->total_residue(); res_i++ ){
		SimpleNode * newnode = dynamic_cast< SimpleNode * >(get_node( res_i ));
		runtime_assert( newnode );
		newnode->set_current( new core::conformation::Residue( pose_->residue( res_i )) );
	}


   //NEIGHBORS DETERMINED THROUGH ARE_THEY_NEIGHBORS function
	for(core::Size ii = 1; ii <= pose_->total_residue(); ii++){
		for(core::Size jj = 1; jj < ii; jj++){
			//TR.Debug << "examining nodes " << jj << " " << ii << std::endl;
		  if( sfxn_->are_they_neighbors( *pose_, jj, ii ) ) {
		    if( Graph::get_edge_exists( jj, ii ) ) {
		      SimpleEdge * existing_edge = dynamic_cast< SimpleEdge * >( Graph::find_edge( jj, ii ) );
		      if( existing_edge ){
			existing_edge->compute_energy( true, true );
		      }
		    } else {
		      SimpleEdge* newedge( new SimpleEdge( this, jj, ii ) ); //automatically adds edge upon creation
		      newedge->compute_energy( true, true );
		      //TR.Debug << "DEBUG2 these two residues are neighbors " << ii << " " << jj << std::endl;
		    }
		  }
		}
	}
}


void
SimpleInteractionGraph::commit_change( core::Size node_id ){
  dynamic_cast< SimpleNode * >(get_node( node_id ))->commit_change();
  //SimpleNode * node = dynamic_cast< SimpleNode * >(get_node( node_id ));
  //node->set_current( node->get_alternate() );
}

void
SimpleInteractionGraph::reject_change( core::Size node_id ){
  dynamic_cast< SimpleNode * >( get_node( node_id ) )->reject_change();
}

core::Real
SimpleInteractionGraph::consider_substitution( core::Size node_id, core::conformation::ResidueOP new_state ){
  core::Real previous_energy = 0.0;

  SimpleNode* thisnode = dynamic_cast< SimpleNode *>(get_node( node_id ));

  for( EdgeListIterator edge_itr = thisnode->edge_list_begin();
       edge_itr != thisnode->edge_list_end();
       ++edge_itr){
    //update energies
    SimpleEdge* current_edge(dynamic_cast< SimpleEdge *>(*edge_itr));
    previous_energy += current_edge->get_current_energy();
  }
  previous_energy += thisnode->current_one_body_energy();

  thisnode->set_alternate( new_state );
  //iterate over all edges
  core::Real alternate_energy = 0.0;
  for( EdgeListIterator edge_itr = thisnode->edge_list_begin();
       edge_itr != thisnode->edge_list_end();
       ++edge_itr){
    //update energies
    SimpleEdge* current_edge(dynamic_cast< SimpleEdge *> (*edge_itr));
    alternate_energy += current_edge->get_proposed_energy();
  }
  alternate_energy += thisnode->proposed_one_body_energy();
  return (previous_energy - alternate_energy);
}


core::Real
SimpleInteractionGraph::total_energy() {
	//iterate over node ids
	core::Real total_energy = 0.0;
	for(core::Size node_itr = 1; node_itr <= pose_->total_residue(); node_itr++){
		SimpleNode* thisnode = dynamic_cast< SimpleNode *> (get_node( node_itr ));
		core::Real residue_energy = thisnode->one_body_energy();
		for( EdgeListIter edge_iter = thisnode->edge_list_begin(); edge_iter != thisnode->edge_list_end(); ++edge_iter){
			residue_energy += (dynamic_cast< SimpleEdge *>( *edge_iter ))->get_current_energy();
		}
		//TR.Debug << node_itr << ": total residue energy is " << residue_energy << std::endl;
		total_energy += residue_energy;
	}
	return total_energy;

}

Node* SimpleInteractionGraph::create_new_node( platform::Size node_index ){
  return new SimpleNode( this, node_index );
}

Edge* SimpleInteractionGraph::create_new_edge( platform::Size index1, platform::Size index2 ){
  return new SimpleEdge( this, index1, index2 );
}

  /**
Edge* SimpleInteractionGraph::create_new_edge( Edge* example_edge ){
  dynamic_cast < SimpleEdge * >
  return new SimpleEdge( example_edge );
}
  **/

  /**
core::Real
SimpleInteractionGraph::total_alternate_energy() {
  if( moved_ ){
    //compute energies
    moved_ = false;
  }
  return alternate_one_body_energy_ + alt_scsc_E + alt_bbsc_E + alt_scbb_E;
}
  **/

/**
bool
SimpleInteractionGraph::node_altered( core::Real const node_id ){
  return node(node_id).is_altered();
}
**/ //needed?


} //scoring
} //core
