// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// Copyright in the Rosetta software belongs to the developers and their institutions.
// For more information, see www.rosettacommons.org.

/// @file ./src/protocols/fldsgn/topology/Sheet.cc
/// @brief
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

// unit headers
#include <protocols/fldsgn/topology/Sheet.hh>

// Project headers
#include <core/util/Tracer.hh>
#include <core/graph/DisjointSets.hh>
#include <protocols/fldsgn/topology/StrandPairing.hh>
#include <protocols/fldsgn/topology/SS_Info2.hh>

// utility headers
#include <utility/exit.hh>

// C++ headers
#include <iostream>
#include <map>

static core::util::Tracer TR( "protocols.topology.Sheet" );

using namespace core;

namespace protocols {
namespace fldsgn {
namespace topology {


/// @brief default constructor
Sheet::Sheet() :
	num_strands_( 0 ),
	is_barrel_( false )
{
	initialize();
}

/// @brief value constructor
Sheet::Sheet( VecSize order_strands,	VecReal orient_strands, bool is_barrel ):
	num_strands_( order_strands.size() ),
	is_barrel_( is_barrel ),
	order_strands_( order_strands ),
	orient_strands_( orient_strands )
{
	runtime_assert( order_strands.size() == orient_strands.size() );
	initialize();
}

/// @brief copy constructor
Sheet::Sheet( Sheet const & s ):
	ReferenceCount(),
	num_strands_( s.num_strands_ ),
	is_barrel_( s.is_barrel_ ),
	order_strands_( s.order_strands_ ),
	orient_strands_( s.orient_strands_ ),
	strand_order_( s.strand_order_ )
{}

/// @brief initialize this class
void
Sheet::initialize()
{
	num_strands_ = order_strands_.size();
	for( Size i=1; i<=order_strands_.size(); i++ ){
		strand_order_.insert( std::map< Size, Size >::value_type( order_strands_[i], i ) );
	}
}

/// @brief default destructor
Sheet::~Sheet(){}


/// @brief return strand pairing
std::ostream & operator<<( std::ostream & out, const Sheet &s )
{
	utility::vector1<Size> const order( s.order_strands() );
	utility::vector1<Real> const orient( s.orient_strands() );

	for( Size i=1; i<= order.size(); i++ ){
		out << "#Strand " << i << ": " << order[ i ] << "," << orient[ i ] << std::endl;;
	}
	return out;
}

/// @brief whether the given strand number belongs to this sheet or not
bool
Sheet::is_member( Size const s )
{
	for( utility::vector1< Size >::const_iterator iter = order_strands_.begin(),
				 iter_end = order_strands_.end(); iter != iter_end; ++iter ) {
		if( s == *iter ){
			return true;
		}
	}
	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
/// @brief default constructor
SheetSet::SheetSet()
{}

/// @brief value constructor
SheetSet::SheetSet( Sheets const & sheets ):
	sheets_( sheets )
{}

/// @brief value constructor
SheetSet::SheetSet( SS_Info2_OP const & ssinfo, StrandPairingSetOP const & spairset )
{
	initialize( ssinfo, spairset );
}

/// @brief copy constructor
SheetSet::SheetSet( SheetSet const & s ) :
	ReferenceCount(),
	sheets_( s.sheets_ )
{}

/// @brief return strand pairing
std::ostream & operator<<( std::ostream & out, const SheetSet &s )
{
	out << "#### Sheet Info " << std::endl;
	for( Size i=1; i<=s.num_sheets() ; i++ ){
		SheetOP const & sop ( s.sheet( i ) );
		if( sop->is_barrel() ){
			out << "## Barrel " << i << ":" << std::endl;
		}else{
			out << "## Sheet " << i << ":" << std::endl;
		}
		out << (*sop);
	}
	return out;
}

/// @brief
void
SheetSet::push_back( SheetOP const & sop )
{
	sheets_.push_back( sop );
}

/// @brief
void SheetSet::clear()
{
	sheets_.clear();
}

/// @brief
SheetOP const &
SheetSet::sheet( Size const s ) const
{
	runtime_assert( s <= sheets_.size() );
	return sheets_[ s ];
}

/// @brief
Size
SheetSet::which_sheet( Size const s ) const
{
	return sheet_number_[ s ];
}

/// @brief
void
SheetSet::set_sheet_number() const
{
	for( Size i=1; i<=sheets_.size(); i++ ){
		SheetOP const & sheet ( sheets_[ i ] );
		VecSize const & order ( sheet->order_strands() );
		for( VecSize::const_iterator it=order.begin(), ite=order.end(); it!=ite; ++it ){
			sheet_number_[ *it ] = i;
		}
	}
}

/// @brief
void
SheetSet::initialize( SS_Info2_OP const & ssinfo, StrandPairingSetOP const & spairset_in )
{
	using core::graph::DisjointSets;
	using protocols::fldsgn::topology::StrandPairings;
	using protocols::fldsgn::topology::StrandPairingOP;
	using protocols::fldsgn::topology::StrandPairingSet;

	// initialize sheet_number_
	for( Size i=1; i<=ssinfo->strands().size(); i++ ) {
		sheet_number_.insert( std::map< Size, Size >::value_type( i, 0 ) );
	}

	// do nothing if strand pairs < 2
	if( spairset_in->num_strands() < 2 ) {
		return;
	}

	StrandPairingSet spairset = *( spairset_in->clone() );
	spairset.make_strand_neighbor_two();

	runtime_assert( spairset.finalized() );

	// strand pairings
	StrandPairings spairs = spairset.strand_pairings();

	// number of strands included in spairset
	// Size num_strands = spairset.num_strands();

	// set neighbor strands and sheet_set
	DisjointSets sheet_set( ssinfo->strands().size() );
  for( StrandPairings::const_iterator iter = spairs.begin(); iter != spairs.end(); ++iter ) {
    StrandPairing const & sp( **iter );
		sheet_set.ds_union( sp.s1(), sp.s2() );
  }

	// calc order of strands
	std::map< Size, VecSize > sset =  sheet_set.sets();
  std::map< Size, VecSize >::iterator it = sset.begin();
  while( it != sset.end() ) {

		bool ibarrel (true);
		VecSize order_strands;

		VecSize list_strands = (*it).second;
		if( list_strands.size() <= 1 ) {
			it++;
			continue;
		}

		for( VecSize::const_iterator itt = list_strands.begin(); itt != list_strands.end(); ++itt ) {
			Size s( *itt );
			if( spairset.neighbor_strands( s ).size() == 1 ) {
				order_strands.push_back( s );
				ibarrel = false;
			} else if( spairset.neighbor_strands( s ).size() > 2 ) {
				TR.Error << "Error, num of neighbor strands > 2 " << std::endl;
				runtime_assert( false );
			}
		}

		if( ibarrel ){
			order_strands.push_back( list_strands.front() );
		}else{
			runtime_assert( order_strands.size() == 2 );
			sort ( order_strands.begin(), order_strands.end() );
			order_strands.erase( order_strands.end() - 1 );
		}

		VecSize const & neighbor ( spairset.neighbor_strands( order_strands.front() ) );
		order_strands.insert( order_strands.begin() + 1, neighbor[1] );

		for( Size i=2; i<= list_strands.size()-1; i++ ){
			Size s1 = order_strands[ i ];
			VecSize const & neighbor( spairset.neighbor_strands( s1 ) );

			for( Size j=1; j<=neighbor.size(); j++ ){
				if( neighbor[ j ] != order_strands[ i-1 ] ){
					order_strands.insert( order_strands.begin() + i, neighbor[j] );
				}
			}
		}

		// set orient of strands in sheet
		VecReal orient_strands;
		orient_strands.push_back( 1 );
		for( Size i=1; i<=order_strands.size()-1; i++ ){
			Size s1( order_strands[ i ] );
			Size s2( order_strands[ i+1 ] );
			StrandPairingOP spairop( spairset.strand_pairing( s1, s2 ) );

			if( spairop->orient() == 'P' ){
				orient_strands.push_back( orient_strands[ i ] );
			}else{
				orient_strands.push_back( orient_strands[ i ]*-1 );
			}
		}

		SheetOP sop = new Sheet( order_strands, orient_strands, ibarrel );
		sheets_.push_back( sop );

    ++it;

  } // while( it !=sset.end() )

	set_sheet_number();

} // SheetTopology::calc_sheetset



} // namespace topology
} // namespace fldsgn
} // namespace protocols

