// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// Copyright in the Rosetta software belongs to the developers and their institutions.
// For more information, see www.rosettacommons.org.

/// @file ./src/protocols/fldsgn/topology/Sheet.hh
/// @brief
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

#ifndef INCLUDED_protocols_fldsgn_topology_Sheet_HH
#define INCLUDED_protocols_fldsgn_topology_Sheet_HH

// unit headers
#include <protocols/fldsgn/topology/Sheet.fwd.hh>

// project headers
#include <core/types.hh>
#include <protocols/fldsgn/topology/StrandPairing.fwd.hh>
#include <protocols/fldsgn/topology/SS_Info2.fwd.hh>

// utility headers
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <map>

//Auto Headers
#include <utility/vector1_bool.hh>


namespace protocols {
namespace fldsgn {
namespace topology {

class Sheet : public utility::pointer::ReferenceCount {
public:

  typedef core::Size Size;
	typedef core::Real Real;
	typedef utility::vector1< Size > VecSize;
	typedef utility::vector1< Real > VecReal;


public:// construct/destruct


  /// @brief default constructor
  Sheet();

  /// @brief value constructor
  Sheet( VecSize order_strands,	VecReal orient_strands, bool is_barrel );

	/// @brief copy constructor
  Sheet( Sheet const & s );

  /// @brief default destructor
  ~Sheet();

	/// @brief clone this object
	SheetOP clone() {
		return new Sheet( *this );
	}

  /// @brief return strand pairing
	friend
	std::ostream & operator<<( std::ostream & out, const Sheet &s );


public:


	/// @brief intialize this class
	void initialize();


public: //accessors


	/// @brief the number strands inclued in
	Size num_strands() const {	return num_strands_;	}

	bool is_barrel() const { return is_barrel_; }

	VecSize order_strands() const {	return order_strands_; }

	VecReal orient_strands() const { return orient_strands_; }

	Size strand_order( Size const & s ) {	return strand_order_[ s ];	}


public: //

	bool is_member( Size const s );

private:  // data

	/// @brief
	Size num_strands_;

	/// @brief
	bool is_barrel_;

	/// @brief order of strand in sheet -> number of strand
	VecSize order_strands_;

	/// @brief order of strand in sheet -> number of strand
	VecReal orient_strands_;

	/// @brief number of strand -> order of strand in sheet
	std::map< Size, Size > strand_order_;

};


class SheetSet : public utility::pointer::ReferenceCount {
public:


  typedef core::Size Size;
  typedef core::Real Real;
  typedef utility::vector1< Size > VecSize;
	typedef utility::vector1< Real > VecReal;
  typedef protocols::fldsgn::topology::StrandPairingSetOP StrandPairingSetOP;
  typedef protocols::fldsgn::topology::SS_Info2_OP SS_Info2_OP;


public:// construct/destruct


  /// @brief default constructor
  SheetSet();

  /// @brief value constructor
  SheetSet( Sheets const & sheets );


  /// @brief value constructor
  SheetSet( SS_Info2_OP const & ssinfo, StrandPairingSetOP const & spairset );

	/// @brief copy constructor
  SheetSet( SheetSet const & s );

  /// @brief return strand pairing
	friend
	std::ostream & operator<<( std::ostream & out, const SheetSet &s );


public:


	void initialize( SS_Info2_OP const & ssinfo, StrandPairingSetOP const & spairset );


public: // mutators


	/// @brief
	void push_back( SheetOP const & sop );

	/// @brief
	void clear();


public: // accessors


	/// @brief
	SheetOP const & sheet( Size const s ) const;

	/// @brief
	inline Sheets	sheets() const { return sheets_; }

	/// @brief
	inline Size	num_sheets() const { return sheets_.size();	}

	/// @brief
	Size which_sheet( Size const s ) const;


private://


	/// @brief
	void set_sheet_number() const;


private:  // data


	/// @brief sheet number given a strand
	mutable std::map< Size, Size > sheet_number_;

	/// @brief
	Sheets sheets_;


};

} // namespace topology
} // namespace fldsgn
} // namespace protocol

#endif
