// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/HighResDocker.hh>
#include <protocols/ligand_docking/HighResDockerCreator.hh>
#include <protocols/ligand_docking/InterfaceBuilder.hh>
#include <protocols/ligand_docking/ligand_options/Interface.hh>
#include <protocols/ligand_docking/MinimizeLigand.hh>
#include <protocols/ligand_docking/MoveMapBuilder.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>

#include <protocols/ligand_docking/UnconstrainedTorsionsMover.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/RigidBodyMover.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/conformation/Conformation.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/rotamer_set/UnboundRotamersOperation.hh>

#include <core/scoring/ScoreFunction.hh>
// Package Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/constraints/ResidueTypeConstraint.hh>
#include <core/kinematics/MoveMap.hh>

// Project Headers
#include <core/chemical/ResidueType.hh>
#include <core/options/option.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>

// Utility Headers
#include <utility/string_util.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <core/kinematics/Edge.hh>
#include <core/pack/task/PackerTask.hh>

// Scripter Headers
#include <utility/Tag/Tag.hh>
#include <protocols/moves/DataMap.hh>

//STL headers
#include <string>

#include <set>

//Auto Headers
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>
#include <numeric/conversions.hh>

#include <core/options/keys/enzdes.OptionKeys.gen.hh>

namespace protocols {
namespace ligand_docking {

static core::util::Tracer high_res_docker_tracer("protocols.ligand_docking.ligand_options.Protocol", core::util::t_debug);

std::string
HighResDockerCreator::keyname() const
{
	return HighResDockerCreator::mover_name();
}

protocols::moves::MoverOP
HighResDockerCreator::create_mover() const {
	return new HighResDocker;
}

std::string
HighResDockerCreator::mover_name()
{
	return "HighResDocker";
}

///@brief
HighResDocker::HighResDocker():
		Mover("HighResDocker"),
		num_cycles_(0),
		repack_every_Nth_(0),
		score_fxn_(NULL),
		movemap_builder_(NULL)
{
	enum_map_["meiler2006"]=meiler2006;
	enum_map_["abbreviated"]= abbreviated;
	enum_map_["abbrev2"]= abbrev2;
	enum_map_["min_only"]= min_only;
	enum_map_["custom"]= custom;
}

HighResDocker::HighResDocker(HighResDocker const & that):
		protocols::moves::Mover( that ),
		num_cycles_(that.num_cycles_),
		repack_every_Nth_(that.repack_every_Nth_),
		enum_map_(that.enum_map_),
		chains_(that.chains_),
		score_fxn_(that.score_fxn_),
		minimize_ligands_(that.minimize_ligands_),
		movemap_builder_(that.movemap_builder_)
{}

HighResDocker::~HighResDocker() {}

protocols::moves::MoverOP HighResDocker::clone() const {
	return new HighResDocker( *this );
}

protocols::moves::MoverOP HighResDocker::fresh_instance() const {
	return new HighResDocker;
}

std::string HighResDocker::get_name() const{
	return "HighResDocker";
}

///@brief parse XML (specifically in the context of the parser/scripting scheme)
void
HighResDocker::parse_my_tag(
		utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & datamap,
		protocols::filters::Filters_map const & filters,
		protocols::moves::Movers_map const & movers,
		core::pose::Pose const & pose
)
{
	if ( tag->getName() != "HighResDocker" ){
		utility_exit_with_message("This should be impossible");
	}

	/// Ligand Chains to dock...

	if ( ! tag->hasOption("chains") ) utility_exit_with_message("'HighResDocker' requires 'chains' tag (comma separated chains to dock)");

	std::string chains_str = tag->getOption<std::string>("chains");
	chains_= utility::string_split(chains_str, ',');

	/// Protocol Tag ///
	if ( ! tag->hasOption("protocol") ) utility_exit_with_message("'HighResDocker' requires 'protocol' tag");
	std::string protocol_name= tag->getOption<std::string>("protocol");
	std::map< std::string, protocol_enum >::const_iterator const enum_pair=
			enum_map_.find(protocol_name);
	if(enum_pair == enum_map_.end()){
		utility_exit_with_message("In your ligand option file, you specified a protocol that doesn't exist");
	}
	protocol_enum p= enum_pair->second;
	set_cycles(p, tag);

	/// Score Function ///
	if ( ! tag->hasOption("scorefxn") ) utility_exit_with_message("'HighResDocker' requires 'scorefxn' tag");
	std::string scorefxn_name= tag->getOption<std::string>("scorefxn");
	score_fxn_= datamap.get< core::scoring::ScoreFunction * >( "scorefxns", scorefxn_name);

	/// MoveMapBuilder///
	if ( ! tag->hasOption("movemap_builder") ) utility_exit_with_message("'HighResDocker' requires 'movemap_builder' tag");
	std::string movemap_builder_name= tag->getOption<std::string>("movemap_builder");
	movemap_builder_= datamap.get< MoveMapBuilder * >( "movemap_builders", movemap_builder_name);

	/// Child Tags ///

	utility::vector0< utility::Tag::TagPtr >::const_iterator begin=tag->getTags().begin();
	utility::vector0< utility::Tag::TagPtr >::const_iterator end=tag->getTags().end();
	for(; begin != end; ++begin){
		utility::Tag::TagPtr tag= *begin;
		std::string name= tag->getName();

		if( name != "MinimizeLigand")
			utility_exit_with_message("HighResDocker only takes 'MinimizeLigand' children");

		MinimizeLigandOP minimize_ligand = new MinimizeLigand();
		minimize_ligand->parse_my_tag(tag, datamap, filters, movers, pose);
		minimize_ligands_.push_back(minimize_ligand);
	}
}

void HighResDocker::apply(core::pose::Pose & pose) {
	assert(num_cycles_ > 0);

	MinimizeLigandOPs::iterator minimize_ligand= minimize_ligands_.begin();
	for(; minimize_ligand != minimize_ligands_.end(); ++minimize_ligand){
		(*minimize_ligand)->apply(pose);
	}
	assert(movemap_builder_ && score_fxn_ ); // make sure the pointers point
	core::kinematics::MoveMapOP movemap = movemap_builder_->build(pose);

	protocols::moves::MonteCarloOP monteCarlo = new protocols::moves::MonteCarlo(pose, *score_fxn_, 2.0);/* temperature, from RosettaLigand paper */
	score_fxn_->score( pose ); // without this neither of the movers below were working
	// I believe that this may have been related to adding constraints incorrectly at other places in my code.
	// Rigid body exploration
	utility::vector1<protocols::moves::MoverOP> rigid_body_movers= create_rigid_body_movers(pose);

	//bool native_favoring_has_been_setup=false;
	for( core::Size cycle = 1; cycle <= num_cycles_; ++cycle ) {
		core::pack::task::PackerTaskOP packer_task = make_packer_task(pose);// has to be in the loop to be updated after each design cycle (w/resfiles)
//		if (!native_favoring_has_been_setup){
//			setup_native_residue_favoring(pose, packer_task);
//			native_favoring_has_been_setup= true;
//		}

		protocols::moves::MoverOP pack_mover;

		if(cycle % repack_every_Nth_ == 1){
			high_res_docker_tracer.Debug << "making PackRotamersMover" << std::endl;
			pack_mover= (protocols::moves::Mover *) new protocols::moves::PackRotamersMover(score_fxn_, packer_task);
		}
		else{
			high_res_docker_tracer.Debug << "making RotamerTrialsMover" << std::endl;
			pack_mover= (protocols::moves::Mover *) new protocols::moves::RotamerTrialsMover(score_fxn_, *packer_task);
		}

		// Wrap it in something to disable the torsion constraints before packing!
		pack_mover = new protocols::ligand_docking::UnconstrainedTorsionsMover( pack_mover, minimize_ligands_ );

		protocols::moves::MinMoverOP min_mover = new protocols::moves::MinMover( movemap, score_fxn_, "dfpmin_armijo_nonmonotone_atol", 1.0, true /*use_nblist*/ );
		min_mover->min_options()->nblist_auto_update(true); // does this cost us lots of time in practice?

		core::Real const score1 = (*score_fxn_)( pose );
		apply_rigid_body_moves(pose, rigid_body_movers);
		pack_mover->apply(pose);

		core::Real const score2 = (*score_fxn_)( pose );
		if(score2 - score1 < 15.0) {
			min_mover->apply(pose);
		}

		monteCarlo->boltzmann( pose );

		// We always want the option (after the initial unbiased pack)
		// of sticking with our current nicely minimized conformation.
		packer_task->or_include_current(true);
	}

	// keep the best structure we found, not the current one
	monteCarlo->show_scores();
	monteCarlo->recover_low(pose);
}

void HighResDocker::set_cycles(
		protocol_enum const p,
		utility::Tag::TagPtr const tag
){
	switch (p){
		case meiler2006:
			num_cycles_= 50;
			repack_every_Nth_= 8;
			if( tag->hasOption("cycles") || tag->hasOption("repack_every_Nth") ){
				utility_exit_with_message("'cycles' and 'repack_every_Nth' are only used with 'HighResDocker's 'custom' tag");
			}
			break;
		case abbreviated:
			num_cycles_= 5;
			repack_every_Nth_= 4;
			break;
		case abbrev2:
			num_cycles_= 6;
			repack_every_Nth_= 3;
			break;
		case min_only:
			num_cycles_=0;
			repack_every_Nth_=0;
		case custom:
			if ( ! tag->hasOption("cycles") ) utility_exit_with_message("'HighResDocker' mover requires cycles tag");
			if ( ! tag->hasOption("repack_every_Nth") ) utility_exit_with_message("'HighResDocker' mover requires repack_every_Nth tag");
			num_cycles_= tag->getOption<core::Size>("cycles");
			repack_every_Nth_= tag->getOption<core::Size>("repack_every_Nth");
			break;
		default:
			utility_exit_with_message("'HighResDocker' option must provide ONE protocol argument.\
				Choices are rescore, meiler2006, abbreviated, abbrev2, min_only, or 'custom int int'");
	}
}

//const std::set<core::Size> HighResDocker::get_docking_jump_ids() const{
//	return jump_ids_;
//}

core::pack::task::PackerTaskOP
HighResDocker::make_packer_task_from_vector(
		core::pose::Pose const & pose,
		ligand_options::Interface const allow_repack
) const{
	core::pack::task::PackerTaskOP pack_task = core::pack::task::TaskFactory::create_packer_task(pose);
	pack_task->initialize_from_command_line(); // -ex1 -ex2  etc.

	core::pack::rotamer_set::UnboundRotamersOperationOP unboundrot_ = new core::pack::rotamer_set::UnboundRotamersOperation();
	unboundrot_->initialize_from_command_line();
	pack_task->append_rotamerset_operation( unboundrot_ );

	for(core::Size i = 1; i <= pose.total_residue(); ++i) {
		/// If several params files have the same name, allow switching among them
		/// This was previously only enabled with mutate_same_name3.  Now default.
		if( ! pose.residue(i).is_ligand()) continue;
		high_res_docker_tracer.Debug<<  "enabling packing for ligand residue "<< i << std::endl;
		enable_ligand_rotamer_packing(pose, i, pack_task);
	}

	bool const use_resfile= core::options::option[ core::options::OptionKeys::packing::resfile ].user() ;

	if (use_resfile){
		high_res_docker_tracer<< "using resfile"<< std::endl;
		pack_task->read_resfile();
	}
	else{
		high_res_docker_tracer<< "restricting to repack"<< std::endl;
		for(core::Size i = 1; i <= pose.total_residue(); ++i) {
			if( ! pose.residue(i).is_ligand() )
			{
				pack_task->nonconst_residue_task( i ).restrict_to_repacking();
			}
		}
	}

	for(core::Size i = 1; i <= pose.total_residue(); ++i) {
		if ( allow_repack[i].type == ligand_options::InterfaceInfo::non_interface  ){
			pack_task->nonconst_residue_task( i ).prevent_repacking();
		}
	}
	return pack_task;
}

core::pack::task::PackerTaskOP
HighResDocker::make_packer_task(
		core::pose::Pose const & pose,
		bool all_residues
) const{
	if ( all_residues ){
		ligand_options::Interface interface(pose.n_residue(), ligand_options::InterfaceInfo(ligand_options::InterfaceInfo::is_interface)); // 0 is false, #
		return make_packer_task_from_vector(pose, interface);
	}else{ // the packer task interface should match the movemap interface
		InterfaceBuilderOP sc_interface_builder= movemap_builder_->get_sc_interface_builder();
		ligand_options::Interface side_chain_interface= sc_interface_builder->build(pose);
		return make_packer_task_from_vector(pose, side_chain_interface);
	}
}


void
HighResDocker::enable_ligand_rotamer_packing(
		core::pose::Pose const & pose,
		core::Size const ligand_residue_id,
		core::pack::task::PackerTaskOP & pack_task
) const{
	core::conformation::Residue const & this_residue= pose.residue(ligand_residue_id);
	core::chemical::ResidueTypeSet const & rsd_type_set = this_residue.residue_type_set();
	core::chemical::ResidueTypeCAPs allowed_types = rsd_type_set.name3_map( this_residue.name3() ); // a vector1

	assert(allowed_types.size() > 0);
	/// TODO consider removing this so resfiles can specify ligand mutations to allow
	if( allowed_types.size() == 1){
		pack_task->nonconst_residue_task( ligand_residue_id ).restrict_to_repacking();
		return;
	}
	// else
	for( core::Size j = 1; j <= allowed_types.size(); ++j ) {
		if( allowed_types[j]->name() == this_residue.name() ) continue; // already in the task's list
		///TODO figure out why this is nonconst.  Perhaps it could be const
		pack_task->nonconst_residue_task( ligand_residue_id ).allow_noncanonical_aa( allowed_types[j]->name() );
	}
}

utility::vector1<protocols::moves::MoverOP>
HighResDocker::create_rigid_body_movers(core::pose::Pose const & pose) const{
	utility::vector1<protocols::moves::MoverOP> rigid_body_movers;
	std::vector<std::string>::const_iterator chain= chains_.begin();
	for(; chain!= chains_.end(); ++chain){
		core::Size jump_id= ligand_options::get_jump_id_from_chain(*chain, pose);
		protocols::moves::MoverOP rigid_body_mover= new protocols::moves::RigidBodyPerturbMover( jump_id, numeric::conversions::degrees(0.05), 0.1);
		rigid_body_movers.push_back(rigid_body_mover);
	}
	return rigid_body_movers;
}

void HighResDocker::apply_rigid_body_moves(
		core::pose::Pose & pose,
		utility::vector1<protocols::moves::MoverOP> & rigid_body_movers
){
	utility::vector1<protocols::moves::MoverOP>::iterator rigid_body_mover= rigid_body_movers.begin();
	for(; rigid_body_mover != rigid_body_movers.end(); ++rigid_body_mover){
		(*rigid_body_mover)->apply(pose);
	}
}

/// Non-member functions

// Favor Native is part of the APPLY_TO_POSE section

//void setup_native_residue_favoring(core::pose::Pose & pose, core::pack::task::PackerTaskOP task){
//	if( ! core::options::option[core::options::OptionKeys::enzdes::favor_native_res].user())
//		return;
//	core::Real bonus = core::options::option[core::options::OptionKeys::enzdes::favor_native_res].value();
//
//	high_res_docker_tracer.Info << "favor_native_res: adding a bonus of " << bonus << " for native residues to pose." << std::endl;
//
//	utility::vector1< core::scoring::constraints::ConstraintCOP > favor_native_constraints;
//	for( core::Size i = 1; i <= pose.total_residue(); ++i){
//
//		if( task->design_residue(i) ){
//			core::scoring::constraints::ConstraintOP resconstraint =
//					new core::scoring::constraints::ResidueTypeConstraint( pose, i, bonus );
//			favor_native_constraints.push_back( resconstraint );
//		}
//	}
//	pose.add_constraints( favor_native_constraints );
//
//}

} //namespace ligand_docking
} //namespace protocols
