// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/MoveMapBuilder.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>
#include <protocols/ligand_docking/InterfaceBuilder.hh>

//Project Headers
#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/Pose.hh>
#include <core/kinematics/Edge.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>

// Utility Headers
#include <utility/exit.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
#include <utility/string_util.hh>

// Scripter Headers
#include <utility/Tag/Tag.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

//STL headers

namespace protocols {
namespace ligand_docking {

static core::util::Tracer MoveMapBuilder_tracer("protocols.ligand_docking.ligand_options.MoveMapBuilder", core::util::t_debug);

MoveMapBuilder::MoveMapBuilder():
		ReferenceCount(),
		sc_interface_builder_(NULL),
		bb_interface_builder_(NULL),
		minimize_water_(false)
{}

MoveMapBuilder::MoveMapBuilder(MoveMapBuilder const & that):
		ReferenceCount(),
		sc_interface_builder_(that.sc_interface_builder_),
		bb_interface_builder_(that.sc_interface_builder_),
		minimize_water_(that.minimize_water_)
{}

MoveMapBuilder::~MoveMapBuilder() {}

//@brief parse XML (specifically in the context of the parser/scripting scheme)
void
MoveMapBuilder::parse_my_tag(
		utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & datamap
){
	if ( ! tag->hasOption("chains") ) utility_exit_with_message("'MoveMapBuilder' requires 'chains' tag");

	std::string chains_string= tag->getOption<std::string>("chains");
	chains_= utility::string_split(chains_string, ',');

	if ( tag->hasOption("sc_interface") ){
		std::string sc_interface_name= tag->getOption<std::string>("sc_interface");
		sc_interface_builder_= datamap.get< protocols::ligand_docking::InterfaceBuilder * >( "interface_builders", sc_interface_name);
	}
	if ( tag->hasOption("bb_interface") ){
		std::string bb_interface_name= tag->getOption<std::string>("bb_interface");
		bb_interface_builder_= datamap.get< protocols::ligand_docking::InterfaceBuilder * >( "interface_builders", bb_interface_name);
	}

	if ( tag->hasOption("minimize_water") ){
		if(tag->getOption<std::string>("minimize_water") == "true")
			minimize_water_= true;
		else if(tag->getOption<std::string>("minimize_water") != "false")
			utility_exit_with_message("'minimize_water' option is true or false");
	}
}

core::kinematics::MoveMapOP
MoveMapBuilder::build(core::pose::Pose const & pose) const{
	core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();

	std::vector<std::string>::const_iterator chain= chains_.begin();
	for(; chain != chains_.end(); ++chain){
		core::Size jump_id= ligand_options::get_jump_id_from_chain(*chain, pose);
		movemap->set_jump(jump_id, true);
	}

	if(sc_interface_builder_) set_all_chi(pose, movemap);
	if(bb_interface_builder_) set_all_bb(pose, movemap);

	if( minimize_water_ ) {
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
			if( ! pose.residue(i).has_property("WATER") ) continue;
			core::kinematics::Edge const & e = pose.fold_tree().get_residue_edge(i);
			if( ! e.is_jump() ) continue;
			movemap->set_jump( e.label(), true );
			MoveMapBuilder_tracer << "Minimize water jump " << e.label() << " to residue " << i << " " << pose.residue_type(i).name3() << std::endl;
		}
	}
	return movemap;
}

InterfaceBuilderOP
MoveMapBuilder::get_sc_interface_builder()const{
	assert( sc_interface_builder_); // does the pointer point
	return sc_interface_builder_;
}

void
MoveMapBuilder::set_all_chi(
		core::pose::Pose const & pose,
		core::kinematics::MoveMapOP movemap
)const{
	ligand_options::Interface side_chain_interface= sc_interface_builder_->build(pose);
	remove_ligands_to_not_minimize(pose, side_chain_interface);
	MoveMapBuilder_tracer.Debug<< "moveMap interface: "<< side_chain_interface << std::endl;
	for(core::Size i=1; i <= side_chain_interface.size(); ++i) {
		if ( side_chain_interface[i].type != ligand_options::InterfaceInfo::non_interface) { // Allow residue to minimize
			movemap->set_chi(i, true);
		}
	}
}

///@details You MUST call set_all_chi first
void
MoveMapBuilder::set_all_bb(
		core::pose::Pose const & pose,
		core::kinematics::MoveMapOP movemap
)const{
	ligand_options::Interface bb_interface = bb_interface_builder_->build(pose);

	for(core::Size i=1; i <= bb_interface.size(); ++i) {
		if ( bb_interface[i].type != ligand_options::InterfaceInfo::non_interface
				&& pose.residue(i).is_protein()
				&& movemap->get_chi(i)
		) { // Allow residue to minimize
			movemap->set_bb(i, true);
		}
	}
}

void
MoveMapBuilder::remove_ligands_to_not_minimize(
		core::pose::Pose const & pose,
		ligand_options::Interface & interface
)const{
	for ( core::Size chain_id = 1; chain_id <= pose.conformation().num_chains(); ++chain_id) {
		core::Size begin = pose.conformation().chain_begin( chain_id);
		if(pose.residue(begin).is_ligand()){
			char const chain= ligand_options::get_chain_from_chain_id(chain_id, pose);
			utility::vector1<std::string>::const_iterator found=
					std::find(chains_.begin(), chains_.end(), std::string(1,chain) );
			if( found == chains_.end()){
				core::Size const end = pose.conformation().chain_end( chain_id );
				for(; begin <= end; ++begin) interface[begin].type= ligand_options::InterfaceInfo::non_interface;
			}
		}
	}
}


} //namespace ligand_docking
} //namespace protocols
