// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/BackboneDB.cc
/// @brief
/// @author Mike Tyka

#include <protocols/loophash/BackboneDB.hh>


#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/Jump.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/RT.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>

#include <stdio.h>



using namespace core;
using namespace core::pose;
using namespace kinematics;




namespace protocols {
namespace loophash {

  static core::util::Tracer TR("BackboneDB");

  short RealAngleToShort( core::Real angle ){
    while( angle > 180.0) angle -= 360.0;
    while( angle <-180.0) angle += 360.0;
    // range for short: -32768 to 32767
    short result = short( angle * 182.0 );
    return result;
  }

  core::Real ShortToRealAngle( short angle ){
    core::Real result = core::Real( angle ) / 182.0;
    return result;
  }


  void
  BackboneSegment::apply_to_pose( core::pose::Pose &pose, core::Size ir, bool cut ) const
  {
    core::Size length = phi_.size();
    core::Size jr = ir + length;
    if(cut){
      FoldTree f;
      Size cutpoint= jr-1;

      f.add_edge( 1, ir, Edge::PEPTIDE );
      f.add_edge( ir, cutpoint, Edge::PEPTIDE );
      f.add_edge( cutpoint + 1, jr, Edge::PEPTIDE );
      f.add_edge( jr, pose.total_residue() , Edge::PEPTIDE );
      f.add_edge( ir, jr, 1 );  // this is the jump !!
      core::Size theroot = 1;
      if( ir == 1 ) theroot = pose.total_residue();
      if( f.reorder(theroot) == false ){
        std::cerr << "ERROR During reordering of fold tree - am ignoring this LOOP ! bailing: The root: " << theroot << " NRES " << pose.total_residue() << "   IR: " << ir << "  JR: " << jr << std::endl;
        return;
      }
      pose.fold_tree( f );
    }

    for( core::Size i = 0; i < length; i++){
      core::Size ires = ir + i;
      if( ires > pose.total_residue() ) return;
      pose.set_phi( ires, phi_[i] );
      pose.set_psi( ires, psi_[i] );
      pose.set_omega( ires, omega_[i] );
    }

  }

  void
  BackboneSegment::read_from_pose( core::pose::Pose const &pose, core::Size ir, core::Size length )
  {
    phi_.clear();
    psi_.clear();
    omega_.clear();

    for( core::Size i = 0; i < length; i++){
      core::Size ires = ir + i;
      if( ires > pose.total_residue() ) return;

      phi_.  push_back( pose.phi(   ires ));
      psi_.  push_back( pose.psi(   ires ));
      omega_.push_back( pose.omega( ires ));
    }

  }



  void BackboneSegment::print() const {
    for(  std::vector<core::Real>::const_iterator it = phi_.begin(); it != phi_.end(); ++it ) TR << *it << "  " ;
    for(  std::vector<core::Real>::const_iterator it = psi_.begin(); it != psi_.end(); ++it ) TR << *it << "  " ;
    for(  std::vector<core::Real>::const_iterator it = omega_.begin(); it != omega_.end(); ++it ) TR << *it << "  " ;
    TR << std::endl;
  }





  core::Real get_rmsd( const BackboneSegment &bs1, const BackboneSegment &bs2 ){
    core::Real sumsqr = 0;
    core::Size count = 0;
    if( bs1.phi().size() != bs2.phi().size() ) return -1;
    for( core::Size i = 0; i < bs1.phi().size(); i++ ){
      if( bs1.phi()[i] == 0 || bs2.phi()[i] == 0 ) continue;
      core::Real diff = bs1.phi()[i] - bs2.phi()[i];
      while( diff > 180  ) diff -= 360;
      while( diff < -180  ) diff += 360;
      sumsqr += diff*diff;
      count++;
    }
    if( bs1.psi().size() != bs2.psi().size() ) return -1;
    for( core::Size i = 0; i < bs1.psi().size(); i++ ){
      if( bs1.psi()[i] == 0 || bs2.psi()[i] == 0 ) continue;
      core::Real diff = bs1.psi()[i] - bs2.psi()[i];
      while( diff > 180  ) diff -= 360;
      while( diff < -180  ) diff += 360;
      sumsqr += diff*diff;
      count++;
    }
    if( bs1.omega().size() != bs2.omega().size() ) return -1;
    for( core::Size i = 0; i < bs1.omega().size(); i++ ){
      if( bs1.omega()[i] == 0 || bs2.omega()[i] == 0 ) continue;
      core::Real diff = bs1.omega()[i] - bs2.omega()[i];
      while( diff > 180  ) diff -= 360;
      while( diff < -180  ) diff += 360;
      sumsqr += diff*diff;
      count++;
    }

    return sqrt( sumsqr/core::Real(count) );
  }







  core::Real
  BackboneDB::angle( core::Size pos )
  {
    if( pos >= data_.size() ){
      utility_exit_with_message( "Out of bounds error" );
    }
    return ShortToRealAngle( data_[ pos ] );
  }

  void
  BackboneDB::add_pose( const core::pose::Pose &pose, core::Size &offset )
  {
    offset = data_.size(); // Index of first resdidue's phi angle
    for( core::Size i = 0; i < pose.total_residue(); i++){
      data_.push_back( RealAngleToShort(pose.phi( 1 + i )));
      data_.push_back( RealAngleToShort(pose.psi( 1 + i )));
      data_.push_back( RealAngleToShort(pose.omega( 1 + i )));
    }
  }


  void
  BackboneDB::get_backbone_segment(
      core::Size start,
      core::Size len,
      BackboneSegment &bs
  ) const
  {
    std::vector<core::Real> phi;
    std::vector<core::Real> psi;
    std::vector<core::Real> omega;
    core::Size pos = start;
    for( core::Size i = 0; i < len; i++){
      phi.push_back( ShortToRealAngle(data_[pos]) ); pos ++ ;
      psi.push_back( ShortToRealAngle(data_[pos]) ); pos ++ ;
      omega.push_back( ShortToRealAngle(data_[pos]) ); pos ++ ;
    }
    bs = BackboneSegment( phi, psi, omega );
  }



  void
  BackboneDB::write_db_to_binary(std::string filename )
  {
    // use basic C output  - C++ are too memory hungry to deal with these potentially v large files

    FILE *file = fopen( filename.c_str(), "w" );
    if( file == NULL ){
      utility_exit_with_message( "Cannot write angle DB file: " + filename + "\n" );
    }

    for( core::Size i = 0; i < data_.size(); i ++ ){
      fwrite(&data_[i],sizeof(short),1,file);
    }

    fclose( file );
  }


  void
  BackboneDB::read_db_from_binary(std::string filename )
  {
    // use basic C input - C++ are too memory hungry to deal with these potentially v large files
    FILE *file = fopen( filename.c_str(), "r" );
    if( file == NULL ){
      utility_exit_with_message( "Cannot read angle DB file" + filename + "\n" );
    }

    data_.clear();
    unsigned count = 0;
		while( !feof( file ) ){
			count++;
			TR << "C: " << count << std::endl;
      const unsigned int bufsize = 16384;
      short bufferdata[16384];
      size_t readshorts =	fread(&bufferdata[0],sizeof(short),bufsize,file);
      for( unsigned i = 0; i< readshorts; i ++ ){
        data_.push_back( bufferdata[i] );
      }
    }
    data_.reserve( data_.size()+1 );
		fclose( file );
  	TR << "End of read_db_from_binary" << std::endl;
	}



} // namespace loops
} // namespace protocols




