// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/LoopHashSampler.cc
/// @brief
/// @author Mike Tyka

#include <protocols/loophash/LoopHashSampler.hh>
#include <protocols/loophash/LoopHashMap.hh>
#include <protocols/loophash/LoopHashLibrary.hh>
#include <protocols/loophash/LocalInserter.hh>
#include <protocols/loophash/BackboneDB.hh>

#include <core/pose/util.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>
#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/kinematics/MoveMap.hh>


#if defined(WIN32) || defined(__CYGWIN__)
	#include <ctime>
#endif





namespace protocols {
namespace loophash {

  static core::util::Tracer TR("LocalHashSampler");


  // @brief create a set of structures for a the given range of residues and other parameters
 void
 LoopHashSampler::build_structures(
		const core::pose::Pose& start_pose,
    std::vector< core::io::silent::SilentStructOP > &lib_structs
	)
  {
    using namespace core;
    using namespace core::pose;
    using namespace conformation;
    using namespace kinematics;
    using namespace protocols::match;
    using namespace optimization;
    using namespace id;

		runtime_assert( library_ );
		TR.Trace << "Testing libstructs: " << "NStruct: " << lib_structs.size() << std::endl;


		long starttime = time(NULL);


    core::pose::Pose original_pose = start_pose;
    core::pose::Pose edit_pose = start_pose;
	 	TR.Debug  << "Setting options.." << std::endl;
	  core::optimization::MinimizerOptions options( "dfpmin", 0.2, true , false );
	  core::optimization::MinimizerOptions options2( "dfpmin", 0.02,true , false );

    kinematics::MoveMap final_mm;
    final_mm.set_bb(true);
   // setup movemap & minimisation

  //				for ( Size ii=ir; ii<= jr; ++ii ) {
  //					final_mm.set_bb( ii, true );
  //					if ( newpose->residue(ii).aa() == chemical::aa_pro ) final_mm.set( TorsionID( phi_torsion, BB, ii ), false );
  //				}

    Size nres = start_pose.total_residue();
    Size ir, jr;
    //Size newpep_index = 0;

    //core::Size backbone_offset;
    //bbdb_.add_pose( pose, backbone_offset );

    int runcount=0;
    runcount++;

    core::Size start_res = start_res_;
    core::Size stop_res = stop_res_;

    // figure out start and stop residues
    if ( stop_res == 0 ) stop_res = nres;  // to do the whole protein just set stop_res to 0
    start_res = std::max( start_res, (core::Size)2 ); // dont start before 2  - WHY ?
		if( start_res > stop_res ) stop_res = start_res;

	 	TR << "Running: Start:" << start_res << "  End: " << stop_res << std::endl;
    for( ir = start_res; ir <= stop_res; ir ++ ){
      for( core::Size k = 0; k < library_->hash_sizes().size(); k ++ ){
        core::Size loop_size = library_->hash_sizes()[ k ];

        jr = ir + loop_size;
        if ( ir > nres ) continue;
        if ( jr > nres ) continue;

        // get the rigid body transform for the current segment
        BackboneSegment pose_bs;
        pose_bs.read_from_pose( start_pose, ir, loop_size );
        Real6 loop_transform;
        if(!get_rt_over_leap( original_pose, ir, jr, loop_transform )) continue;

        // Look up the bin index of that transform in the hash map
        LoopHashMap &hashmap = library_->gethash( loop_size );
        std::vector < core::Size > leap_index_bucket;
        hashmap.lookup( loop_transform, leap_index_bucket );

        //TR.Info << "G: " << runcount << " " << ir << "  " << jr << " " << leap_index_bucket.size() << "  " << loop_transform[1] << "  " << loop_transform[2] << "  " << loop_transform[3] << "  " << loop_transform[4] << "  " << loop_transform[5] << "  " << loop_transform[6] << std::endl;



        // Now for every hit, get the internal coordinates and make a short list of replacement loops
        // according to the RMS criteria

        if( leap_index_bucket.size() == 0) continue;
        std::vector < core::Size > filter_leap_index_bucket;
        for(  std::vector < core::Size >::const_iterator it = leap_index_bucket.begin();
            it != leap_index_bucket.end();
            ++it ){

          // Get the actual strucure index (not just the bin index)
          core::Size retrieve_index = (core::Size) (*it);
          LeapIndex cp = hashmap.get_peptide( retrieve_index );

          // Retrieve the actual backbone structure
          BackboneSegment new_bs;
          library_->backbone_database().get_backbone_segment( cp.ba, hashmap.get_loop_size() , new_bs );

          // Check the values against against any RMS limitations imposed by the caller
          core::Real BBrms = get_rmsd( pose_bs, new_bs );
          if( ( BBrms > min_bbrms_ ) && ( BBrms < max_bbrms_ ) ){
            filter_leap_index_bucket.push_back( *it );
          }
        }

        // If no loops pass the previous filter - abort
        if( filter_leap_index_bucket.size() == 0) continue;

        // Now go through the chosen loops in random order
        core::Size explore_count = 0;
        std::random_shuffle( filter_leap_index_bucket.begin(), filter_leap_index_bucket.end());
        for(  std::vector < core::Size >::const_iterator it = filter_leap_index_bucket.begin();
            it != filter_leap_index_bucket.end();
            ++it ){

          explore_count ++;

          clock_t starttime = clock();


          core::Size retrieve_index = *it;
          LeapIndex cp = hashmap.get_peptide( retrieve_index );

          BackboneSegment new_bs;
          library_->backbone_database().get_backbone_segment( cp.ba, hashmap.get_loop_size() , new_bs );

          //core::Real BBrms = get_rmsd( pose_bs, new_bs );
          // Distance measures of end point


          if( TR.Debug.visible() ){
            core::Real xyzdist = sqrt(sqr(loop_transform[1] - cp.vecx) + sqr(loop_transform[2] - cp.vecy) + sqr(loop_transform[3] - cp.vecz));
            core::Real ang1 = loop_transform[4] - cp.rotx; while( ang1 > 180 ) ang1 -= 360.0; while( ang1 < -180.0 ) ang1 += 360.0;
            core::Real ang2 = loop_transform[5] - cp.roty; while( ang2 > 180 ) ang2 -= 360.0; while( ang2 < -180.0 ) ang2 += 360.0;
            core::Real ang3 = loop_transform[6] - cp.rotz; while( ang3 > 180 ) ang3 -= 360.0; while( ang3 < -180.0 ) ang3 += 360.0;
            core::Real angdist = sqrt(sqr(ang1)+sqr(ang2)+sqr(ang3) );
            TR.Info << "  X: " << xyzdist << "  " << angdist << "  " << cp.rotx << "  " << cp.roty << "  " << cp.rotz << std::endl;
          }


          core::pose::Pose newpose( start_pose );
          transfer_phi_psi( start_pose, newpose );

          core::Real final_rms = inserter_->make_local_bb_change( newpose, original_pose, new_bs, ir );


///          // set newpose
///          protocols::loops::Loops exclude_region;
///          exclude_region.add_loop( protocols::loops::Loop( ir, jr ) );
///          core::pose::Pose newpose( start_pose );
///          transfer_phi_psi( start_pose, newpose );
///          add_coordinate_constraints_to_pose( newpose, original_pose, exclude_region );
///          new_bs.apply_to_pose( newpose, ir );
///          //scorefxn_rama_cst.show( TR.Info, *newpose );
///
///
///          // just for comparison with cut!
///          //core::pose::PoseOP newpose2( new Pose( original_pose ) );
///          //new_bs.apply_to_pose( *newpose2, ir, true );
///          //newpose2->dump_pdb("rep_" + utility::to_string( ir ) + "_" + utility::to_string( jr ) + "_" + utility::to_string( int(xyzdist) ) + "_" + utility::to_string( int(angdist) ) + ".cut.pdb" );
///
///
///
///
///          //scorefxn_rama_cst.show( TR.Info, *newpose );
///          //newpose->dump_pdb("rep_" + utility::to_string( ir ) + "_" + utility::to_string( jr ) + "_" + utility::to_string( int(xyzdist) ) + "_" + utility::to_string( int(angdist) ) + ".bef.pdb" );
///          AtomTreeMinimizer().run( newpose, final_mm, scorefxn_rama_cst_, options );
///          //scorefxn_rama_cst.show( TR.Info, *newpose );
///          //newpose->dump_pdb("rep_" + utility::to_string( ir ) + "_" + utility::to_string( jr ) + "_" + utility::to_string( int(xyzdist) ) + "_" + utility::to_string( int(angdist) ) + ".aft.pdb" );
///          //newpose->dump_pdb("rep_" + utility::to_string( ir ) + "_" + utility::to_string( jr ) + "_" + utility::to_string( int(xyzdist) ) + "_" + utility::to_string( int(angdist) ) + ".pdb" );
///
///          core::Real premin_rms = core::scoring::CA_rmsd( newpose, original_pose );
///          TR.Info << "Premin RMS: " << premin_rms << std::endl;
///          TR.Info << "Min Score3 " << std::endl;
///          //scorefxn_cen_cst.show( TR.Info, *newpose );
///          AtomTreeMinimizer().run( newpose, final_mm, scorefxn_cen_cst_, options2 );
///          //scorefxn_cen_cst.show( TR.Info, *newpose );
///
///          // get final RMS
///
///          core::Real final_rms = core::scoring::CA_rmsd( newpose, original_pose );
///          TR.Info << "Final RMS: " << final_rms << std::endl;

					bool isok = false;
          if ( ( final_rms < max_rms_ ) && ( final_rms > min_rms_ ) ){

            core::pose::Pose mynewpose( start_pose );

            transfer_phi_psi( newpose, mynewpose );

            core::io::silent::SilentStructOP new_struct = core::io::silent::SilentStructFactory::get_silent_struct_out();
            new_struct->fill_struct( mynewpose );      // make the silent struct from the copy pose
            new_struct->energies_from_pose( newpose ); // take energies from the modified pose, not the copy pose
						//TR << "SAMPLER: " << new_struct->get_energy("censcore") << std::endl;
						lib_structs.push_back( new_struct );
          	isok = true;
					}

          //if ( lib_structs.size() > 2  ) return;

          clock_t endtime = clock();

          TR.Debug << "Clocks: " << endtime - starttime << "  " << final_rms << (isok ? " OK" : " Reject") << std::endl;

        }
      }
    }


		long endtime = time(NULL);


		TR.Info << "LHS: " << start_res << "-" << stop_res << ":  " <<lib_structs.size() << "structures " << endtime - starttime << " seconds" << std::endl;

		for( std::vector< core::io::silent::SilentStructOP >::iterator it=lib_structs.begin();
				it != lib_structs.end(); ++it ){
			TR.Debug << "SAMPLER2" << (*it)->get_energy("censcore") << std::endl;
		}



	}






} // namespace loops
} // namespace protocols




