// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Mike Tyka

#ifndef INCLUDED_protocols_loops_LoopMover_QuickCCD_HH
#define INCLUDED_protocols_loops_LoopMover_QuickCCD_HH

// package headers
#include <protocols/loops/IndependentLoopMover.hh>
#include <protocols/moves/Mover.hh>

// project headers
#include <core/types.hh>
#include <core/fragment/FragSet.fwd.hh>
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

// C++ Headers


namespace protocols {
namespace loops {


/// @brief LoopMover utilizing fragment insertion, ccd loop closure, and
/// minimization
class LoopMover_Perturb_QuickCCD: public IndependentLoopMover {

public: // construct/destruct

	/// @brief Loops constructor
	/// @param[in] loops_in the set of loops to model
	/// @param[in] frags_from_file read fragments from files specified on command line?
	/// @remarks Will be initialized with centroid level score function 'score4L'.
	LoopMover_Perturb_QuickCCD(
		protocols::loops::Loops loops_in
	);

	/// @brief Loops & ScoreFunction constructor
	/// @param[in] loops_in the set of loops to model
	/// @param[in] scorefxn desired ScoreFunction
	/// @param[in] frags_from_file read fragments from files specified on command
	/// line?
	LoopMover_Perturb_QuickCCD(
		protocols::loops::Loops loops_in,
		core::scoring::ScoreFunctionOP scorefxn
	);

	/// @brief Loops & ScoreFunction constructor
	/// @param[in] loops_in the set of loops to model
	/// @param[in] scorefxn desired ScoreFunction
	/// @param[in] fragset is the FragSet to be used
	/// line?
	LoopMover_Perturb_QuickCCD(
		protocols::loops::Loops loops_in,
		core::scoring::ScoreFunctionOP scorefxn,
		core::fragment::FragSetOP fragset
	);

	//destructor
	~LoopMover_Perturb_QuickCCD();

	virtual std::string get_name() const;

public: // virtual constructors

	/// @brief clone this object
	virtual protocols::moves::MoverOP clone() const;

public: // accessors

	/// @brief randomize loops prior to loop modeling?
	inline
	bool randomize_loop() const {
		return randomize_loop_;
	}


public: // mutators


	/// @brief indicate whether loops should be randomized prior to modeling
	inline
	void randomize_loop( bool const flag ) {
		randomize_loop_ = flag;
	}


	/// @brief set default settings
	/// @details default settings are as follows:
	///  <ul>
	///      <li> randomize_loop() = true
	///  </ul>
	void set_default_settings() {
		randomize_loop_ = true;
	}

protected: // virtual loop operations

	/// @brief loop modeling protocol implementation
	LoopResult model_loop(
		core::pose::Pose & pose,
		protocols::loops::Loop const & loop
	);

protected: // data. should be private!

	/// @brief randomize loops prior to performing loop modeling?
	bool randomize_loop_;
}; // class LoopMover_Perturb_QuickCCD


void fast_ccd_close_loops(
	core::pose::Pose & pose,
	protocols::loops::Loop const & loop,
	core::kinematics::MoveMap & mm
);

} //namespace loops
} //namespace protocols

#endif //INCLUDED_protocols_loops_LoopMover_QuickCCD_HH
