// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/output/MatchConsolidator.cc
/// @brief
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/output/MatchConsolidator.hh>

// Package headers
#include <protocols/match/Hit.hh>

#include <protocols/match/output/MatchEvaluator.hh>
#include <protocols/match/output/MatchFilter.hh>
#include <protocols/match/output/MatchGrouper.hh>
#include <protocols/match/output/OutputWriter.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/sort_predicates.hh>
#include <utility/vector1.functions.hh>

// C++ headers
#include <algorithm>

namespace protocols {
namespace match {
namespace output {

MatchConsolidator::MatchConsolidator() : n_to_output_per_group_( 5 ) {}

MatchConsolidator::~MatchConsolidator() {}

void
MatchConsolidator::begin_processing()
{
	runtime_assert( grouper_ );
	runtime_assert( evaluator_ );
	MatchProcessor::begin_processing();
	match_groups_.clear();
	grouper_->reset();

}

void
MatchConsolidator::process_match(
	match const & m
)
{
	runtime_assert( grouper_ );
	runtime_assert( evaluator_ );
	runtime_assert( writer_ );
	runtime_assert( match_dspos1_groups_.size() == 0 );
	note_match_processed();

	for ( std::list< MatchFilterOP >::const_iterator
			iter = filters_.begin(), iter_end = filters_.end();
			iter != iter_end; ++iter ) {
		if (  ! (*iter)->passes_filter( m ) ) {
			note_filter_fail( (*iter)->filter_name() );
			return;
		}
	}

	Size group = grouper_->assign_group_for_match( m );

	if ( group > match_groups_.size() ) {
		match_groups_.resize( group );
	}

	match_groups_[ group ].push_back( m );
}

void
MatchConsolidator::process_match(
	match_dspos1 const & m
)
{
	runtime_assert( grouper_ );
	runtime_assert( evaluator_ );
	runtime_assert( writer_ );
	runtime_assert( match_groups_.size() == 0 );
	note_match_processed();

	for ( std::list< MatchFilterOP >::const_iterator
			iter = filters_.begin(), iter_end = filters_.end();
			iter != iter_end; ++iter ) {
		if (  ! (*iter)->passes_filter( m ) ) {
			note_filter_fail( (*iter)->filter_name() );
			return;
		}
	}

	Size group = grouper_->assign_group_for_match( m );

	if ( group > match_dspos1_groups_.size() ) {
		match_dspos1_groups_.resize( group );
	}

	match_dspos1_groups_[ group ].push_back( m );

}


void
MatchConsolidator::end_processing()
{
	/// Can only handle regular match objects or match_dspos1's but not both.
	runtime_assert( match_dspos1_groups_.size() == 0 || match_groups_.size() == 0 );

	runtime_assert( grouper_ );
	runtime_assert( evaluator_ );

	MatchProcessor::end_processing();
	if ( match_dspos1_groups_.size() != 0 ) {
		end_processing_of_match_dspos1_groups();
	} else {
		end_processing_of_regular_match_groups();
	}
}

void
MatchConsolidator::end_processing_of_regular_match_groups()
{

	for ( Size ii = 1; ii <= match_groups_.size(); ++ii ) {
		Size ii_n_matches = match_groups_[ ii ].size();
		utility::vector1< match > ii_matches( ii_n_matches );
		utility::vector1< Real  > ii_scores(  ii_n_matches, 0.0 );
		std::copy( match_groups_[ ii ].begin(), match_groups_[ ii ].end(), ii_matches.begin() );
		for ( Size jj = 1; jj <= ii_n_matches; ++jj ) {
			ii_scores[ jj ] = evaluator_->score( ii_matches[ jj ] );
		}
		utility::vector1< Size > top_score_indices( n_to_output_per_group_ );

		// either sort or, if it's going to be fast, use utility::arg_least_several
		if ( n_to_output_per_group_ > 50 ) {
			utility::vector1< std::pair< Real, Size > > score_index_pairs( ii_n_matches );
			for ( Size jj = 1; jj <= ii_n_matches; ++jj ) {
				score_index_pairs[ jj ] = std::make_pair( ii_scores[ jj ], jj );
			}
			std::sort( score_index_pairs.begin(), score_index_pairs.end(), utility::SortFirst< Real, Size >() );
			for ( Size jj = 1; jj <= n_to_output_per_group_; ++jj ) {
				if ( jj > ii_n_matches ) {
					top_score_indices.resize( ii_n_matches );
					break;
				}
				top_score_indices[ jj ] = score_index_pairs[ jj ].second;
			}
		} else {
			utility::arg_least_several( ii_scores, top_score_indices );
		}

		for ( Size jj = 1; jj <= top_score_indices.size(); ++jj ) {
			writer_->record_match( ii_matches[ top_score_indices[ jj ]] );
		}
	}
}

void
MatchConsolidator::end_processing_of_match_dspos1_groups()
{

	for ( Size ii = 1; ii <= match_dspos1_groups_.size(); ++ii ) {
		Size ii_n_matches = match_dspos1_groups_[ ii ].size();
		utility::vector1< match_dspos1 > ii_matches( ii_n_matches );
		utility::vector1< Real >         ii_scores(  ii_n_matches, 0.0 );
		std::copy( match_dspos1_groups_[ ii ].begin(), match_dspos1_groups_[ ii ].end(), ii_matches.begin() );
		for ( Size jj = 1; jj <= ii_n_matches; ++jj ) {
			ii_scores[ jj ] = evaluator_->score( ii_matches[ jj ] );
		}
		utility::vector1< Size > top_score_indices( n_to_output_per_group_ );

		// either sort or, if it's going to be fast, use utility::arg_least_several
		if ( n_to_output_per_group_ > 50 ) {
			utility::vector1< std::pair< Real, Size > > score_index_pairs( ii_n_matches );
			for ( Size jj = 1; jj <= ii_n_matches; ++jj ) {
				score_index_pairs[ jj ] = std::make_pair( ii_scores[ jj ], jj );
			}
			std::sort( score_index_pairs.begin(), score_index_pairs.end(), utility::SortFirst< Real, Size >() );
			for ( Size jj = 1; jj <= n_to_output_per_group_; ++jj ) {
				if ( jj > ii_n_matches ) {
					top_score_indices.resize( ii_n_matches );
					break;
				}
				top_score_indices[ jj ] = score_index_pairs[ jj ].second;
			}
		} else {
			utility::arg_least_several( ii_scores, top_score_indices );
		}

		for ( Size jj = 1; jj <= top_score_indices.size(); ++jj ) {
			writer_->record_match( ii_matches[ top_score_indices[ jj ]] );
		}
	}
}


void
MatchConsolidator::set_n_to_output_per_group( Size setting )
{
	n_to_output_per_group_ = setting;
}

void
MatchConsolidator::add_filter( MatchFilterOP filter )
{
	filters_.push_back( filter );
}

void
MatchConsolidator::set_grouper( MatchGrouperOP grouper )
{
	grouper_ = grouper;
}

void
MatchConsolidator::set_evaluator( MatchEvaluatorOP evaluator )
{
	evaluator_ = evaluator;
}

void
MatchConsolidator::reset_grouper()
{
	if ( grouper_ ) grouper_->reset();
}


}
}
}
