// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file ./protocols/moves/MakePolyXMover.cc
/// @brief  convert pose to poly XXX: any amino acid, default poly Ala
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )

// Unit headers
#include <protocols/moves/MakePolyXMover.hh>
#include <protocols/moves/MakePolyXMoverCreator.hh>

// Project Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/util/Tracer.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/toolbox/pose_manipulation.hh>

#include <utility/Tag/Tag.hh>
#include <string>

//Auto Headers
#include <protocols/jobdist/Jobs.hh>
#include <iostream>


static core::util::Tracer TR( "protocols.moves.MakePolyXMover" );

namespace protocols {
namespace moves {

std::string
MakePolyXMoverCreator::keyname() const
{
	return MakePolyXMoverCreator::mover_name();
}

protocols::moves::MoverOP
MakePolyXMoverCreator::create_mover() const {
	return new MakePolyXMover;
}

std::string
MakePolyXMoverCreator::mover_name()
{
	return "MakePolyX";
}

MakePolyXMover::MakePolyXMover():
	Mover( MakePolyXMoverCreator::mover_name() )
{}

MakePolyXMover::~MakePolyXMover()
{}

/// @brief clone this object
MakePolyXMover::MoverOP MakePolyXMover::clone() const {
	return new MakePolyXMover( *this );
}

/// @brief create this type of object
MakePolyXMover::MoverOP MakePolyXMover::fresh_instance() const {
	return new MakePolyXMover();
}

/// @details virtual main
void MakePolyXMover::apply( Pose & pose )
{
	// flip to poly-ala-gly-pro-disulf pose
	utility::vector1< Size > protein_residues;
	for ( Size i = 1, ie = pose.n_residue(); i <= ie; ++i ) {
		if ( pose.residue( i ).is_protein() ) {
			protein_residues.push_back( i );
		}
	}
	using protocols::toolbox::pose_manipulation::construct_poly_XXX_pose;
	TR << "Pose is converted to poly " << aa_ << std::endl;
	construct_poly_XXX_pose( aa_, pose, protein_residues, keep_pro_, keep_gly_, keep_disulfide_cys_ );
}

std::string
MakePolyXMover::get_name() const {
	return MakePolyXMoverCreator::mover_name();
}

/// @brief parse xml
void
MakePolyXMover::parse_my_tag(
	TagPtr const tag,
	DataMap &,
	Filters_map const &,
	Movers_map const &,
	Pose const & )
{
	aa_ = tag->getOption<std::string>( "aa", "ALA" );
	keep_pro_  = tag->getOption<bool>( "keep_pro", 0 );
	keep_gly_  = tag->getOption<bool>( "keep_gly", 1 );
	keep_disulfide_cys_  = tag->getOption<bool>( "keep_disulufide_cys", 0 );

	TR << "MakePolyXMover was loaded" << std::endl;

	if( keep_pro_ || keep_gly_ || keep_disulfide_cys_ ) {
		TR << "but keep AA types of ";
		if( keep_pro_ ) TR << "Pro ";
		if( keep_gly_ ) TR << "Gly  ";
		if( keep_disulfide_cys_ ) TR << "Disulfide Cys";
		TR << std::endl;
	}

}

}  // namespace moves
}  // namespace protocols
