// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/ParsedProtocol.cc
/// @author Sarel Fleishman (sarelf@u.washington.edu)

// Unit Headers
#include <protocols/moves/ParsedProtocol.hh>
#include <protocols/moves/ParsedProtocolCreator.hh>
#include <protocols/moves/NullMover.hh>

#include <protocols/viewer/viewers.hh>
// Project Headers
//#include <protocols/moves/ResidueMover.hh>
#include <protocols/moves/Mover.hh>
#include <core/pose/Pose.hh>
#include <protocols/moves/MoverStatus.hh>

#include <core/kinematics/Jump.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/chemical/util.hh>

#include <core/util/Tracer.hh>


#include <protocols/filters/Filter.hh>
#include <protocols/filters/BasicFilters.hh>
#include <utility/Tag/Tag.hh>
#include <protocols/moves/ResId.hh>

// JD2 headers
#include <protocols/jd2/JobDistributor.hh>

// Utility Headers

#include <core/conformation/symmetry/util.hh>

// C++ headers
#include <map>
#include <string>

//Auto Headers
#include <protocols/jobdist/Jobs.hh>
#include <utility/options/keys/BooleanOptionKey.hh>


namespace protocols {
namespace moves {

static core::util::Tracer TR( "protocols.moves.ParsedProtocol" );
static core::util::Tracer TR_report( "protocols.moves.ParsedProtocol.REPORT" );

typedef core::Real Real;
typedef core::pose::Pose Pose;

using namespace core;
using namespace std;

std::string
ParsedProtocolCreator::keyname() const
{
	return ParsedProtocolCreator::mover_name();
}

protocols::moves::MoverOP
ParsedProtocolCreator::create_mover() const {
	return new ParsedProtocol;
}

std::string
ParsedProtocolCreator::mover_name()
{
	return "ParsedProtocol";
}

/// @detailed Takes care of the docking, design and filtering moves. pre_cycle and pose_cycle can
/// be setup in derived classes to setup variables before and after these cycles.
void
ParsedProtocol::apply( Pose & pose )
{
//	runtime_assert( movers_.size() );

	protocols::moves::Mover::set_last_move_status( protocols::moves::FAIL_RETRY );
	protocols::viewer::add_conformation_viewer( pose.conformation(), "start_pose" );

	pose.update_residue_neighbors();
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it ) {
		std::string const mover_name( mover_it->first->get_name() );
		std::string const filter_name( mover_it->second->get_user_defined_name() );

		(*mover_it).first->set_native_pose( get_native_pose() );
		TR<<"=======================BEGIN MOVER "<<mover_name<<"=======================\n{"<<std::endl;
		(*mover_it).first->apply( pose );
		TR<<"\n}\n=======================END MOVER "<<mover_name<<"======================="<<std::endl;
		// collect Mover info: jd2 JobDistributor passes this info to Job,
		// and JobOutputters may then write this info to output files
		TR<<"=======================BEGIN FILTER "<<filter_name<<"=======================\n{"<<std::endl;
		info().insert( info().end(), mover_it->first->info().begin(), mover_it->first->info().end() );
		pose.update_residue_neighbors();
		moves::MoverStatus status( (*mover_it).first->get_last_move_status() );
		bool const pass( status==protocols::moves::MS_SUCCESS  && (*mover_it).second->apply( pose ) );
		TR<<"\n}\n=======================END FILTER "<<filter_name<<"======================="<<std::endl;
		if( pass ) {
			if( (unsigned int) ( mover_it - movers_.begin() + 1) == movers_.size() ) { // return successfully only on last mover
				protocols::moves::Mover::set_last_move_status( protocols::moves::MS_SUCCESS ); // tell jobdistributor to save pose
				TR<<"setting status to success"<<std::endl;

				// report filter values to the job object as string_real_pair
				report_filters_to_job( pose );
				// report filter values to tracer output
				report_all( pose );
				// rescore the pose with either score12 or a user-specified scorefunction. this ensures that all output files end up with scores.
				core::scoring::ScoreFunctionOP scorefxn = core::scoring::getScoreFunction();
				(*scorefxn)(pose);
				return;
			}
		}
		else{ // fail
			if( status != protocols::moves::MS_SUCCESS )
				protocols::moves::Mover::set_last_move_status( status );
			return;
		}
	}
}

std::string
ParsedProtocol::get_name() const {
	return ParsedProtocolCreator::mover_name();
}

void
ParsedProtocol::report_all( Pose const & pose ) const {
	TR_report<<"=============Starting final report================"<<std::endl;
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it ){
		TR_report<<"============Begin report for "<<(*mover_it).second->get_user_defined_name()<<"=================="<<std::endl;
		(*mover_it).second->report( TR_report, pose );
		TR_report<<"============End report for "<<(*mover_it).second->get_user_defined_name()<<"=================="<<std::endl;
	}
	TR_report.flush();
}

void
ParsedProtocol::report_filters_to_job( Pose const & pose) const {
	using protocols::jd2::JobDistributor;
	protocols::jd2::JobOP job_me( JobDistributor::get_instance()->current_job() );
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
			 mover_it!=movers_.end(); ++mover_it ) {
		core::Real const filter_value( (*mover_it).second->report_sm( pose ) );
		if( filter_value > -9999 )
			job_me->add_string_real_pair((*mover_it).second->get_user_defined_name(), filter_value);
	}
}

void
ParsedProtocol::report_all_sm( std::map< std::string, core::Real > & score_map, Pose const & pose ) const {
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it ) {
		 core::Real const filter_value( (*mover_it).second->report_sm( pose ) );
		 if( filter_value >= -9999 )
			score_map[ (*mover_it).second->get_user_defined_name() ] = filter_value;
	}
}

ParsedProtocol::iterator
ParsedProtocol::begin(){
	return movers_.begin();
}

ParsedProtocol::const_iterator
ParsedProtocol::begin() const{
	return movers_.begin();
}

ParsedProtocol::iterator
ParsedProtocol::end(){
	return movers_.end();
}

ParsedProtocol::const_iterator
ParsedProtocol::end() const{
	return movers_.end();
}

/// @details sets resid for the constituent filters and movers
void
ParsedProtocol::set_resid( core::Size const resid ){
	for( iterator it( movers_.begin() ); it!=movers_.end(); ++it ){
		using namespace protocols::moves;
		modify_ResId_based_object( it->first, resid );
		modify_ResId_based_object( it->second, resid );
	}
}

void
ParsedProtocol::parse_my_tag( TagPtr const tag,
	protocols::moves::DataMap &,
	protocols::filters::Filters_map const &filters,
	protocols::moves::Movers_map const &movers,
	core::pose::Pose const & )
{
	using namespace protocols::moves;
	using namespace utility::Tag;

	TR<<"ParsedProtocol mover with the following movers and filters\n";
	utility::vector0< TagPtr > const dd_tags( tag->getTags() );
	for( utility::vector0< TagPtr >::const_iterator dd_it=dd_tags.begin(); dd_it!=dd_tags.end(); ++dd_it ) {
		TagPtr const tag_ptr = *dd_it;

		MoverOP mover_to_add;
		protocols::filters::FilterOP filter_to_add;

		bool mover_defined( false ), filter_defined( false );

		std::string mover_name="null"; // user must specify a mover name. there is no valid default.
		runtime_assert( !( tag_ptr->hasOption("mover_name") && tag_ptr->hasOption("mover") ) );
		if( tag_ptr->hasOption( "mover_name" ) ){
			mover_name = tag_ptr->getOption<string>( "mover_name", "null" );
			mover_defined = true;
		}
		else if( tag_ptr->hasOption( "mover" ) ){
			mover_name = tag_ptr->getOption<string>( "mover", "null" );
			mover_defined = true;
		}
		//runtime_assert( mover_name ); // redundant with mover find below

		std::string filter_name="true_filter"; // used in case user does not specify a filter name.
		runtime_assert( !( tag_ptr->hasOption("filter_name") && tag_ptr->hasOption( "filter" ) ) );
		if( tag_ptr->hasOption( "filter_name" ) ){
			filter_name = tag_ptr->getOption<string>( "filter_name", "true_filter" );
			filter_defined = true;
		} else if( tag_ptr->hasOption( "filter" ) ){
			filter_name = tag_ptr->getOption<string>( "filter", "true_filter" );
			filter_defined = true;
		}

		if( mover_defined ){
			Movers_map::const_iterator find_mover( movers.find( mover_name ) );
			if( find_mover == movers.end() ) {
				TR.Error<<"mover not found in map. skipping:\n"<<tag_ptr<<std::endl;
				runtime_assert( find_mover != movers.end() );
				continue;
			}
			mover_to_add = find_mover->second;
		}	else {
			mover_to_add = new NullMover;
		}
		if( filter_defined ){
			protocols::filters::Filters_map::const_iterator find_filter( filters.find( filter_name ));
			if( find_filter == filters.end() ) {
				TR.Error<<"filter not found in map. skipping:\n"<<tag_ptr<<std::endl;
				runtime_assert( find_filter != filters.end() );
				continue;
			}
			filter_to_add = find_filter->second;
		} else {
			filter_to_add = new protocols::filters::TrueFilter;
		}
		add_mover( mover_to_add, filter_to_add );
		TR << "added mover \"" << mover_name << "\" with filter \"" << filter_name << "\"\n";
	}
	TR.flush();
}

} //ProteinInterfaceDesign
} //protocols

