// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/moves/RollMover.cc
/// @brief RollMover methods implemented
/// @author

// Unit Headers
#include <protocols/moves/RollMover.hh>
// Package Headers

// Project Headers
#include <core/conformation/Residue.hh>
#include <core/pose/Pose.hh>
#include <core/id/AtomID.hh>
#include <numeric/xyz.functions.hh>
// AUTO-REMOVED #include <numeric/xyz.io.hh>
// Random number generator
#include <numeric/random/random.hh>
// Utility Headers
#include <core/util/Tracer.hh>
#include <core/types.hh>

// C++ Headers

using core::util::T;
using core::util::Error;
using core::util::Warning;
static numeric::random::RandomGenerator RG(456732);
static core::util::Tracer TR( "protocols.moves.RollMover" );

using namespace core;

namespace protocols {
namespace moves {

///@details
void RollMover::apply( core::pose::Pose & pose ){

	utility::vector1< utility::vector1< numeric::xyzVector< core::Real> > >  coords; // some of these will change for sure

	// now we have a vector1 of vector1s with a numeric::xyzVector
	// access will look like coords[residue][atom] to get xyz

	core::Size const nres( pose.total_residue() );
  coords.resize( nres );

  for ( Size i=1; i<= nres; ++i ) {
		core::conformation::Residue const & rsd( pose.residue(i) );
		core::Size const number_atoms_this_residue( rsd.natoms() );
    if ( number_atoms_this_residue ) {
      coords[i].resize( number_atoms_this_residue );
      for ( Size j=1; j <= number_atoms_this_residue; ++j ) {
        coords[i][j] = rsd.atom( j ).xyz();
			}
		}
	}

	angle_ = min_angle_ + ( max_angle_ - min_angle_ ) * RG.uniform();
	numeric::xyzMatrix< core::Real > rotation_matrix( numeric::rotation_matrix_degrees(axis_, angle_ ) );
	//move to origin
	for ( core::Size i =start_res_; i <= stop_res_; ++i ) {
		for ( core::Size j = 1; j <= coords[i].size(); ++j ) {

			// this may look strange but in a global coordinate system
			// rotation about an axis is easily done by movement to the origin
			// rotation and then movement back

			coords[i][j] = coords[i][j] - translate_; // translate to origin
      coords[i][j] = rotation_matrix * coords[i][j]; // rotate atom
      coords[i][j] = coords[i][j] + translate_; // reverse translate

    }
  }

	// now update pose with new coordinates
  for ( core::Size i =start_res_; i <= stop_res_; ++i ) {
		for ( core::Size j = 1; j <= coords[i].size(); ++j ) {
			id::AtomID id( j, i );
			pose.set_xyz( id, coords[i][j]);
		}
	}



}//apply

std::string
RollMover::get_name() const {
	return "RollMover";
}

///@brief
RollMover::RollMover(
) : Mover()
{
	Mover::type( "RollMover" );
}

	RollMover::RollMover( core::Size start_res, core::Size stop_res, core::Real min_angle, core::Real max_angle, numeric::xyzVector< core::Real > axis, numeric::xyzVector< core::Real > translate
												): Mover(), start_res_(start_res), stop_res_(stop_res), min_angle_(min_angle), max_angle_(max_angle), axis_(axis), translate_(translate)
{
	Mover::type( "RollMover" );
}

RollMover::~RollMover(){}

}//moves
}//protocols

