// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/rotamer_recovery/RRComparer.cc
/// @author Matthew O'Meara (mattjomeara@gmail.com)
/// Adapted from:
/// protocols::optimize_weights::IterativeOptEDriver::measure_rotamer_recovery()
/// and apps::pilot::doug::rotamer_prediction_benchmark()


// Unit Headers
#include <protocols/rotamer_recovery/RRReporter.hh>

// Project Headers
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/conformation/Residue.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <protocols/jd2/Job.hh>
#include <protocols/jd2/JobDistributor.hh>

// Utility Headers
#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/CacheableString.hh>




//C++ Headers
#include <ostream>
#include <string>

using std::endl;
using std::ostream;
using std::string;
using core::Real;
using core::pose::Pose;
using core::conformation::Residue;
using core::util::Tracer;
using protocols::jd2::JobDistributor;
using core::pose::datacache::CacheableDataType::JOBDIST_OUTPUT_TAG;
using core::util::datacache::CacheableString;

#ifdef DB_SQLITE3
using utility::sql_database::Sqlite3Interface;
using utility::sql_database::begin_row;
#endif

namespace protocols {
namespace rotamer_recovery {

static Tracer TR("protocol.rotamer_recovery.RRReporter");

RRReporterSimple::RRReporterSimple() :
	residues_considered_( 0 ),
	rotamers_recovered_( 0 )
{}

RRReporterSimple::RRReporterSimple( RRReporterSimple const & src ) :
	RRReporter(),
	residues_considered_(src.residues_considered_),
	rotamers_recovered_(src.rotamers_recovered_)
{}

RRReporterSimple::~RRReporterSimple() {}


//void
//RRReporterSimple::set_comparer_info(
//	string const & /*comparer_name*/,
//	string const & /*comparer_params*/
//){ }

void
RRReporterSimple::reset_recovery(){
	residues_considered_=0;
	rotamers_recovered_=0;
}

void
RRReporterSimple::report_rotamer_recovery(
	Pose const & /*pose1*/,
	Pose const & /*pose2*/,
	Residue const & /*res1*/,
	Residue const & /*res2*/,
	Real const /*score*/,
	bool const recovered
) {
	residues_considered_++;
	rotamers_recovered_ += recovered;
}

void
RRReporterSimple::show( ostream & out ) const {

	out << "Recovered " << rotamers_recovered_
		<< " at " << residues_considered_
		<< " for a recovery rate of " << recovery_rate() << "." << endl;
}

void
RRReporterSimple::show( ) const {

	TR << "Recovered " << rotamers_recovered_ << " rotamers"
		<< " at " << residues_considered_ << " locations"
		<< " for a recovery rate of " << recovery_rate() << "." << endl;
}


Real
RRReporterSimple::recovery_rate() const {
	return Real(rotamers_recovered_) / Real(residues_considered_);
}

#ifdef DB_SQLITE3

RRReporterSQLite::RRReporterSQLite() :
	comparer_name_(),
	comparer_params_(),
	residues_considered_( 0 ),
	rotamers_recovered_( 0 ),
	db_interface_( "rotamer_recovery.db3" )
{}

RRReporterSQLite::RRReporterSQLite(
	string const & database_fname
) :
	comparer_name_(),
	comparer_params_(),
	residues_considered_( 0 ),
	rotamers_recovered_( 0 ),
	db_interface_( database_fname )
{
	db_interface_.execute_sql(schema());
}

RRReporterSQLite::RRReporterSQLite( RRReporterSQLite const & src ) :
	RRReporter(),
	comparer_name_( src.comparer_name_ ),
	comparer_params_( src.comparer_params_ ),
	residues_considered_( src.residues_considered_ ),
	rotamers_recovered_( rotamers_recovered_ ),
	db_interface_( src.db_interface_ )
{}

RRReporterSQLite::~RRReporterSQLite() {}

string
RRReporterSQLite::schema() const{
	return "\
CREATE TABLE IF NOT EXISTS rotamer_recovery (\
	struct1_name TEXT,\
	chain1 TEXT,\
	res1 INTEGER,\
	struct2_name TEXT,\
	chain2 TEXT,\
	res2 INTEGER,\
	comparer_name TEXT,\
	comparer_params TEXT,\
	score REAL,\
	recovered BOOLEAN);\
";
}

void
RRReporterSQLite::set_comparer_info(
	string const & comparer_name,
	string const & comparer_params
)	{
	comparer_name_ = comparer_name;
	comparer_params_ = comparer_params;
}

void
RRReporterSQLite::reset_recovery(){
	residues_considered_=0;
	rotamers_recovered_=0;
}

void
RRReporterSQLite::report_rotamer_recovery(
	Pose const & pose1,
	Pose const & pose2,
	Residue const & res1,
	Residue const & res2,
	Real const score,
	bool recovered
){

	// Argh why isn't there a good way go get the name of a pose?
	//silent files and pdbs set the name of the pose differently
	string struct1_name = "No_Name_Found";
	if (pose1.pdb_info() && ( pose1.pdb_info()->name() != "" ) ){
		struct1_name = pose1.pdb_info()->name();
	} else if ( pose1.data().has( JOBDIST_OUTPUT_TAG ) ) {
		struct1_name = static_cast< CacheableString const & >
			( pose1.data().get( JOBDIST_OUTPUT_TAG ) ).str();
	} else {
		struct1_name = JobDistributor::get_instance()->current_job()->input_tag();
	}

	string struct2_name = "No_Name_Found";
	if (pose2.pdb_info() && ( pose1.pdb_info()->name() != "" ) ){
		struct2_name = pose2.pdb_info()->name();
	} else if ( pose2.data().has( JOBDIST_OUTPUT_TAG ) ) {
		struct2_name = static_cast< CacheableString const & >
			( pose2.data().get( JOBDIST_OUTPUT_TAG ) ).str();
	} else {
		struct2_name = JobDistributor::get_instance()->current_job()->input_tag();
	}

	db_interface_ << begin_row("rotamer_recovery")
		<< struct1_name
		<< res1.chain()
		<< res1.seqpos()
		<< struct2_name
		<< res2.chain()
		<< res2.seqpos()
		<< comparer_name_
		<< comparer_params_
		<< score
		<< recovered
		<< Sqlite3Interface::end_row;

	residues_considered_++;
	rotamers_recovered_ += recovered;

}

void
RRReporterSQLite::show( ostream & out ) const {

	out << "Recovered " << rotamers_recovered_
		<< " at " << residues_considered_
		<< " for a recovery rate of " << recovery_rate() << "." << endl;
	out << "Data stored in SQLite3 Database: '" << db_interface_.filename() << "'" << endl;
	out << "With database schema:" << endl;
}

void
RRReporterSQLite::show( ) const {
	show( TR );
}


Real
RRReporterSQLite::recovery_rate() const {
	return Real( rotamers_recovered_ / residues_considered_ );
}


#endif // DB_SQLITE3

} // rotamer_recovery
} // protocols

