// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
/*                                                                            */
/*                           ----  SPARTA   ----                              */
/*     Shifts Prediction from Analogue of Residue type and Torsion Angle      */
/*                           Yang Shen and Ad Bax                             */
/*                    J. Biomol. NMR, xx, xxx-xxx (2010)                      */
/*                 NIH, NIDDK, Laboratory of Chemical Physics                 */
/*                     version, 1.00 (build 2010.0607.00)                     */
/*                                                                            */
/*                      for any problem, please contact                       */
/*                          shenyang@niddk.nih.gov                            */
/*                                                                            */
/******************************************************************************/
///  modified for use inside CS-Rosetta  by Oliver Lange
///
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.


///Problems found during porting:
// ANN is loaded with residues r1+1 .. rN-1
// but PRED_SUM is loaded from r1 .. rN --> first residue can be uninitialized (came only up in MPI runs for some reason... )
// -- HA3 -> means aN.size -> 9 need to have extra memory alloacated
// string functions were weird ... potential memory problems... replaced in util.cc where fishy...




/// @author Oliver Lange

// Unit Headers
#include <protocols/sparta/Sparta.hh>
#include <protocols/sparta/util.hh>

// Package Headers
#include <core/types.hh>
// Project Headers

// Utility headers
#include <core/util/Tracer.hh>

// Utility headers
#include <core/options/option_macros.hh>


//// C++ headers
#include <cstdlib>
#include <string>
#include <cmath>

#ifdef WIN32
#include <direct.h>
#else
#include <dirent.h>
#include <sys/stat.h>
#include <sys/timeb.h>
#endif

#define MAX_NUM          9999.000

#define PI              3.14159265358979323846
#define RAD             57.29577951
#define MAXNUM          9999


#include <core/options/keys/evaluation.OptionKeys.gen.hh>
#include <core/io/database/open.hh>

#include <stdio.h>


//  argList.push_back("spartaDir");
//   argList.push_back("shiftDir");
//   argList.push_back("pdbDir");
//   argList.push_back("tabDir");
//   argList.push_back("predDir");
//   argList.push_back("in");
//   argList.push_back("ins");
//   argList.push_back("sum");
//   argList.push_back("out"); //same as "-sum"
//   argList.push_back("homo");
//   argList.push_back("ref");
//   argList.push_back("db");
//   argList.push_back("w");
//   argList.push_back("fitting");
//   argList.push_back("pdbList");
//   argList.push_back("rand");
//   argList.push_back("adj");
//   argList.push_back("prev");
//   argList.push_back("next");
//   argList.push_back("atom");
//   argList.push_back("excl");
//   argList.push_back("first");
//   argList.push_back("last");
//   argList.push_back("count");
//   argList.push_back("help");
//   argList.push_back("time");
//   argList.push_back("notime");
//   argList.push_back("oldsparta");


bool protocols::sparta::Sparta::options_registered_( false );


void protocols::sparta::Sparta::register_options() {
  using namespace core::options;
  using namespace OptionKeys;
  if ( options_registered_ ) return;
  options_registered_ = true;
}


static core::util::Tracer tr("protocols.sparta");

namespace protocols {
namespace sparta {
using namespace core;
using namespace std;

Sparta::Sparta( std::string const& cs_file ) :
	REF_CS_Tab( cs_file ),
	bCreateOutput_( false )
{
	refCSFileName=cs_file;
  string libvar;
  if( getenv( "SPARTA_DIR" ) == NULL )	{
    SPARTA_DIR = ".";
	} else{
		SPARTA_DIR = getenv( "SPARTA_DIR" );
	}

	using namespace core::options;
  using namespace OptionKeys;

	slash_char = "/"; //default Unix
	SPARTA_DIR=core::io::database::full_name( "SPARTA+" );
	if ( option[ OptionKeys::evaluation::sparta_dir ].user() ) SPARTA_DIR=option[ OptionKeys::evaluation::sparta_dir ]();

	if ( SPARTA_DIR.find("/") != string::npos ) slash_char = "/"; // unix
	else if ( SPARTA_DIR.find("\\") != string::npos ) slash_char = "\\"; // Windows
	else SPARTA_DIR = ".";

	string temp;
	if( getenv( "PATH" ) != NULL ) {
		temp = getenv( "PATH" );
		if(temp.find("/") != string::npos ) slash_char = "/"; // unix
		else if(temp.find("\\") != string::npos ) slash_char = "\\"; // Windows
	};

	aN[1]="N"; aN[2]="HA"; aN[3]="C"; aN[4]="CA"; aN[5]="CB"; aN[6]="HN"; //aN[7]="H3";
  aN_ALL[1]="N"; aN_ALL[2]="HA"; aN_ALL[3]="C"; aN_ALL[4]="CA"; aN_ALL[5]="CB"; aN_ALL[6]="HN"; //aN_ALL[7]="H3";

	//if ( option[ OptionKeys::sparta::dir ].user() ) SPARTA_DIR = option[ OptionKeys::sparta::dir ]();

	init();

}


Real Sparta::score_pose( core::pose::Pose const& pose ) {
	inName = "INTERNAL";
	inPDB.loadPDB( pose );

  residList = inPDB.residListOne;

  r1 = inPDB.r1;
  rN = inPDB.rN;

  if (firstRes < r1) firstRes = r1;
  if (lastRes  < r1) lastRes = r1;

  if (firstRes > rN) firstRes = rN;
  if (lastRes  > rN) lastRes = rN;

  if (firstRes > lastRes) {
      int itemp = firstRes;
      firstRes  = lastRes;
      lastRes   = itemp;
	}
	tr.Info << "run ANN Sparta for pose with " << rN-r1+1 << " residues " << std::endl;

	return run_A_ANN_Prediction();
}

void Sparta::printSyntax()
{
	/*
  cerr << "File Name Settings: " << endl;
  cerr << " -in        [None]            Input PDB coordinate file." << endl;
  cerr << " -sum       [pred.tab]        Prediction summary output." << endl;
  //	cerr << " -db        [sparta.tab]      SPARTA triplet database file." << endl;
  cerr << " -w         [weight.tab]      SPARTA triplet weighting file." << endl;
  cerr << " -homo      [homology.tab]    Residue homology table." << endl;
  //	cerr << " -fitting   [fitting.tab]     Fitting parameters to calculate predict error." << endl;
  cerr << " -ref       [None]            Refernce(Observed) chemical shift table." << endl;
  //	cerr << " -pdbList   [None]            Candidate protein name table (For compiling database only)" << endl;
  cerr << "Random Coil and Secondary Shift Tables:" << endl;
  cerr << " -rand      [randcoil.tab]    Random coil shift table." << endl;
  cerr << " -adj       [rcadj.tab]       Adjustment table." << endl;
  //	cerr << " -prev      [rcprev.tab]      Previous residue adjustment." << endl;
  //	cerr << " -next      [rcnext.tab]      Next residue adjustment." << endl << endl;
  cerr << "Other Options:" << endl;
  cerr << " -atom      [N HA C CA CB HN] List of shift types to predict. (Default: All)" << endl;
  //	cerr << " -excl      [None]            List of proteins to exclude." << endl;
  cerr << " -first     [First]           First RESID to use." << endl;
  cerr << " -last      [Last]            Last RESID to use." << endl;
  //	cerr << " -count     [20]              Max match count per triplet." << endl;
  cerr << "Examples:" << endl;
  cerr << " SPARTA -in ubq.pdb          Form shifts predictions from a given PDB coordinate file." << endl;

  exit(0);
	*/
}


//preset the args form command line SHIFT_DIR
void Sparta::setup_defaults()
{

  TAB_DIR = SPARTA_DIR + slash_char+ "tab";
  SHIFT_DIR = SPARTA_DIR + slash_char+ "shifts";
  PDB_DIR = SPARTA_DIR + slash_char+ "pdb";

	//later use Evaluator to determine scratch dir as in ExternalEvaluator...
	PRED_DIR = "pred";
	inName = "INTERNAL";
 //  if( args["in"].length() > 0 ) inName = args["in"];
//   if( args["ins"].length() > 0 ) inNames = args["ins"];

  tripFileName = TAB_DIR + slash_char+ "sparta.tab";
  weightFileName = TAB_DIR + slash_char + "weight.tab";
  homoFileName = TAB_DIR + slash_char + "homology.tab";
  fitFileName = TAB_DIR + slash_char + "fitting.tab";
  sumName = PRED_DIR + slash_char +"pred.tab";

 //  if( args["ref"].length() > 0 ) refCSFileName = args["ref"];

  rcFileName = TAB_DIR + slash_char + "randcoil.tab";
  adjFileName = TAB_DIR + slash_char + "rcadj.tab";
  prevFileName = TAB_DIR + slash_char + "rcprev.tab";
  nextFileName = TAB_DIR + slash_char + "rcnext.tab";

  //Other Options
  EXCLUDED="";

  // if(args["atom"].length() > 0 ) {
//     aN.clear();
//     vector<string> temp = GDB::split(" ", args["atom"]);
//     int cnt = 1;
//     for(int i = 0; i < temp.size(); i++)
//       {
// 				if( temp[i]!="N" && temp[i]!="HA"&& temp[i]!="C"&& temp[i]!="CA"&& temp[i]!="CB"&& temp[i]!="HN")
// 					{
// 						cerr << "\tInvalid atom -" << temp[i] << endl;
// 						exit(0);
// 					}
// 				aN[cnt++] = temp[i];
//       }
//   }

  matchCount = 20;
  tVal = 500.0; // not used
  firstRes = -9999;
	lastRes = 9999;

}



void Sparta::init()
{
	setup_defaults();
  tr.Info << "Reading Random Coil Shifts from " << rcFileName << endl;
  RC_Tab.loadGDB( rcFileName );

  tr.Info << "Reading RC Adjustments from " << adjFileName << endl;
  ADJ_Tab.loadGDB( adjFileName );

  //load BLOSUM62 table
  AAlist = "A C D E F G H I K L M N P Q R S T V W Y";
  GDB B62;
  string B62_fname = TAB_DIR + slash_char+ "BLOSUM62.tab";
  tr.Info << "Reading BLOSUM62 Table from " << B62_fname << endl;
  B62.loadGDB( B62_fname );
  map<string, string >::iterator itS;
  for ( it = B62.Entries.begin(); it != B62.Entries.end(); it++ ) {
		//int index=it->first;
		string aa = (it->second)["RESNAME"];

		for ( itS = (it->second).begin(); itS != (it->second).end(); itS++ ) {
			if(AAlist.find(itS->first) != string::npos) {
				BLOSUM_62[aa].push_back( atof( (itS->second).c_str() )/10.0 );
			}
		}

	} // end of assigning sequence homology vector (using blosum62 matrix)

	tr.Info << "Load ANN parameters ..." << endl;
  for(itN = aN.begin(); itN != aN.end(); itN++) {
		string atomName = itN->second;
		if( atomName == "H" ) atomName="HN";

		SPARTA_ANN[atomName].init(113,30,1,9,6,3,TAB_DIR,atomName);
	}
  init_PredErrorSurface();

}





//compare the OBS and PRED chemical shifts if the "-ref" option is active
//and calculate RMS deviation between OBS and PRED chemical shifts for each shifts type
Real Sparta::compareRef(GDB & Pred_Sum)
{
	Real SCORE_SUM( 0.0 );

	//  int len = sumName.length();
  int pos = sumName.find_last_of(".");
  if ( bCreateOutput_ ) Pred_Sum.saveGDB( sumName.substr(0, pos) + "_full.tab" ); //save the original prediction summary file to a new name


  GDB COMP_Tab;

  COMP_Tab.REMARKS = Pred_Sum.REMARKS;
  COMP_Tab.DATA = Pred_Sum.DATA;

	// to compute chemshift score take CS_DIFF^2 / SIGMA ^2

  COMP_Tab.VARS_str_parser("  RESID RESNAME ATOMNAME CS_OBS SHIFT RC_SHIFT CS_DIFF SIGMA W");
  COMP_Tab.FORMAT_str_parser("  %4d %4s %4s %9.3f %9.3f %9.3f %9.3f %9.3f %.2f");
  COMP_Tab.addRemark( "Observed chemical shift from: " + refCSFileName );

  map<int, string>::iterator itN;
  map< int, map<string, string> >::iterator it;

  vector<float> OBS_V, PRED_V, DIFF_V, OBS_V_CORRECTED;
  float obs_shift, pred_shift, obs_shift2( 0.0 );

  for( itN = aN.begin(); itN != aN.end(); itN++ ) {
		string aName = itN->second;
		if( aName == "H" ) aName="HN";
		bool floating_sign( REF_CS_Tab.isVarFloat("SHIFT2") );
		if ( floating_sign ) tr.Info << " use floating sign1 " << std::endl;
		else tr.Info << " no floating sign " << std::endl;
		for ( it = REF_CS_Tab.Entries.begin(); it != REF_CS_Tab.Entries.end(); it++ )	{
			string aName_ref = it->second["ATOMNAME"];
			if( aName_ref == "H" ) aName_ref = "HN";

			if( aName == "HA" && aName_ref == "G")	{
				if( aName_ref.find("HA") != 0 ) continue;
			}	else if( aName_ref != aName ) continue;

			GDB_Entry temp = Pred_Sum.getEntry("RESID",it->second["RESID"],"ATOMNAME",aName,1);

			if(temp["SHIFT"].length() <= 0) continue;

			obs_shift = atof( (it->second["SHIFT"]).c_str() );
			if ( floating_sign ) obs_shift2 = atof( (it->second["SHIFT2"]).c_str() );

			pred_shift = atof( (temp["SHIFT"]).c_str() );

			if( aName == "HA" && it->second["RESNAME"] == "G") { //for HA of Gly				{
				if( aName_ref == "HA2" ) { // assign HA2 to the one with smaller shift
					float shift_HA3 = 9999.000;
					float shift_HA3_2 = 9999;
					GDB_Entry HA3 = REF_CS_Tab.getEntry("RESID",it->second["RESID"],"ATOMNAME","HA3",1);
					if( HA3["ATOMNAME"] == "HA3") shift_HA3 = atof( (HA3["SHIFT"]).c_str() );
					if( floating_sign && HA3["ATOMNAME"] == "HA3") shift_HA3_2 = atof( (HA3["SHIFT2"]).c_str() );

					if( shift_HA3 < obs_shift) obs_shift = shift_HA3;
					if ( floating_sign && shift_HA3_2 < obs_shift2 ) obs_shift2 = shift_HA3_2;

					shift_HA3 = 9999.000;
					HA3 = Pred_Sum.getEntry("RESID",it->second["RESID"],"ATOMNAME","HA3",1);
					if( HA3["ATOMNAME"] == "HA3") shift_HA3 = atof( (HA3["SHIFT"]).c_str() );

					if( shift_HA3 < pred_shift) pred_shift = shift_HA3;
				}	else if( aName_ref == "HA3" ) { // assign HA3 to the one with larger shift

					float shift_HA2 = -9999.000;
					float shift_HA2_2 = -9999;
					GDB_Entry HA2 = REF_CS_Tab.getEntry("RESID",it->second["RESID"],"ATOMNAME","HA2",1);
					if( HA2["ATOMNAME"] == "HA2") shift_HA2 = atof( (HA2["SHIFT"]).c_str() );
					if( floating_sign && HA2["ATOMNAME"] == "HA2") shift_HA2_2 = atof( (HA2["SHIFT2"]).c_str() );

					if( shift_HA2 > obs_shift) obs_shift = shift_HA2;
					if ( floating_sign && shift_HA2_2 > obs_shift2 ) obs_shift2 = shift_HA2_2;

					shift_HA2 = -9999.000;
					HA2 = Pred_Sum.getEntry("RESID",it->second["RESID"],"ATOMNAME","HA2",1);
					if( HA2["ATOMNAME"] == "HA2") shift_HA2 = atof( (HA2["SHIFT"]).c_str() );

					if( shift_HA2 > pred_shift) pred_shift = shift_HA2;
				}
			}


			if( pred_shift > 999.0 ) continue;


			if ( std::fabs( pred_shift-obs_shift ) > std::fabs( pred_shift-obs_shift2 ) ) obs_shift=obs_shift2;
			OBS_V.push_back( obs_shift );
			PRED_V.push_back( pred_shift );
			DIFF_V.push_back( pred_shift-obs_shift );

// 			COMP_Tab.setEntry(it->first,"RESID",it->second["RESID"]);
// 			COMP_Tab.setEntry(it->first,"RESNAME",it->second["RESNAME"]);
// 			COMP_Tab.setEntry(it->first,"ATOMNAME",it->second["ATOMNAME"]);
// 			COMP_Tab.setEntry(it->first,"CS_OBS",ftoa(obs_shift, buf));
// 			COMP_Tab.setEntry(it->first,"SHIFT",ftoa(pred_shift, buf));
// 			COMP_Tab.setEntry(it->first,"RC_SHIFT",temp["RC_SHIFT"]);
// 			// rc+adj+rxprev+rcnext, (+ HM?)

// 			COMP_Tab.setEntry(it->first,"CS_DIFF",ftoa(obs_shift-pred_shift, buf) );
// 			COMP_Tab.setEntry(it->first,"SIGMA",temp["SIGMA"] );

			Real sigma( atof( (temp["SIGMA"]).c_str() ) );
			if ( sigma > 0.1 ) {
				SCORE_SUM += (obs_shift-pred_shift)*(obs_shift-pred_shift)/(sigma*sigma);
			}

			//COMP_Tab.setEntry(it->first,"W","1.00" );
		}

		//		string rms_str = ftoa( getRMS(OBS_V, PRED_V), buf);
		//		string size_str = itoa( OBS_V.size(), buf);

		//		string str = "RMS(OBS, PRED) for " + aName + " (n=" + size_str + "): " + rms_str +" ppm";
		tr.Info << "RMS(OBS, PRED) for " << aName << " (n=" << OBS_V.size() << "): " <<  getRMS(OBS_V, PRED_V) <<  " ppm " << std::endl;
// 		COMP_Tab.setData( "NUM_"+aName, size_str);
// 		COMP_Tab.setData( "RMS_"+aName, rms_str);

		float avg_diff = getAVG(DIFF_V);
		//		float error_diff = getSTD(DIFF_V);
		//		int size = DIFF_V.size();

// 		if( avg_diff >=  3*error_diff/sqrt((float)size) ) { // check if the referencing correction is required
// 			tr.Info << "Std.dev. of prediction error for " << aName << ": " << error_diff << " ppm" << endl;
// 			tr.Info << "Average  of prediction error for " << aName << ": " << avg_diff
// 							<< " (> 3*" << error_diff << "/sqrt(n) )" << endl;
// 			for ( it = REF_CS_Tab.Entries.begin(); it != REF_CS_Tab.Entries.end(); it++ )	{
// 				if( aName == "HA" && it->second["RESNAME"] == "G") {
// 					if( it->second["ATOMNAME"].find("HA") != 0 ) continue;
// 				}	else if( it->second["ATOMNAME"] != aName ) continue;

// 				//				obs_shift = atof( (it->second["SHIFT"]).c_str() );
// 				//				REF_CS_Tab.Entries[ it->first ]["SHIFT"] = ftoa( obs_shift+avg_diff, buf); //referencing correction
// 			}
// 		}

// 		if( DIFF_V.size() > 0 )
// 			COMP_Tab.setData( "AVG_DIFF_"+aName, ftoa(avg_diff, buf) );
// 		else
// 			COMP_Tab.setData( "AVG_DIFF_"+aName, "0" );
		//	tr.Info << str << endl;

		for( Size i = 0; i< OBS_V.size(); i++) {
			OBS_V_CORRECTED.push_back( OBS_V[i] + avg_diff);
		}
		//COMP_Tab.setData( "RMS_"+aName+"_CORRECTED", ftoa( getRMS(OBS_V_CORRECTED, PRED_V), buf) );
		// the predicted shifts data after correction using <Pred-Obs>

		OBS_V.clear(); PRED_V.clear(); DIFF_V.clear(); OBS_V_CORRECTED.clear();
	}

	if ( tr.Debug.visible() ) {
		tr.Debug << " ============== COMP_Tab ==================== " << std::endl;
		COMP_Tab.showGDB( tr.Debug );
		tr.Debug << " ============== END COMP_Tab ==================== " << std::endl;
		tr.Debug << " ============== REF_CS_TAB ==================== " << std::endl;
		REF_CS_Tab.showGDB( tr.Debug );
		tr.Debug << " ============== END_CS_TAB ==================== " << std::endl;
	}
	if ( bCreateOutput_ ) {
		COMP_Tab.saveGDB( sumName );
		REF_CS_Tab.saveGDB( PRED_DIR + slash_char +"ref.tab" );
	}
	return SCORE_SUM/4;
}



//Get the list of angles\ring shifts\h-bond information from coordinates for all possbile residues
//**************** NOT ABLE TO HANDLE PROTEIN WITH MULTIPLE CHAINS ****************
void Sparta::getResInfo()
{
  inTab.Entries.clear();

  // allocation
  int n = rN-r1+1, m = 10;
  U_ANGLES = new float* [n];
  U_ANGLES[0] = new float [n*m];
  for(int i = 1; i < n; ++i)
    U_ANGLES[i] = U_ANGLES[i-1] + m;

  U_RING_SHIFTS = new float* [n];
  U_RING_SHIFTS[0] = new float [n*(aN.size()+1)];
  for(int i = 1; i < n; ++i)
    U_RING_SHIFTS[i] = U_RING_SHIFTS[i-1] + aN.size()+1;

  n = rN-r1+1; m = 4;
  U_NAME = new string* [n];
  U_NAME[0] = new string [n*m];
  for(int i = 1; i < n; ++i)
    U_NAME[i] = U_NAME[i-1] + m;

  U_HN_HB = new float [n];
  U_HA_HB = new float [n];
  U_CO_HB = new float [n];

  int pos0 = inName.find_last_of(slash_char)+1;
  int pos1 = inName.find_last_of(".");

  sourceName=inName.substr(pos0,pos1-pos0);

  map<int, string>::iterator itN;
  int cnt = 0;
  // format the sequence read from PDB coordinates
  sequence="";
  for(itN = residList.begin(); itN != residList.end(); itN++){
    sequence += itN->second;
    cnt++;
    if( cnt%10 == 0 ) sequence += " "; //separator for each 10 residues

    itN++;
    if(itN != residList.end()) {//add "?" if sequence numbers are not consecutive

			int j = itN->first;
			itN--;
			for(int i = 1; i< j - itN->first; i++) {

				sequence += "?"; cnt++;
				if( cnt%10 == 0 ) sequence += " ";
			}
		}
    else itN--;
  }

//   clock_t start, finish;
//   start = clock();

  inPDB.initOrbitalShift();
  //finish = clock();
  //tr.Info << "\n\t initOrbitalShift running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;
  inPDB.initHBond();
  //finish = clock();
  //tr.Info << "\n\t initHBond running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;
  inPDB.collect_HN_S2_and_EF();
  //inPDB.calc_HN_S2();
  //finish = clock();
  //tr.Info << "\n\t calc_HN_S2 running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;

  inTab.setData("SEQUENCE", sequence);
  inTab.VARS_str_parser("  RESID_R1 RESNAME_R1 PHI_R1 PSI_R1 CHI1_R1 RESID_R2 RESNAME_R2 PHI_R2 PSI_R2 CHI1_R2 RESID_R3 RESNAME_R3 PHI_R3 PSI_R3 CHI1_R3 N_HM HA_HM C_HM CA_HM CB_HM H_HM H_HB HA_HB CO_HB SOURCE");
  inTab.FORMAT_str_parser("%4d %s %8.3f %8.3f %8.3f %4d %s %8.3f %8.3f %8.3f %4d %s %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %s");

  //calculate H-bond for the first residue
  float dist = inPDB.HBDistList[r1]["HN"];
  U_HN_HB[0] = dist;
  dist = inPDB.HBDistList[r1]["HA"];
  U_HA_HB[0] = dist;
  dist = inPDB.HBDistList[r1]["O"];
  U_CO_HB[0] = dist;


  for ( int i = r1; i <= rN; i++ ) {
		CHI2_ANGLES[i] = inPDB.getChi2(1,i); // chi2 angle for residue r1, index by i-r1 (confusing..., but consistent with the loop)
		//OMEGA_ANGLES[i] = inPDB.getOmega(1,i); // chi2 angle for residue r1, index by i-r1 (confusing..., but consistent with the loop)
	}

  float shift;
  //loop for the polypeptide chain
  for ( int i = r1+1; i < rN; i++ ) {
		int index = i-r1;

		if( residList.find(i-1) == residList.end() ||
			residList.find(i) == residList.end() ||
			residList.find(i+1) == residList.end()) continue;


		shift = inPDB.getPhi(1,i+1);
		U_ANGLES[index][7] = shift;
		shift = inPDB.getPsi(1,i+1);
		U_ANGLES[index][8] = shift;
		shift = inPDB.getChi1(1,i+1);
		U_ANGLES[index][9] = shift;

		U_NAME[index][3] = residList[i+1];

		inTab.Entries[index]["PHI_R3"] = ftoa(U_ANGLES[index][7], buf);
		inTab.Entries[index]["PSI_R3"] = ftoa(U_ANGLES[index][8], buf);
		inTab.Entries[index]["CHI1_R3"] = ftoa(U_ANGLES[index][9], buf);
		inTab.Entries[index]["RESID_R3"] = itoa(i+1, buf);
		inTab.Entries[index]["RESNAME_R3"] = residList[i+1];
		inTab.Entries[index]["SOURCE"] = inName.substr(pos0,pos1-pos0);

		//Ring current shifts
		for(itN = aN.begin(); itN != aN.end(); itN++)	{
			string name = itN->second;
			if( name == "H" ) {
				name = "HN";
				if( residList[i] == "P" ) continue;
	    }	else if( name == "HA" && residList[i] == "G" ) {
	      U_RING_SHIFTS[index][7-1] = inPDB.getOrbitalShift(1,i,"HA3"); // change to use standard HA2/3 names
	      name = "HA2";
	    }	else if( name == "CB" && residList[i] == "G" ) continue;

			U_RING_SHIFTS[index][itN->first-1] = inPDB.getOrbitalShift(1,i,name) ;
			inTab.Entries[index][itN->second+"_HM"] = ftoa(U_RING_SHIFTS[index][itN->first-1], buf);
		}

		//H-Honds
		dist = inPDB.HBDistList[i]["HN"];
		inTab.Entries[index]["H_HB"] = ftoa(dist, buf);
		U_HN_HB[index] = dist;

		dist = inPDB.HBDistList[i]["HA"];
		inTab.Entries[index]["HA_HB"] = ftoa(dist, buf);
		U_HA_HB[index] = dist;

		dist = inPDB.HBDistList[i]["O"];
		inTab.Entries[index]["CO_HB"] = ftoa(dist, buf);
		U_CO_HB[index] = dist;

		if( inTab.Entries.find(index-1) != inTab.Entries.end() ) {
			//if tripet i-1 exist
			//assign the values of positions 1 and 2 of triplet i using the postions 2 and 3 of triplet i-1
			U_ANGLES[index][1] = U_ANGLES[index-1][4];
			U_ANGLES[index][2] = U_ANGLES[index-1][5];
			U_ANGLES[index][3] = U_ANGLES[index-1][6];
			U_ANGLES[index][4] = U_ANGLES[index-1][7];
			U_ANGLES[index][5] = U_ANGLES[index-1][8];
			U_ANGLES[index][6] = U_ANGLES[index-1][9];
			U_NAME[index][1] = U_NAME[index-1][2];
			U_NAME[index][2] = U_NAME[index-1][3];

			inTab.Entries[index]["PHI_R1"] = inTab.Entries[index-1]["PHI_R2"];
			inTab.Entries[index]["PSI_R1"] = inTab.Entries[index-1]["PSI_R2"];
			inTab.Entries[index]["CHI1_R1"] = inTab.Entries[index-1]["CHI1_R2"];
			inTab.Entries[index]["RESID_R1"] = inTab.Entries[index-1]["RESID_R2"];
			inTab.Entries[index]["RESNAME_R1"] = inTab.Entries[index-1]["RESNAME_R2"];

			inTab.Entries[index]["PHI_R2"] = inTab.Entries[index-1]["PHI_R3"];
			inTab.Entries[index]["PSI_R2"] = inTab.Entries[index-1]["PSI_R3"];
			inTab.Entries[index]["CHI1_R2"] = inTab.Entries[index-1]["CHI1_R3"];
			inTab.Entries[index]["RESID_R2"] = inTab.Entries[index-1]["RESID_R3"];
			inTab.Entries[index]["RESNAME_R2"] = inTab.Entries[index-1]["RESNAME_R3"];
		}
		else {	//else, calculate the values from coordinates

			shift = inPDB.getPhi(1,i-1);
			U_ANGLES[index][1] = shift;
			shift = inPDB.getPsi(1,i-1);
			U_ANGLES[index][2] = shift;
			shift = inPDB.getChi1(1,i-1);
			U_ANGLES[index][3] = shift;
			U_NAME[index][1] = residList[i-1];

			inTab.setEntry(index, "PHI_R1", ftoa(U_ANGLES[index][1], buf) );
			inTab.setEntry(index, "PSI_R1", ftoa(U_ANGLES[index][2], buf) );
			inTab.setEntry(index, "CHI1_R1", ftoa(U_ANGLES[index][3], buf) );
			inTab.setEntry(index, "RESID_R1", itoa(i-1, buf) );
			inTab.setEntry(index, "RESNAME_R1", residList[i-1] );

			shift = inPDB.getPhi(1,i);
			U_ANGLES[index][4] = shift;
			shift = inPDB.getPsi(1,i);
			U_ANGLES[index][5] = shift;
			shift = inPDB.getChi1(1,i);
			U_ANGLES[index][6] = shift;
			U_NAME[index][2] = residList[i];

			inTab.Entries[index]["PHI_R2"] = ftoa(U_ANGLES[index][4], buf);
			inTab.Entries[index]["PSI_R2"] = ftoa(U_ANGLES[index][5], buf);
			inTab.Entries[index]["CHI1_R2"] = ftoa(U_ANGLES[index][6], buf);
			inTab.Entries[index]["RESID_R2"] = itoa(i, buf);
			inTab.Entries[index]["RESNAME_R2"] = residList[i];

			if ( tr.Trace.visible() ) {
				tr.Trace << std::endl;
			}
		}

		//ANN input preparation
		// (20 BLOSSUM + 2 PHI + 2 PSI + 2 CHI1 + 2 CHI2)*3 + (4 ASA)*3 -chi2_c_asa
		// (20 BLOSSUM + 2 PHI + 2 PSI + 2 CHI1 + 2 CHI2 + 2 Oemga)*3 + (4 H-bond)*5 [O(i-1),HN,HA,O,HN(i+1)]
		vector<float> temp;
		//add ANN input for residue i-1
		string resName=residList[i-1]; if(resName=="c") resName="C";
		temp.insert(temp.end(), BLOSUM_62[resName].begin(), BLOSUM_62[resName].end());
		float phi = U_ANGLES[index][1], psi = U_ANGLES[index][2], chi1 = U_ANGLES[index][3], chi2 = CHI2_ANGLES[i-1];//, omega=OMEGA_ANGLES[i-1];
		if( phi<999 ) { temp.push_back(sin(phi*PI/180.0)); temp.push_back(cos(phi*PI/180.0));}//phi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( psi<999 ) { temp.push_back(sin(psi*PI/180.0)); temp.push_back(cos(psi*PI/180.0));}//psi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( chi1<999 ) { temp.push_back(sin(chi1*PI/180.0)); temp.push_back(cos(chi1*PI/180.0));}//chi1
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi1<999);
		if( chi2<999 ) { temp.push_back(sin(chi2*PI/180.0)); temp.push_back(cos(chi2*PI/180.0));}//chi2
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi2<999);

		//add ANN input for residue i
		resName=residList[i]; if(resName=="c") resName="C";
		temp.insert(temp.end(), BLOSUM_62[resName].begin(), BLOSUM_62[resName].end());
		phi = U_ANGLES[index][4]; psi = U_ANGLES[index][5]; chi1 = U_ANGLES[index][6]; chi2 = CHI2_ANGLES[i];//, omega=OMEGA_ANGLES[i];
		if( phi<999 ) { temp.push_back(sin(phi*PI/180.0)); temp.push_back(cos(phi*PI/180.0));}//phi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( psi<999 ) { temp.push_back(sin(psi*PI/180.0)); temp.push_back(cos(psi*PI/180.0));}//psi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( chi1<999 ) { temp.push_back(sin(chi1*PI/180.0)); temp.push_back(cos(chi1*PI/180.0));}//chi1
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi1<999);
		if( chi2<999 ) { temp.push_back(sin(chi2*PI/180.0)); temp.push_back(cos(chi2*PI/180.0));}//chi2
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi2<999);

		//add ANN input for residue i+1
		resName=residList[i+1]; if(resName=="c") resName="C";
		temp.insert(temp.end(), BLOSUM_62[resName].begin(), BLOSUM_62[resName].end());
		phi = U_ANGLES[index][7]; psi = U_ANGLES[index][8]; chi1 = U_ANGLES[index][9]; chi2 = CHI2_ANGLES[i+1];//, omega=OMEGA_ANGLES[i+1];
		if( phi<999 ) { temp.push_back(sin(phi*PI/180.0)); temp.push_back(cos(phi*PI/180.0));}//phi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( psi<999 ) { temp.push_back(sin(psi*PI/180.0)); temp.push_back(cos(psi*PI/180.0));}//psi
		else { temp.push_back(sin(PI)); temp.push_back(cos(PI));}
		if( chi1<999 ) { temp.push_back(sin(chi1*PI/180.0)); temp.push_back(cos(chi1*PI/180.0));}//chi1
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi1<999);
		if( chi2<999 ) { temp.push_back(sin(chi2*PI/180.0)); temp.push_back(cos(chi2*PI/180.0));}//chi2
		else { temp.push_back(0); temp.push_back(0);}
		temp.push_back(chi2<999);


		float hb = inPDB.HBDistList[i-1]["O"];
		if(hb>0) {
			temp.push_back(1.0); temp.push_back(hb); temp.push_back( cos(inPDB.HB_DHO_AngleList[i-1]["O"]*PI/180.0) ); temp.push_back( cos(inPDB.HB_HOA_AngleList[i-1]["O"]*PI/180.0) );
		}
		else {temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0);}
		hb = inPDB.HBDistList[i]["HN"];
		if(hb>0) {
			temp.push_back(1.0); temp.push_back(hb); temp.push_back( cos(inPDB.HB_DHO_AngleList[i]["HN"]*PI/180.0) ); temp.push_back( cos(inPDB.HB_HOA_AngleList[i]["HN"]*PI/180.0) );
		}
		else {temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0);}
		hb = inPDB.HBDistList[i]["HA"];
		if(hb>0) {
			temp.push_back(1.0); temp.push_back(hb); temp.push_back( cos(inPDB.HB_DHO_AngleList[i]["HA"]*PI/180.0) ); temp.push_back( cos(inPDB.HB_HOA_AngleList[i]["HA"]*PI/180.0) );
		}
		else {temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0);}
		hb = inPDB.HBDistList[i]["O"];
		if(hb>0) {
			temp.push_back(1.0); temp.push_back(hb); temp.push_back( cos(inPDB.HB_DHO_AngleList[i]["O"]*PI/180.0) ); temp.push_back( cos(inPDB.HB_HOA_AngleList[i]["O"]*PI/180.0) );
		}
		else {temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0);}
		hb = inPDB.HBDistList[i+1]["HN"];
		if(hb>0) {
			temp.push_back(1.0); temp.push_back(hb); temp.push_back( cos(inPDB.HB_DHO_AngleList[i+1]["HN"]*PI/180.0) ); temp.push_back( cos(inPDB.HB_HOA_AngleList[i+1]["HN"]*PI/180.0) );
		}	else {temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0); temp.push_back(0.0);}

		temp.push_back(inPDB.HN_S2[i-1]);
		temp.push_back(inPDB.HN_S2[i]);
		temp.push_back(inPDB.HN_S2[i+1]);

		if( temp.size() == 113) ANN_IN_MTX[i]=temp;
	}


  //calculate H-bond for the last residue
  dist = inPDB.HBDistList[rN]["HN"];
  U_HN_HB[rN-r1] = dist;
  dist = inPDB.HBDistList[rN]["HA"];
  U_HA_HB[rN-r1] = dist;
  dist = inPDB.HBDistList[rN]["O"];
  U_CO_HB[rN-r1] = dist;

	if ( bCreateOutput_ ) inTab.saveGDB(PRED_DIR+slash_char+inName.substr(pos0,pos1-pos0) + "_in.tab");

  if ( tr.Trace.visible() ) {
		map<int, vector<float> >::iterator itX;
		for(itX = ANN_IN_MTX.begin(); itX != ANN_IN_MTX.end(); itX++) {
			for(int i=0; i< (int)(itX->second).size();i++)
				tr.Trace << (itX->first) << " " << (itX->second)[i] << std::endl;
		}
	}
}



// run ANN prediction for a single protein chain
void Sparta::runANN_Prediction()
{
  clock_t start/*, finish*/;
  start = clock();

	//  init(); now in constructor
	if ( bCreateOutput_ ) {
  // mkdir for prediction
		if (PRED_DIR.find_last_of(slash_char) == PRED_DIR.length()-1 )
			PRED_DIR = PRED_DIR.substr(0,PRED_DIR.length()-1);
		mkdir_pred(PRED_DIR);
	}
  //for( itN = aN.begin(); itN != aN.end(); itN++ )
  //	mkdir_pred(PRED_DIR+slash_char+itN->second);

  tr.Info << "Reading PDB Coordinates from " << inName << endl;
  inPDB.loadPDB(inName);

  residList = inPDB.residListOne;

  r1 = inPDB.r1;
  rN = inPDB.rN;

  if (firstRes < r1) firstRes = r1;
  if (lastRes  < r1) lastRes = r1;

  if (firstRes > rN) firstRes = rN;
  if (lastRes  > rN) lastRes = rN;

  if (firstRes > lastRes)
    {
      int itemp = firstRes;
      firstRes  = lastRes;
      lastRes   = itemp;
    }

  run_A_ANN_Prediction();


}



// run ANN prediction for a single protein chain
Real Sparta::run_A_ANN_Prediction()
{
  clock_t start/*, finish*/;
  //start = clock();

  tr.Info << "Analyzing " << inName << " " ;
  getResInfo();
  tr.Info << residList.size() << " residues read " << endl;


  //finish = clock();
  //tr.Info << "\t getResInfo() running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;

  start = clock();
  tr.Info << "ANN prediction ..." << endl;
  for(itN = aN.begin(); itN != aN.end(); itN++) {
		string atomName = itN->second;
		if( atomName == "H" ) atomName="HN";

		SPARTA_ANN[atomName].ANN_OUT_MTX_LEVEL1.clear();
		SPARTA_ANN[atomName].runSpartaANN(ANN_IN_MTX);

		ANN_CS_OUTPUT_FULL[atomName] = SPARTA_ANN[atomName].ANN_OUT_MTX_LEVEL1;
	}

  //finish = clock();
  //tr.Info << "\t ANNPredict() running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;

  GDB PRED_SUM;
  PRED_SUM.VARS_str_parser("  RESID RESNAME ATOMNAME SS_SHIFT SHIFT RC_SHIFT HM_SHIFT EF_SHIFT SIGMA SOURCE");
  PRED_SUM.FORMAT_str_parser(" %4d %4s %4s %9.3f %9.3f %9.3f %9.3f %9.3f %9.3f %s");
  string str = itoa(r1, buf);
  PRED_SUM.setData("FIRST_RESID", str+"\n");
  PRED_SUM.setData("SEQUENCE", sequence);

  float RC, RCadj, pred_2nd_shift, pred_shift/*, HB*/;
  for ( int i = r1+1; i <= rN-1; i++ ) { //olange: we have not loaded the ANN with stuff for residue 1 or rN as it would be the 0,1,2 triplett.. ignore here TOO!
		for(itN = aN.begin(); itN != aN.end(); itN++)	{
			string atomName = itN->second;
			if( atomName == "H" ) atomName="HN";
			int index = PRED_SUM.Entries.size()+1;

			if( residList[i].empty() ) continue;
			if( residList[i] == "P" && (atomName == "HN" || atomName == "N") ) continue;
			if( residList[i] == "G" && atomName == "CB" ) continue;
			if( i==r1 && (atomName == "HN"|| atomName == "N") ) continue; //added from email Yang Shen/ Aug 6th.
			if( i==rN && atomName == "C" ) continue; //added from email Yang Shen/ Aug 6th

			PRED_SUM.setEntry(index, "RESID",  itoa(i,buf));
			PRED_SUM.setEntry(index, "RESNAME",  residList[i]);
			if (atomName=="HA" && residList[i] == "G") PRED_SUM.setEntry(index, "ATOMNAME",  "HA2"); //added from email YangShen Aug 6th.
			else PRED_SUM.setEntry(index, "ATOMNAME",  atomName);


			RC = getRC(residList[i],atomName);
			RCadj = getRCadj(residList[i],atomName);
			if (i==r1 || i==rN) pred_2nd_shift = 0.0; //may not good for the last residue, for which the neighoring residue effect is not considered.
			else {
				if(atomName == "HA") pred_2nd_shift = ANN_CS_OUTPUT_FULL[atomName][i][0]/4.0;
				else if(atomName == "HN") pred_2nd_shift = ANN_CS_OUTPUT_FULL[atomName][i][0]/2.0;
				else if(atomName == "N") pred_2nd_shift = ANN_CS_OUTPUT_FULL[atomName][i][0]*2.5;
				else pred_2nd_shift = ANN_CS_OUTPUT_FULL[atomName][i][0];
				if( pred_2nd_shift > 20.0 || pred_2nd_shift < -20.0 ) pred_shift = 0.0;
			}

			pred_shift = pred_2nd_shift + RC + RCadj; // + PrevRCadj + NextRCadj;
			if( pred_shift > 999.0) pred_shift = MAX_NUM;


			PRED_SUM.setEntry(index, "SS_SHIFT",  ftoa(pred_2nd_shift,buf));

			pred_shift += 0.6*atof(inTab.Entries[i-r1][atomName+"_HM"].c_str());
			if(atomName == "HN" || atomName == "HA" ) pred_shift-= inPDB.ElectricField[i][atomName]; //marked off to exclude shifts from "global" contacts and to test MFR
			PRED_SUM.setEntry(index, "SHIFT",  ftoa(pred_shift,buf));

			PRED_SUM.setEntry(index, "RC_SHIFT", ftoa(RC+RCadj ,buf) );
			PRED_SUM.setEntry(index, "SOURCE", sourceName );
			PRED_SUM.setEntry(index, "SIGMA", ftoa(getANN_PredError(U_ANGLES[i-r1][4],U_ANGLES[i-r1][5],residList[i],atomName),buf) );
			//tr.Info << U_ANGLES[i-r1][4] << "\t" << U_ANGLES[i-r1][5] << "\t" << getANN_PredError(U_ANGLES[i-r1][4],U_ANGLES[i-r1][5],residList[i],atomName) << endl;
			PRED_SUM.setEntry(index, "HM_SHIFT", inTab.Entries[i-r1][atomName+"_HM"] );
			PRED_SUM.setEntry(index, "EF_SHIFT", ftoa(inPDB.ElectricField[i][atomName],buf) );

			if(atomName=="HA" && residList[i] == "G") { // for GLY HA3
				index++;
				PRED_SUM.setEntry(index, "RESID",  itoa(i,buf));
				PRED_SUM.setEntry(index, "RESNAME",  residList[i]);
				PRED_SUM.setEntry(index, "ATOMNAME",  "HA3");
				PRED_SUM.setEntry(index, "SS_SHIFT",  ftoa(pred_2nd_shift,buf));
				pred_shift = pred_2nd_shift + RC + RCadj + 0.6*U_RING_SHIFTS[i-r1][0]; // atof(inTab.Entries[i-r1][atomName+"_HM"].c_str());;
				pred_shift-= inPDB.ElectricField[i]["HA"];
				PRED_SUM.setEntry(index, "SHIFT",  ftoa(pred_shift,buf));
				PRED_SUM.setEntry(index, "RC_SHIFT", ftoa(RC+RCadj ,buf) );
				PRED_SUM.setEntry(index, "SOURCE", sourceName );
				PRED_SUM.setEntry(index, "SIGMA", ftoa(getANN_PredError(U_ANGLES[i-r1][4],U_ANGLES[i-r1][5],residList[i],atomName),buf) );
				PRED_SUM.setEntry(index, "HM_SHIFT", ftoa(U_RING_SHIFTS[i-r1][0],buf) );
				PRED_SUM.setEntry(index, "EF_SHIFT", ftoa(inPDB.ElectricField[i]["HA"],buf) );
			}

		}
	}

	if ( tr.Debug.visible() ) {
		tr.Debug << " ============== PRED_SUM ==================== " << std::endl;
		PRED_SUM.showGDB( tr.Debug );
		tr.Debug << " ============== END_ PRED_SUM ==================== " << std::endl;
	}
  //finish = clock();
  //tr.Info << "\t ANNPredict() running time: " << (float)(finish - start)/ CLOCKS_PER_SEC << " seconds" << endl;
	if ( bCreateOutput_ ) {
    PRED_SUM.saveGDB(sumName);
	}

	Real score( compareRef( PRED_SUM ) );

  // deallocation
  delete [] U_ANGLES[0];
  delete [] U_ANGLES;
  delete [] U_NAME[0];
  delete [] U_NAME;
  delete [] U_RING_SHIFTS[0];
  delete [] U_RING_SHIFTS;
  delete [] U_HN_HB;
  delete [] U_HA_HB;
  delete [] U_CO_HB;

	return score;

}


// run ANN prediction for multiple protein chains
void Sparta::runANN_Predictions() {
  //init(); //now in constructor
  // mkdir for prediction
	if ( bCreateOutput_ ) {

		if (PRED_DIR.find_last_of(slash_char) == PRED_DIR.length()-1 )
			PRED_DIR = PRED_DIR.substr(0,PRED_DIR.length()-1);
		mkdir_pred(PRED_DIR);
		//	for( itN = aN.begin(); itN != aN.end(); itN++ )
		//		mkdir_pred(PRED_DIR+slash_char+itN->second);
	}

  vector<string> temp = split(" ", inNames);
  string  outName = sumName;

  tr.Info << inNames << endl;

  for( Size i = 0; i < temp.size(); i++)
    {
      inName = temp[i];
      //tr.Info << "Reading PDB Coordinates from " << inName << endl;
      inPDB.loadPDB(inName);

      residList = inPDB.residListOne;

      r1 = inPDB.r1;
      rN = inPDB.rN;

      if (firstRes < r1) firstRes = r1;
      if (lastRes  < r1) lastRes = r1;

      if (firstRes > rN) firstRes = rN;
      if (lastRes  > rN) lastRes = rN;

      if (firstRes > lastRes) {
				int itemp = firstRes;
				firstRes  = lastRes;
				lastRes   = itemp;
			}

      ANN_IN_MTX.clear();
      ANN_CS_OUTPUT_FULL.clear();

      int pos0 = inName.find_last_of(slash_char)+1;
      int pos1 = inName.find_last_of(".");

      sourceName=inName.substr(pos0,pos1-pos0);
      sumName = PRED_DIR + slash_char + sourceName + "_pred.tab";

      run_A_ANN_Prediction();
      tr.Info << "\tPrediction file " << sumName << " is ready for protein " << inName << endl;

    }

}



// Initiate an ANN prediction for a single protein using its file name
void Sparta::runANN_Prediction(const string& pName)
{
  //init(); now in constructor
	if ( bCreateOutput_ ) {

		// mkdir for prediction
		if (PRED_DIR.find_last_of(slash_char) == PRED_DIR.length()-1 )
			PRED_DIR = PRED_DIR.substr(0,PRED_DIR.length()-1);
		for( itN = aN.begin(); itN != aN.end(); itN++ )
			mkdir_pred(PRED_DIR+slash_char+itN->second);
	}
  inName = pName;
  inPDB.loadPDB(inName);

  run_A_ANN_Prediction();

}



void Sparta::init_PredErrorSurface()
{
  int step = 5;

  for(itN = aN.begin(); itN != aN.end(); itN++)
    {
      string atomName = itN->second;
      if( atomName == "H" ) atomName="HN";

      for( Size i=0; i<AAlist.length();i++)
	{
	  string AA = AAlist.substr(i,1);
	  if( AA == " " ) continue;

	  if( AA == "G" && atomName == "CB" ) continue;
	  if( AA == "P" && atomName == "HN" ) continue;

	  string surfName = TAB_DIR + slash_char + "errorSurface" + slash_char + atomName + slash_char + AA + "..A450.S5.RMS.tab";
	  GDB surf(surfName);

	  for(it = surf.Entries.begin(); it != surf.Entries.end(); it++)
	    {
	      int phi = atoi( it->second["PHI"].c_str() );
	      for(int y=-180; y<180; y+=step)
		{
		  string psi=itoa(y,buf);
		  SPARTA_ERR_SURF[AA][atomName][phi][y] = atof( it->second[psi].c_str() );
		}
	    }

	}
    }
}



float Sparta::getANN_PredError(float phi, float psi, string aa, string aName)
{

  return SPARTA_ERR_SURF[aa][aName][5*int(phi/5)][5*int(psi/5)];
}



// get random coil chemical shift for atom 'aName' of residue 'resName'
float Sparta::getRC(const string& resName, const string& aName)
{
  GDB_Entry temp = RC_Tab.getEntry("RESNAME",resName,1);

  if(temp.size() != 0) return atof( temp[aName].c_str() );

  return 9999.0;
}



float Sparta::getRCadj(const string& resName, const string& aName)
{
  GDB_Entry temp = ADJ_Tab.getEntry("RESNAME",resName,1);

  if(temp.size() != 0) return atof( temp[aName].c_str() );

  return 0.0;
}



float Sparta::getPrevRCadj(const string& prev_rName, const string& aName)
{
  GDB_Entry temp = PREV_Tab.getEntry("RESNAME",prev_rName,1);

  if(temp.size() != 0) return atof( temp[aName].c_str() );

  return 0.0;
}



float Sparta::getNextRCadj(const string& next_rName, const string& aName)
{
  GDB_Entry temp = NEXT_Tab.getEntry("RESNAME",next_rName,1);

  if(temp.size() != 0) return atof( temp[aName].c_str() );

  return 0.0;
}



float Sparta::getWeight(const string& Name, const string& aName)
{
  GDB_Entry temp = WEIGHT_Tab.getEntry("RESNAME",Name,1);

  if(temp.size() != 0) return atof( temp[aName].c_str() );

  return 9999.0;

}



void Sparta::mkdir_pred(const string& d)// create a directory for prediction results
{
  if( ! isDirExists( d ) )
    {
      int i = MKDIR(d.c_str())+1;
      if( !i ) { // if not success
	string parentD = d.substr(0,d.find_last_of(slash_char)+1);
	mkdir_pred(parentD.c_str());
	if ( !(MKDIR(d.c_str())+1) ) {
	  cerr << "\tCan't create prediction directory " << d.c_str() << endl;
	  exit(0);
	}
      }
    }
}



float Sparta::getDiff( float ang1, float ang2)
{
  float a = fabs(ang1-ang2);

  if( a < 180 ) return a;
  else if( a > 180 && a < 360.1 && ang1*ang2 < 0)
    {
      return 360.0-a;
    }

  return 0;
}



char * Sparta::itoa( int n, char *buff, int /*base*/ )
{
  sprintf(buff, "%d", n);
  return buff;
}



char * Sparta::ftoa( float n, char *buff, char f, int prec )
{
  if ( !(f=='f' || f=='F' || f=='e' || f=='E' || f=='g' || f=='G') ) {
    f = 'f';
  }
  char format[20];
  char *fs = format;				// generate format string
  *fs++ = '%';					//   "%.<prec>l<f>"
  if ( prec >= 0 ) {
    if ( prec > 99 )			// buf big enough for precision?
      prec = 99;
    *fs++ = '.';
    if ( prec >= 10 ) {
      *fs++ = prec / 10 + '0';
      *fs++ = prec % 10 + '0';
    } else {
      *fs++ = prec + '0';
    }
  }
  *fs++ = 'l';
  *fs++ = f;
  *fs = '\0';
  sprintf( buff, format, n );

  return buff;
}



int Sparta::MKDIR(const char *dirName)
{
#ifdef WIN32
  return mkdir(dirName);
#else
  return mkdir(dirName, 0777);
#endif
}



bool Sparta::isDirExists(const string &Dir) {
#ifdef WIN32
  char oldDir[100];
  if( NULL == getcwd(oldDir, _MAX_PATH) ) return false;

  if( chdir( Dir.c_str() ) == -1 ) return false;

  if( chdir( oldDir ) == -1 ) return false;

  return true;
#else
  DIR *dp;
  dp = opendir(Dir.c_str());
  if (dp != NULL)
    {
      (void) closedir (dp);
      return true;
    }
  else
    return false;
#endif
}



float Sparta::getAVG(vector<float> &v1)
{
  int n = v1.size();

  float sum = 0.0;

  for(int i = 0; i < n; i++)
    sum += v1[i];

  return sum/(float)n;
}



float Sparta::getSTD(vector<float> &v1)
{
  int n = v1.size();
  float avg = getAVG(v1);

  float sum = 0.0;

  for(int i = 0; i < n; i++)
    sum += (v1[i]-avg)*(v1[i]-avg);

  return sqrt(sum/(float)(n-1));
}


float Sparta::getRMS(vector<float> &v1, vector<float> &v2)
{
  if(v1.size() != v2.size() ) return -1.0;

  int n = v1.size();

  float sum = 0.0;
  for(int i = 0; i < n; i++)
    {
      sum += ( (v1[i]-v2[i])*(v1[i]-v2[i]) );
    }

  return sqrt(sum/(float)(n-1));
}

}
}
