// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   utility/string_util.cc
///
/// @brief  Some std::string helper functions.
/// @author Sergey Lyskov

#include <utility/string_util.hh>
#include <utility/io/izstream.hh>
#include <core/types.hh>
#include <iostream>
#include <sstream>
#include <utility/excn/Exceptions.hh>
#include <utility/vector1.hh>
#include <locale>
namespace utility {
///TODO get rid of std::vector
///TODO implement this function in terms of string_split, found below
/// @details split given std::string using ' ' symbol.
std::vector< std::string > split(const std::string &s)
{
	std::vector<std::string> r;
	unsigned int start=0, i=0;
	while( start < s.size() ) {
		if( s[i] == ' ' /*|| i==s.size()-1 */) {
			std::string add(s.begin()+start, s.begin()+i);
			if( add.size() != 0 ) r.push_back( add );
			start = i+1;
		}
		i++;
		if( i == s.size() ) {
			std::string add(s.begin()+start, s.begin()+i);
			if( add.size() != 0 ) r.push_back( add );
			break;
		}
	}

	// Causes a assertion failure on VC, the empty check is now performed above.
	//for(std::list< std::string >::iterator it=r.end(); it!=r.begin(); ) {  /// removing empty lines
	//	it--;
	//	if( it->size() == 0 ) r.erase( it );
	//}
	return r;
}

std::string join(utility::vector1<std::string> const & s, std::string const & connector){
	std::ostringstream os;
	utility::vector1<std::string>::const_iterator begin= s.begin();
	os << *begin++;
	for(; begin != s.end(); ++begin){
		os<< connector<< *begin;
	}
	return os.str();
}

std::string join(std::vector<std::string> const & s, std::string const & connector){
	std::ostringstream os;
	utility::vector1<std::string>::const_iterator begin= s.begin();
	os << *begin++;
	for(; begin != s.end(); ++begin){
		os<< connector<< *begin;
	}
	return os.str();
}

std::string join(std::string const & string_w_spaces, std::string const & connector){
	std::string trimmed= trim(string_w_spaces);
	std::vector<std::string> pieces= split(string_w_spaces);
	return join(pieces, connector);
}

/// @details split given std::string using ' ' symbol.
std::list< std::string > split_to_list(const std::string &s) {
	std::list<std::string> r;
	unsigned int start=0, i=0;
	while( start < s.size() ) {
		if( s[i] == ' ' /*|| i==s.size()-1 */) {
			std::string add(s.begin()+start, s.begin()+i);
			if( add.size() != 0 ) r.push_back( add );
			start = i+1;
		}
		i++;
		if( i == s.size() ) {
			std::string add(s.begin()+start, s.begin()+i);
			if( add.size() != 0 ) r.push_back( add );
			break;
		}
	}

	// Causes a assertion failure on VC, the empty check is now performed above.
	//for(std::list< std::string >::iterator it=r.end(); it!=r.begin(); ) {  /// removing empty lines
	//	it--;
	//	if( it->size() == 0 ) r.erase( it );
	//}
	return r;
}

/// @details split to vector< std::string > using arbitrary split character
/// @author ashworth
std::vector< std::string >
string_split( std::string const & in, char splitchar /* = ' ' */ )
{
	std::vector< std::string > parts;
	size_t i(0), j(0);
	while ( j != std::string::npos ) {
		j = in.find( splitchar, i );
		parts.push_back( in.substr(i,j-i) );
		i = j+1;
	}
	return parts;
}

/// @details convert a string to a float
float string2float( std::string st ){
	float i;
	std::stringstream ss( st );
	ss >> i;
	if(!ss){
		return -1;
	}
	return i;
}

/// @details convert a string to an int
int string2int( std::string st ){
	int i;
	std::stringstream ss( st );
	ss >> i;
	if(!ss){
		return -1;
	}
	return i;
}

/// @details compares two strings ignoring leading and trailing spaces
bool trimmed_compare( std::string const & s1, std::string const & s2 )
{
	std::string const space( " " );

	std::size_t const s1_start( s1.find_first_not_of( space ) );
	std::size_t const s1_end( s1.find_last_not_of( space ) );
	std::size_t const s2_start( s2.find_first_not_of( space ) );
	std::size_t const s2_end( s2.find_last_not_of( space ) );

	std::size_t const s1_len( s1_end - s1_start + 1 );
	std::size_t const s2_len( s2_end - s2_start + 1 );

//	std::cout << "s1: " << s1 << " s1_start " << s1_start << " s1_end " << s1_end << " s1_len " << s1_len << std::endl;
//	std::cout << "s2: " << s2 << " s2_start " << s2_start << " s2_end " << s2_end << " s2_len " << s2_len << std::endl;

	return ( ( s1_len == s2_len ) && ( s1.compare( s1_start, s1_len, s2, s2_start, s2_len ) == 0 ) );
}

bool startswith(std::string const & haystack, std::string const & needle)
{
	if( haystack.length() < needle.length() ) return false;
	else return ( haystack.compare(0, needle.length(), needle) == 0 );
}

bool endswith(std::string const & haystack, std::string const & needle)
{
	if ( haystack.length() < needle.length() ) return false;
	else return ( haystack.compare(haystack.size()-needle.size(),needle.size(),needle) == 0 );
}

void slurp(std::istream & in, std::string & out)
{
	std::string line;
	std::ostringstream os;
	while (std::getline(in,line)) {
		os << line << std::endl;
	}
	out.append( os.str());
}

void trim( std::string & s, const std::string & drop)
{
	std::string r = s.erase( s.find_last_not_of(drop)+1 );
	r.erase( 0, r.find_first_not_of(drop) );
	s = r;
}

std::string
trim( std::string const & s, std::string const & drop )
{
	std::string copystr( s );
	trim( copystr, drop );
	return copystr;
}

void add_spaces_left_align( std::string & st, std::size_t const newlen )
{

	std::size_t const to_add = newlen - st.length();
	if( to_add > 0 ){
		std::string st_to_add("");
		st_to_add.append(to_add,' ');
		st = st + st_to_add;
	}

}

void add_spaces_right_align( std::string & st, std::size_t const newlen )
{

	std::size_t const to_add = newlen - st.length();
	if( to_add > 0 ){
		std::string st_to_add("");
		st_to_add.append(to_add,' ');
		st = st_to_add + st;
	}
}

bool is_string_numeric(std::string const & input)
{
	std::locale loc;
	for(core::Size i = 0 ; i < input.size();++i)
	{
		char current = input[i];
		if(std::isdigit(current,loc) || current == '-' || current == '+' || current =='E' ||current=='e')
		{
			continue;
		}else
		{
			return false;
		}
	}
	return true;
}


std::string
file_contents( std::string const & file_name )
{
	vector1< std::string > text;
	std::string line;
	io::izstream textstream( file_name );
	if ( ! textstream ) {
		throw excn::EXCN_Msg_Exception( "Could not open file " + file_name  );
	}
	int strsize( 0 );
	while ( getline(textstream, line) ) {
		text.push_back(line + "\n");
		strsize += line.size() + 1;
	}
	textstream.close();

	std::string alltext;
	alltext.reserve( strsize );
	for ( unsigned int ii = 1; ii <= text.size(); ++ ii) {
		alltext += text[ii];
	}
	return alltext;
}



std::string file_basename( std::string const & full_path ) {
	std::string const sep( "/" );

	std::size_t const idx = full_path.find_last_of( sep );
	std::string retval( full_path );
	if ( idx != std::string::npos ) {
		retval = full_path.substr( idx + 1 );
	}
	return retval;
}


} // namespace utility

