// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/io/silent/protein_silent.cxxtest.hh
/// @brief  test suite for protein silent-file format
/// @author James Thompson

// Test headers
#include <cxxtest/TestSuite.h>

#include <test/core/init_util.hh>
#include <core/util/Tracer.hh>
#include <test/UTracer.hh>

#include <core/conformation/ResidueFactory.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/BinaryProteinSilentStruct.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <numeric/random/random.hh>
#include <utility/file/file_sys_util.hh>

//Auto Headers
#include <core/chemical/VariantType.hh>
#include <core/conformation/Atom.fwd.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/DOF_ID.hh>
#include <core/id/NamedStubID.hh>
#include <core/io/pdb/file_data.hh>
#include <core/io/silent/EnergyNames.fwd.hh>
#include <core/kinematics/types.hh>
#include <core/options/option.hh>
#include <core/pose/signals/ConformationEvent.hh>
#include <core/pose/signals/DestructionEvent.hh>
#include <core/pose/signals/EnergyEvent.hh>
#include <core/scoring/types.hh>
#include <core/scoring/constraints/Constraints.fwd.hh>
#include <core/util/OStream.hh>
#include <utility/keys/Key2Tuple.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/model_quality/maxsub.hh>



static core::util::Tracer TR("test.core.io.silent.protein_silent");

using namespace core;

class Binary_Protein_Silent_File_Tests : public CxxTest::TestSuite
{
public:
	Binary_Protein_Silent_File_Tests() {};

	// Shared initialization goes here.
	void setUp() {
		core_init_with_additional_options( "-in::file::silent_struct_type binary -in::file::fullatom" );

		// Residue definitions can't be supplied on the command line b/c
		// the ResidueTypeSet is already initialized.
		using namespace core::chemical;
		utility::vector1< std::string > params_files;
		ResidueTypeSetCAP const_residue_set = ChemicalManager::get_instance()->residue_type_set( FA_STANDARD );
		ResidueTypeSet & residue_set = const_cast< ResidueTypeSet & >(*const_residue_set);
		if(!residue_set.has_name("GTP")) params_files.push_back("core/io/GTP.params");
		residue_set.read_files(params_files,
			ChemicalManager::get_instance()->atom_type_set( FA_STANDARD ),
			ChemicalManager::get_instance()->mm_atom_type_set( FA_STANDARD ) );
	}

	// Shared finalization goes here.
	void tearDown() {
	}


void test_save_and_restore()
{
	double rms_threshold = 1e-2;
	double score_threshold = 1e-1;

	pose::Pose ref_pose, restored_pose;
	core::chemical::ResidueTypeSetCAP rsd =
			core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD );
	core::io::pdb::pose_from_pdb( ref_pose, *rsd, std::string("core/io/bin_silentfile_test.pdb"));

	// Read the ProteinSilentStruct from the silent-file
	core::io::silent::SilentFileData sfd;
	sfd.read_file( "core/io/bin_silentfile_test.out" ); // read file w/ non-ideal geometry
	TS_ASSERT( sfd.size() > 0 );
	core::io::silent::SilentFileData::iterator iter = sfd.begin();
	iter->fill_pose( restored_pose, *rsd );

	// test rms difference
	Real rms_to_restored = scoring::CA_rmsd( ref_pose, restored_pose );
	TR << "RMS error from save/restore: " << rms_to_restored << std::endl;
	TS_ASSERT( rms_to_restored < rms_threshold );

	// test score13 difference
	core::scoring::ScoreFunctionOP scorefxn =
				core::scoring::ScoreFunctionFactory::create_score_function( "score13_env_hb" );
	Real score_ref = (*scorefxn)(ref_pose);
	Real score_restored = (*scorefxn)(restored_pose);
	Real score_del = std::fabs( score_restored - score_ref );
	TR << "Score difference: " << score_del << std::endl;
	TS_ASSERT( score_del < score_threshold );

}

};
