// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file apps/public/revert_design_to_native.cc
/// @brief For protein-interface design, reverts residues at the interface to their wildtype identities and if binding energy
/// isn't adversely affected applies the change. Produces a report at the end of the run of all of the changes.

// Project headers
#include <core/types.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/pose/Pose.hh>

#include <core/options/keys/OptionKeys.hh>
#include <protocols/init.hh>

#include <core/io/pdb/pose_io.hh>

// Unit headers
#include <protocols/ProteinInterfaceDesign/design_utils.hh>

// C++ headers
#include <iostream>

// option key includes

#include <core/options/option.hh>

using namespace core;
using namespace protocols::ProteinInterfaceDesign;

using namespace core::options;

namespace revert_app
{
	StringOptionKey wt( "revert_app:wt" );
	StringOptionKey design( "revert_app:design" );
	IntegerOptionKey ddg_cycles( "revert_app:ddg_cycles" );
	RealOptionKey threshold( "revert_app:threshold" );
}


///////////////////////////////////////////////////////////////////////////////
int
main( int argc, char * argv [] )
{
	using namespace core::pose;
	using namespace core::scoring;
	using namespace conformation;
	using namespace protocols::ProteinInterfaceDesign;
	using namespace core::options;
	using namespace core::options::OptionKeys;
	option.add( revert_app::wt, "The wt file name" );
	option.add( revert_app::design, "The design file name");
	option.add( revert_app::ddg_cycles, "how many ddg cycles to compute (more leads to higher numerical stability)" ).def( 5 );
	option.add( revert_app::threshold, "ddg threshold for acceptance of reversion" ).def( 0.5 );

	// setup random numbers and options
	protocols::init(argc, argv);

	// read the pose
	pose::Pose pose_wt, pose_des;
	std::string const wt_fname(  option[ revert_app::wt ] );
	std::string const des_fname(  option[ revert_app::design ] );
	core::io::pdb::pose_from_pdb( pose_wt, wt_fname );
	core::io::pdb::pose_from_pdb( pose_des, des_fname );


	ScoreFunctionOP scorefxn( core::scoring::getScoreFunction() ); // defaults to sc12
	core::Real const ddg_thres( option[ revert_app::threshold ] );
	Revert rev( scorefxn, ddg_thres, option[ revert_app::ddg_cycles ] );
	rev.apply( pose_wt, pose_des );

	pose_des.dump_scored_pdb( des_fname+".revert.pdb", *scorefxn );//, *scorefxn );

	exit(0);
}

