// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/CompositeFragmentSelector.cc
/// @brief provides a selector that picks fragments diferent enough from the fragments that have been already picked
/// @author Dominik Gront (dgront@chem.uw.edu.pl)

#include <core/fragment/picking/DiversifyDihedralsSelector.hh>

// package headers
#include <core/fragment/picking/FragmentCandidate.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.fwd.hh>

#include <core/util/Tracer.hh>
#include <utility/vector1.hh>
#include <cmath>

namespace core {
namespace fragment {
namespace picking {

static core::util::Tracer trDiversifyDihedralsSelector(
                "core.fragment.picking.DiversifyDihedralsSelector");


/// @brief  Selects desired number of fragments from a given set of candidates
void DiversifyDihedralsSelector::select_fragments(utility::vector1<std::pair<
			FragmentCandidateOP, scores::FragmentScoreMapOP> >& in,
			utility::vector1<std::pair<FragmentCandidateOP,
					scores::FragmentScoreMapOP> >& out) {

	if(in.size()==0) return;

	Size len = in[1].first->get_length();

	if (phi_.size() < len) {
	    phi_.resize(len);
	    psi_.resize(len);
	}

	out.push_back( in[1] );
	for(Size i=2;i<=in.size();i++) {
		if(out.size() >= frags_per_pos() ) break;
		bool is_ok = true;
		for(Size j=1;j<=out.size();j++) {
		    Real rms = dihedral_rmsd(in[i].first, out[j].first);
		    if(rms<cutoff_) {
			is_ok = false;
			trDiversifyDihedralsSelector.Trace<<"Phi-Psi rmsd is "<<rms<<" fragment "<< *in[i].first<<" denied"<<std::endl;;
			break;
		    }
		}
		if(is_ok) {
		    out.push_back( in[i] );
		    trDiversifyDihedralsSelector.Trace<<"Fragment "<< *in[i].first<<" passed"<<std::endl;;
		}
	}
	trDiversifyDihedralsSelector<<out.size()<<" fragments passed through DiversifyDihedralsSelector at query position "
	    << in[1].first->get_first_index_in_query()<<std::endl;
}

Real DiversifyDihedralsSelector::dihedral_rmsd(FragmentCandidateOP f1,FragmentCandidateOP f2) {

    Real rms = 0.0;
    assert ( f1->get_length() == f2->get_length() );
    for(Size k=1;k<=f1->get_length();k++) {
	Real d = f1->get_residue(k)->phi() - f2->get_residue(k)->phi();
	rms += d*d;
	d = f1->get_residue(k)->psi() - f2->get_residue(k)->psi();
	rms += d*d;
	d = f1->get_residue(k)->omega() - f2->get_residue(k)->omega();
	rms += d*d;
    }

    return sqrt(rms) / (3.0 * f1->get_length());
}

} // picking
} // fragment
} // core
