// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite && is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/scores/ScoreEValuator.cc
/// @brief  scores a fragment by an amino acid sequence identity
/// @author Dominik Gront (dgront@chem.uw.edu.pl)


// type headers
#include <core/types.hh>

#include <core/fragment/picking/scores/ScoreEValuator.hh>

// package headers
#include <core/fragment/picking/FragmentPicker.hh>
#include <core/fragment/picking/FragmentCandidate.hh>
#include <core/fragment/picking/VallChunk.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.hh>

// option key includes
// AUTO-REMOVED #include <core/init.hh>
#include <core/options/option.hh>
// AUTO-REMOVED #include <core/options/option_macros.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/frags.OptionKeys.gen.hh>

// mini headers
#include <core/sequence/SequenceProfile.hh>
#include <core/sequence/ScoringScheme.hh>
#include <core/sequence/ScoringSchemeFactory.hh>
#include <core/util/Tracer.hh>

#include <algorithm>

namespace core {
namespace fragment {
namespace picking {
namespace scores {

using namespace options;
using namespace options::OptionKeys;

static core::util::Tracer trProfScore(
		"core.fragment.picking.scores.ScoreEValuator");

void ScoreEValuator::do_caching(VallChunkOP chunk) {

	ProfileScore::do_caching(chunk);
}

void ScoreEValuator::clean_up() {

	ProfileScore::clean_up();
}

bool ScoreEValuator::score(FragmentCandidateOP f, FragmentScoreMapOP empty_map) {

	Real totalScore = 0;
	Real mean = 0;
	Real stdev = 0;

	utility::vector1<Size> columnsQ;
	utility::vector1<Size> columnsV;
	for (Size i = 1; i <= f->get_length(); i++) {
		assert(f->get_first_index_in_query() + i - 1 <= scores_.size());
		assert(f->get_first_index_in_vall()
				+ i - 1<= scores_[1].size());
		totalScore
				+= scores_[f->get_first_index_in_query() + i - 1][f->get_first_index_in_vall()
						+ i - 1];
		columnsQ.push_back(f->get_first_index_in_query() + i - 1);
		columnsV.push_back(f->get_first_index_in_vall() + i - 1);
	}

	for (Size i_rand = 1; i_rand <= max_rand_; ++i_rand) {
		std::random_shuffle(columnsQ.begin(), columnsQ.end());
		std::random_shuffle(columnsV.begin(), columnsV.end());
		Real s = 0;
		for (Size i = 1; i <= columnsQ.size(); i++) {
			s = scores_[columnsQ[i]][columnsV[i]];
			mean += s;
			stdev += s * s;
		}
	}
	mean /= ((Real) max_rand_);
	stdev /= ((Real) max_rand_);
	stdev = sqrt(stdev - mean * mean);

	totalScore = (totalScore - mean) / stdev;

	empty_map->set_score_component(totalScore, id_);
	if ((totalScore > lowest_acceptable_value_) && (use_lowest_ == true))
		return false;
	return true;
}

FragmentScoringMethodOP MakeScoreEValuator::make(Size priority,
		Real lowest_acceptable_value, bool use_lowest, FragmentPickerOP picker) {

	if (option[frags::scoring::profile_score].user()) {
		core::sequence::ScoringSchemeFactory ssf;
		core::sequence::ScoringSchemeOP ss(ssf.get_scoring_scheme(
				option[frags::scoring::profile_score]()));
		Size len = picker->get_vall()->get_largest_chunk_size();
		trProfScore << "Profile scoring method is: "
				<< option[frags::scoring::profile_score]() << std::endl;
		return (FragmentScoringMethodOP) new ScoreEValuator(priority,
				lowest_acceptable_value, use_lowest, picker->get_query_seq(), ss, len);
	}
	utility_exit_with_message(
			"[ERROR] Undefined profile scoring method. Provide it with frags::scoring_scheme flag");

	return NULL;
}

} //scores
} // picking
} // fragment
} // core
