// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/graph/find_neighbors.cc
/// @brief  Sets up the residue neighbor information
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)
///
/// @remarks Thanks to Will Sheffler for his ideas on refining this and extending it to atom neighbors
/// @remarks Adapting libRosetta code for generalized neighbor detection

#ifndef INCLUDED_core_graph_find_neighbors_HH
#define INCLUDED_core_graph_find_neighbors_HH

// Package Headers
#include <core/graph/PointGraph.fwd.hh>

// Project Headers
#include <platform/types.hh>
// AUTO-REMOVED #include <utility/vector1.hh>

//Auto Headers
#include <utility/vector1.fwd.hh>


namespace core {
namespace graph {

enum Strategy {
	NAIVE,
	AUTOMATIC,
	OCTREE,
	THREEDGRID
};


// Find neighbors and place them in a graph
void
find_neighbors(
	PointGraphOP point_graph,
	platform::Real neighbor_cutoff,
	Strategy strategy = AUTOMATIC
);

void
find_neighbors_naive(
	PointGraphOP point_graph,
	platform::Real neighbor_cutoff
);

/// @brief Finds the residue neighbors efficiently using an octree-like spatial sort
void
find_neighbors_octree(
	PointGraphOP point_graph,
	platform::Real neighbor_cutoff,
	Strategy strategy
);

/// @brief Create a 3D grid of points.  O(N^3).  For "spherical" conformations, Theta(N).  Speeds neighbor detection
/// in abinitio by a factor of 2.  Definition: Spherical = span of x,y and z all O(N**1/3).  Note circularity.
/// Adendum: if the 3D grid used a list of point indices instead of a vector, then this would be Theta(N) for
/// spherical conformations; however, with a vector, this is O(NlgN).  With the additional assumption that
/// each cube contains O(1) points, then this implementation is O(N).  Such an assumption is unneccessary
/// in the list implementation.
void
find_neighbors_3dgrid(
	PointGraphOP point_graph,
	platform::Real neighbor_cutoff
);

void
find_neighbors_restricted(
  PointGraphOP point_graph,
  platform::Real neighbor_cutoff,
  utility::vector1< bool > const & residue_selection,
  Strategy strategy = AUTOMATIC
);

void
find_neighbors_naive_restricted(
  PointGraphOP point_graph,
  platform::Real neighbor_cutoff,
  utility::vector1< bool > const & residue_selection
);

/// @brief Finds the residue neighbors efficiently using an octree-like spatial sort
void
find_neighbors_octree_restricted(
  PointGraphOP point_graph,
  platform::Real neighbor_cutoff,
  utility::vector1< bool > const & residue_selection,
  Strategy strategy
);

/// @brief Create a 3D grid of points.  O(N^3).  For "spherical" conformations, Theta(N).  Speeds neighbor detection
/// in abinitio by a factor of 2.  Definition: Spherical = span of x,y and z all O(N**1/3).  Note circularity.
/// Adendum: if the 3D grid used a list of point indices instead of a vector, then this would be Theta(N) for
/// spherical conformations; however, with a vector, this is O(NlgN).  With the additional assumption that
/// each cube contains O(1) points, then this implementation is O(N).  Such an assumption is unneccessary
/// in the list implementation.
void
find_neighbors_3dgrid_restricted(
	PointGraphOP point_graph,
	platform::Real neighbor_cutoff,
  utility::vector1< bool > const &  residue_selection
);

}
}

#endif
