// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/grid/Pockets/QuaternionFingerprint.hh
/// @brief  core::grid::Pockets::QuaternionFingerprint header
/// @author Ragul Gowthaman

#ifndef INCLUDED_core_grid_Pockets_QuaternionFingerprint_HH
#define INCLUDED_core_grid_Pockets_QuaternionFingerprint_HH

#include <utility/pointer/ReferenceCount.hh>
#include <core/grid/Pockets/QuaternionFingerprint.fwd.hh>
#include <core/grid/Pockets/PocketGrid.fwd.hh>
#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>

#include <numeric/constants.hh>
#include <numeric/xyzVector.hh>
#include <utility/vector1_bool.hh>
#include <list>
#include <cmath>


namespace core {
namespace grid {
namespace Pockets {

class Quaternion : public utility::pointer::ReferenceCount {

public:

  Quaternion() {};

  Quaternion( core::Real const & q1_in, core::Real const & q2_in, core::Real const & q3_in, core::Real const & q4_in ) :
		q1_( q1_in ),
		q2_( q2_in ),
		q3_( q3_in ),
		q4_( q4_in )
	{};

	inline void q1( core::Real const & q1_in) { q1_ = q1_in; };
	inline void q2( core::Real const & q2_in) { q2_ = q2_in; };
	inline void q3( core::Real const & q3_in) { q3_ = q3_in; };
	inline void q4( core::Real const & q4_in) { q4_ = q4_in; };

	inline core::Real q1() const { return q1_; };
	inline core::Real q2() const { return q2_; };
	inline core::Real q3() const { return q3_; };
	inline core::Real q4() const { return q4_; };

	Quaternion get_conjugate() const;


private:
	core::Real q1_;
	core::Real q2_;
	core::Real q3_;
	core::Real q4_;

};


typedef struct {
	Quaternion q;
	core::Real rho;
} spherical_coor_pentet;


class QuaternionFingerprintBase : public utility::pointer::ReferenceCount {

public:

  QuaternionFingerprintBase();

  void print_to_file(std::string const & output_filename) const;
  void print_to_file(std::string const & output_filename, Quaternion const & q_offset ) const;

  void print_to_pdb(std::string const & output_pdbname) const;
  void print_to_pdb(std::string const & output_pdbname, Quaternion const & q_offset ) const;

	// const accessor functions
	numeric::xyzVector<core::Real> origin() const { return origin_; };
	std::list< spherical_coor_pentet > const & pentet_fingerprint_data() const { return pentet_fingerprint_data_; };

protected:
	numeric::xyzVector<core::Real> origin_;
	std::list< spherical_coor_pentet > pentet_fingerprint_data_;

};

class NonPlaidQuaternionFingerprint : public QuaternionFingerprintBase {
public:
  NonPlaidQuaternionFingerprint() {};

  void setup_from_PocketGrid( PocketGrid const & pocket_grid );

  void setup_from_file(std::string const & input_filename);

  void setup_from_PlaidQuaternionFingerprint( PlaidQuaternionFingerprint const & pfp );

	void apply_rotation( Quaternion const & q_offset );

	void move_origin(numeric::xyzVector<core::Real> const & new_origin );

};

class PlaidQuaternionFingerprint : public QuaternionFingerprintBase {
public:
  PlaidQuaternionFingerprint() {};

  void build_from_pose(core::pose::Pose const & input_pose);

  void Find_Intersect(Quaternion const & qvalue, core::Real const & atomX, core::Real const & atomY, core::Real const & atomZ, core::Real const & atom_radius, int const & nest_k, int const & nest_l, int const & nest_m, int const & nest_n);

	core::Real find_optimal_rotation( QuaternionFingerprintBase const & fp, core::Real const & q_increment, Quaternion & optimal_q ) const;

	core::Real fp_compare( QuaternionFingerprintBase const & fp ) const;
	core::Real fp_compare( QuaternionFingerprintBase const & fp, Quaternion const & q_offset ) const;

	// RAGUL - FIGURE OUT WHAT TO DO WITH THIS
	core::Real Interpolate(core::Real const & phi_position1, core::Real const & phi_position2, core::Real const & psi_position1, core::Real const & psi_position2, core::Real const & pf_phi, core::Real const & pf_psi) const;


private:

  std::vector<core::Real> q_vector;
	std::vector<std::vector<std::vector<std::vector<core::Real> > > > RhoVector;
};

void apply_rotation_to_point( spherical_coor_pentet const & orig_pentet, Quaternion const &  q_offset, spherical_coor_pentet & new_pentet );

numeric::xyzVector<core::Real> apply_rotation_to_vector(Quaternion const & qrot, numeric::xyzVector<core::Real> const & vec);

numeric::xyzMatrix<core::Real> get_matrix(Quaternion const & quat);

numeric::xyzMatrix<core::Real> get_matrix( spherical_coor_pentet const & pentet);

numeric::xyzVector<core::Real> get_cartesian(Quaternion const & quat);

Quaternion multiply_Quaternions(Quaternion const & quat1, Quaternion const & quat2);

core::Real get_length(Quaternion const & quat);

Quaternion normalaize_Quaternion(Quaternion const & quat);


inline void convert_cartesian_to_spherical_coor_pentet( numeric::xyzVector<core::Real> const & coord, spherical_coor_pentet & pentet ){
	// RAGUL - REWRITE THIS
	pentet.rho = sqrt((coord.x()*coord.x())+(coord.y()*coord.y())+(coord.z()*coord.z()));
	pentet.q.q1(5);
	pentet.q.q2(5);
	pentet.q.q3(5);
	pentet.q.q4(5);
}

void convert_spherical_coor_pentet_to_cartesian( spherical_coor_pentet const & pentet, numeric::xyzVector<core::Real> & coord );

//phi = sph.x(); psi = sph.y(); rho = sph.z()
inline void convert_cartesian_to_spherical( numeric::xyzVector<core::Real> const & coord,numeric::xyzVector<core::Real> & sph){
	sph.z() = sqrt((coord.x()*coord.x())+(coord.y()*coord.y())+(coord.z()*coord.z()));
	sph.x() = acos(coord.z()/sph.z())*(1/numeric::constants::f::pi_over_180) ;
	sph.y() = atan2((coord.y()),(coord.x()))*(1/numeric::constants::f::pi_over_180);
	}

//phi = sph.x(); psi = sph.y(); rho = sph.z()
inline void convert_spherical_to_cartesian( numeric::xyzVector<core::Real> const & sph, numeric::xyzVector<core::Real> & coord ) {
	coord.x() = sph.z()*sin(sph.x()*numeric::constants::f::pi_over_180)*cos(sph.y()*numeric::constants::f::pi_over_180);
	coord.y() = sph.z()*sin(sph.x()*numeric::constants::f::pi_over_180)*sin(sph.y()*numeric::constants::f::pi_over_180);
	coord.z() = sph.z()*cos(sph.x()*numeric::constants::f::pi_over_180);
}

}//Pockets
}//grid
}//core

#endif
