// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/id/AtomID.cc
/// @brief  Kinematics Atom identifier
/// @author Phil Bradley


// Unit headers
#include <core/id/AtomID.hh>
#include <core/id/NamedAtomID.hh>
#include <core/id/NamedStubID.hh>
#include <core/id/Exceptions.hh>
#include <core/pose/Pose.hh>
#include <core/chemical/ResidueType.hh>

#include <utility/exit.hh>


#include <core/util/Tracer.hh>

// C++ headers
#include <ostream>


namespace core {
namespace id {

static core::util::Tracer tr("core.id.AtomID");

///@detail returns an AtomID corresponding to your NamedAtomID
/// check for a valid AtomID after this.
/// following conditions return invalid ID :
/// rsd > total_residue
/// atom not present in residue ( e.g., no CB in GLY )
AtomID::AtomID( NamedAtomID const & id_in, pose::Pose const & pose, bool raise_exception /*default true */ ) :
	atomno_( 0 ),
	rsd_( 0 )
{
	// work out the stubID
	if ( id_in.valid() ) {
		if ( id_in.rsd() <= pose.total_residue() ) { //if in range, ... otherwise leave atomno_ on 0 --> valid() == false
			chemical::ResidueType const& rt ( pose.residue_type ( id_in.rsd() ) );
			if ( rt.has( id_in.atom() ) ) {
				atomno_ = rt.atom_index( id_in.atom() );
			} else {
				// tr.Error << "Error: can't find atom " << id_in.atom() << " in residue "
// 					<< rt.name() << ", residue has " << rt.nheavyatoms() << " heavy atoms." << std::endl;
// 				tr.Error << "atom names are: " << std::endl;
				//rt.show_all_atom_names( tr.Error );
				if ( raise_exception ) throw EXCN_AtomNotFound( id_in );
				*this = id::BOGUS_ATOM_ID;
			}
		} else {
// 			tr.Error << "Error: can't find residue " << id_in.rsd()
// 				<< " in pose (pose.total_residue() = ) "
// 				<< pose.total_residue() << std::endl;
			if ( raise_exception ) throw EXCN_AtomNotFound( id_in );
 			*this = id::BOGUS_ATOM_ID;
		}
		rsd_ = id_in.rsd();
	}



}

///
AtomID const &
BondID::other_atom( AtomID const & id ) const
{
	if ( id == atom1 ) return atom2;
	else if ( id == atom2 ) return atom1;
	else utility_exit_with_message( "BondID::other_atom: unknown atom" );
	return atom1; // wont get here
}

/// @brief stream << AtomID
std::ostream &
operator <<( std::ostream& os, AtomID const & a )
{
	os << " atomno= " << a.atomno() << " rsd= " << a.rsd() << ' ';
	return os;
}

/// @brief Globals
AtomID const BOGUS_ATOM_ID( 0,0 );
AtomID const CHAINBREAK_BOGUS_ATOM_ID( 0,0 );
StubID const BOGUS_STUB_ID( BOGUS_ATOM_ID, BOGUS_ATOM_ID, BOGUS_ATOM_ID );

StubID::StubID( NamedStubID const& id_in, pose::Pose const& pose)
	: center_() {
	atom1 = AtomID( id_in.atom( 1 ), pose );
	atom2 = AtomID( id_in.atom( 2 ), pose );
	atom3 = AtomID( id_in.atom( 3 ), pose );
	if ( id_in.center().valid() ) center_ = AtomID( id_in.center(), pose );
	if ( !center_.valid() ) center_ = atom1;
}

/// @brief stream << AtomID
std::ostream &
operator <<( std::ostream& os, StubID const & stub )
{
	os << " Stub: a1 " << stub.atom1 << " a2 " << stub.atom2 << " a3 " << stub.atom3 << " cen " << stub.center_;
	return os;
}

} // namespace id
} // namespace core
