// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/id/AtomID_Map.Pose.hh
/// @brief  AtomID_Map--Pose interaction functions
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_core_id_AtomID_Map_Pose_HH
#define INCLUDED_core_id_AtomID_Map_Pose_HH


// Package headers
#include <core/id/AtomID_Map.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>

// Utility headers
#include <utility/pointer/owning_ptr.hh>


namespace core {
namespace id {


/// @brief Initialize an AtomID_Map for a given Pose using the AtomID_Map's current default fill values
template< typename T >
void
initialize( AtomID_Map< T > & atom_map, pose::Pose const & pose )
{
	Size const n_res( pose.n_residue() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, pose.residue_type(i).natoms() );
	}
}


/// @brief Initialize an AtomID_Map for a given Pose using a specified fill value
template< typename T >
void
initialize( AtomID_Map< T > & atom_map, pose::Pose const & pose, T const & value )
{
	Size const n_res( pose.n_residue() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, pose.residue_type(i).natoms(), value );
	}
}


/// @brief Initialize an AtomID_Map for a given Conformation using the AtomID_Map's current default fill values
template< typename T >
void
initialize( AtomID_Map< T > & atom_map, conformation::Conformation const & conformation )
{
	Size const n_res( conformation.size() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, conformation.residue_type(i).natoms() );
	}
}


/// @brief Initialize an AtomID_Map for a given Conformation using a specified fill value
template< typename T >
void
initialize( AtomID_Map< T > & atom_map, conformation::Conformation const & conformation, T const & value )
{
	Size const n_res( conformation.size() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, conformation.residue_type(i).natoms(), value );
	}
}

/// @brief Initialize an AtomID_Map for a given Pose using the AtomID_Map's current default fill values
template< typename T >
void
initialize_heavy_only( AtomID_Map< T > & atom_map, pose::Pose const & pose )
{
	Size const n_res( pose.n_residue() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, pose.residue_type(i).nheavyatoms() );
	}
}


/// @brief Initialize an AtomID_Map for a given Pose using a specified fill value
template< typename T >
void
initialize_heavy_only( AtomID_Map< T > & atom_map, pose::Pose const & pose, T const & value )
{
	Size const n_res( pose.n_residue() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, pose.residue_type(i).nheavyatoms(), value );
	}
}


/// @brief Initialize an AtomID_Map for a given Conformation using the AtomID_Map's current default fill values
template< typename T >
void
initialize_heavy_only( AtomID_Map< T > & atom_map, conformation::Conformation const & conformation )
{
	Size const n_res( conformation.size() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, conformation.residue_type(i).nheavyatoms() );
	}
}


/// @brief Initialize an AtomID_Map for a given Conformation using a specified fill value
template< typename T >
void
initialize_heavy_only( AtomID_Map< T > & atom_map, conformation::Conformation const & conformation, T const & value )
{
	Size const n_res( conformation.size() );
	atom_map.clear();
	atom_map.resize( n_res );
	for ( Size i = 1; i <= n_res; ++i ) {
		atom_map.resize( i, conformation.residue_type(i).nheavyatoms(), value );
	}
}


} // namespace id
} // namespace core


#endif // INCLUDED_core_id_AtomID_Map_Pose_HH
