// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

///@file core/scoring/Interface/DDPscore.cc
///@brief Implementation of distance dependent interface score
///@detailed The distance dependent score is a knowledge based potential specialized for
/// protein interfaces. This class loads and accesses the lookup tables.
///@author Hermann Zellner (hermann1.zellner@biologie.uni-regensburg.de)

#include <utility/vector1.hh>
#include <core/scoring/Interface/DDPscore.hh>
#include <core/scoring/Interface/DDPscoreCreator.hh>

#include <core/options/option.hh>
#include <core/options/keys/score.OptionKeys.gen.hh>
#include <core/pose/Pose.hh>

#include <core/util/Tracer.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/ContextGraphTypes.hh>
#include <numeric/constants.hh>

#include <core/conformation/Residue.hh>
#include <core/chemical/AtomType.hh>
#include <core/scoring/EnergyMap.hh>


//Auto Headers
#include <core/conformation/Residue.hh>
#include <core/kinematics/Jump.hh>
#include <core/scoring/EnergyMap.hh>


namespace core{
namespace scoring{
namespace Interface{




methods::EnergyMethodOP
DDPscoreCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new DDPscore;
}

ScoreTypes
DDPscoreCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( interface_dd_pair );
	return sts;
}


DDPscore::DDPscore() :
		parent( new DDPscoreCreator ),
		lookup_table_( core::scoring::ScoringManager::get_instance()->get_DDPLookupTable() )
{ }

methods::EnergyMethodOP DDPscore::clone() const
{
	return new DDPscore(*this);
}

void DDPscore::setup_for_scoring(pose::Pose&, const ScoreFunction& ) const
{
	//pose.update_residue_neighbors();
}

void DDPscore::setup_for_packing(pose::Pose&, const ScoreFunction& ) const
{
	//pose.update_residue_neighbors();
}

void DDPscore::setup_for_derivatives(pose::Pose&, const ScoreFunction& ) const
{
	//pose.update_residue_neighbors();
}

bool DDPscore::defines_intrares_energy(core::scoring::EnergyMap const &) const
{
        return false;
}

void DDPscore::eval_intrares_energy(
                        const core::conformation::Residue &,
                        const core::pose::Pose &,
                        const core::scoring::ScoreFunction &,
                        core::scoring::EnergyMap &
                        ) const {}


void DDPscore::residue_pair_energy(
	core::conformation::Residue const & rsd2,
	core::conformation::Residue const & rsd1,
	const core::pose::Pose &,
	const core::scoring::ScoreFunction &,
	core::scoring::EnergyMap & emap
) const
{
	assert (rsd1.seqpos() != rsd2.seqpos()); //only call for distinct residues

	if (rsd1.chain() != rsd2.chain()) // Only score contacts across the interface
	{
		core::Real distance = 1e3;
		for ( core::conformation::Atoms::const_iterator atom_it_1 = rsd1.atom_begin();
				atom_it_1 != rsd1.heavyAtoms_end(); atom_it_1++)
		{
			for ( core::conformation::Atoms::const_iterator atom_it_2 = rsd2.atom_begin();
					atom_it_2 != rsd2.heavyAtoms_end(); atom_it_2++)
			{
				if ( atom_it_1->xyz().distance(atom_it_2->xyz()) < distance )
				{
					distance = atom_it_1->xyz().distance(atom_it_2->xyz());
				}
			}
		}

		if (
				distance >= 10. ||
				distance < 1.5 ||
				lookup_table_.get_potentials( rsd1.aa(), rsd2.aa(), distance ) > 0. )
		{
		  emap[ interface_dd_pair ] += 0.; // noop
		}
		else
		{
			emap[ interface_dd_pair ] += lookup_table_.get_potentials( rsd1.aa(), rsd2.aa(), distance );
		}
	}
	else
	{
	  emap[ interface_dd_pair ] += 0.; // noop
	}

}



void DDPscore::indicate_required_context_graphs(utility::vector1< bool > & /*context_graphs_required*/ ) const
{

}

core::Distance DDPscore::atomic_interaction_cutoff() const{
        return 8.0;
}


} // InterfacePotentials
} // scoring
} // core
