// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/DunbrackEnergy.cc
/// @brief  Dunbrack energy method implementation
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)

// Unit headers
#include <core/scoring/methods/DunbrackEnergy.hh>
#include <core/scoring/methods/DunbrackEnergyCreator.hh>

// Package Headers
#include <core/scoring/EnergyMap.hh>
//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/dunbrack/RotamerLibrary.hh>
#include <core/scoring/dunbrack/RotamerLibraryScratchSpace.hh>

#include <core/scoring/ScoreType.hh>

// Project headers
#include <core/conformation/Residue.hh>
#include <core/pose/Pose.hh>

#include <core/id/TorsionID.hh>

// Utility headers
#include <numeric/conversions.hh>

namespace core {
namespace scoring {
namespace methods {


/// @details This must return a fresh instance of the DunbrackEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
DunbrackEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new DunbrackEnergy;
}

ScoreTypes
DunbrackEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( fa_dun );
	return sts;
}



/// ctor
DunbrackEnergy::DunbrackEnergy() :
	parent( new DunbrackEnergyCreator )
{}

DunbrackEnergy::~DunbrackEnergy() {}

/// clone
EnergyMethodOP
DunbrackEnergy::clone() const
{
	return new DunbrackEnergy;
}

/////////////////////////////////////////////////////////////////////////////
// methods for ContextIndependentOneBodyEnergies
/////////////////////////////////////////////////////////////////////////////

/// @details Allocate the scratch space object on the stack to
/// alieviate thread-safety concerns.  Scratch does not use new.
void
DunbrackEnergy::residue_energy(
	conformation::Residue const & rsd,
	pose::Pose const &,
	EnergyMap & emap
) const
{
	//static boost::detail::atomic_count count_present( 0 );
	//++count_present;
	//std::cout << "D" << count_present << std::flush;

	/* old		emap[ fa_dun ] = rot_lib_.rotamer_energy( rsd ); */
	scoring::dunbrack::SingleResidueRotamerLibraryCAP rotlib = rsd.type().get_RotamerLibrary();
	if ( rotlib ) {
		dunbrack::RotamerLibraryScratchSpace scratch;
		emap[ fa_dun ] += rotlib->rotamer_energy( rsd, scratch );
	}
	//--count_present;
	//std::cout << "D" << count_present << std::flush;

}

bool DunbrackEnergy::defines_dof_derivatives( pose::Pose const & ) const { return true; }

Real
DunbrackEnergy::eval_residue_dof_derivative(
	conformation::Residue const & rsd,
	ResSingleMinimizationData const & ,//min_data,
	id::DOF_ID const & ,// dof_id,
	id::TorsionID const & tor_id,
	pose::Pose const & ,//pose,
	ScoreFunction const & ,//sfxn,
	EnergyMap const & weights
) const
{

	Real deriv(0.0);
	if ( tor_id.valid() ) {
		assert( rsd.seqpos() == tor_id.rsd() );
		//utility::vector1< Real > dE_dbb, dE_dchi;
		//		std::cerr << __FILE__<< ' ' << __LINE__ << ' ' << tor_id.rsd() << std::endl;
		scoring::dunbrack::SingleResidueRotamerLibraryCAP rotlib =
			rsd.type().get_RotamerLibrary();
		if ( rsd.is_protein() && rotlib ) {
			dunbrack::RotamerLibraryScratchSpace scratch;
			rotlib->rotamer_energy_deriv( rsd, scratch );
			if ( tor_id.type() == id::BB  && tor_id.torsion() <= dunbrack::DUNBRACK_MAX_BBTOR ) {
				deriv = scratch.dE_dbb()[ tor_id.torsion() ];
			} else if ( tor_id.type() == id::CHI && tor_id.torsion() <= dunbrack::DUNBRACK_MAX_SCTOR ) {
				deriv = scratch.dE_dchi()[ tor_id.torsion() ];
			}
		}
	}
	return numeric::conversions::degrees( weights[ fa_dun ] * deriv );
}


///
Real
DunbrackEnergy::eval_dof_derivative(
	id::DOF_ID const &,// dof_id,
	id::TorsionID const & tor_id,
	pose::Pose const & pose,
	ScoreFunction const &,
	EnergyMap const & weights
) const
{
	//static boost::detail::atomic_count count_present( 0 );
	//++count_present;
	//std::cout << "dD" << count_present << std::flush;

	Real deriv(0.0);
	if ( tor_id.valid() ) {
		//utility::vector1< Real > dE_dbb, dE_dchi;
		//		std::cerr << __FILE__<< ' ' << __LINE__ << ' ' << tor_id.rsd() << std::endl;
		scoring::dunbrack::SingleResidueRotamerLibraryCAP rotlib =
			pose.residue( tor_id.rsd() ).type().get_RotamerLibrary();
		if ( rotlib ) {
			dunbrack::RotamerLibraryScratchSpace scratch;
			rotlib->rotamer_energy_deriv
				( pose.residue( tor_id.rsd() ), scratch );

			/// ASSUMPTION: Derivatives for amino acids only!
			if ( pose.residue_type( tor_id.rsd() ).is_protein() ) {
				if ( tor_id.type() == id::BB  && tor_id.torsion() <= dunbrack::DUNBRACK_MAX_BBTOR ) {
					deriv = scratch.dE_dbb()[ tor_id.torsion() ];
				} else if ( tor_id.type() == id::CHI && tor_id.torsion() <= dunbrack::DUNBRACK_MAX_SCTOR ) {
					deriv = scratch.dE_dchi()[ tor_id.torsion() ];
				}
			}
		}
	}
	//--count_present;
	//std::cout << "dD" << count_present << std::flush;
	return numeric::conversions::degrees( weights[ fa_dun ] * deriv );
}

/// @brief DunbrackEnergy is context independent; indicates that no context graphs are required
void
DunbrackEnergy::indicate_required_context_graphs(
	utility::vector1< bool > & /*context_graphs_required*/
) const
{}



} // methods
} // scoring
} // core

