// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.




#include <core/scoring/rna/RNA_Util.hh>

#include <utility/vector1.hh>
#include <numeric/interpolation/spline/Bicubic_spline.hh>
#include <numeric/MathVector_operations.hh>
#include <iostream>
#include <fstream>
#include <sstream>
#include <utility/io/izstream.hh>
#include <core/scoring/orbitals/OrbitalsLookup.hh>
#include <core/scoring/orbitals/OrbitalsScore.hh>
#include <core/scoring/orbitals/OrbitalsScoreCreator.hh>
#include <core/io/database/open.hh>



namespace core{
namespace scoring{
namespace orbitals{

//this function reads in the potentials from database. The potentials are already in the form of Energy.
OrbitalsLookup::OrbitalsLookup(const  utility::vector1<std::string> & filename){

	//construct the arrays that will have the data from files placed into them
	core::Real hpol_e1[1050];
	core::Real hpol_e2[1050];
	core::Real hpol_e3[1050];
	core::Real hpol_e4[1050];
	core::Real hpol_e5[1050];
	core::Real hpol_e6[1050];
	core::Real hpol_e7[1050];

	core::Real haro_e1[1050];
	core::Real haro_e2[1050];
	core::Real haro_e3[1050];
	core::Real haro_e4[1050];
	core::Real haro_e5[1050];
	core::Real haro_e6[1050];
	core::Real haro_e7[1050];

	core::Real orb_e1[1050];
	core::Real orb_e2[1050];
	core::Real orb_e3[1050];
	core::Real orb_e4[1050];
	core::Real orb_e5[1050];
	core::Real orb_e6[1050];
	core::Real orb_e7[1050];

	std::string line;

	utility::io::izstream stream;
	core::io::database::open( stream, filename[1] );


	//read in database file. This is the hpol file, put files into the array constructed above
	for( core::Size count=0; utility::io::getline(stream, line); ++count ){
		std::istringstream l( line );
		l >> hpol_e1[count] >> hpol_e2[count] >> hpol_e3[count] >> hpol_e4[count] >> hpol_e5[count] >> hpol_e6[count] >> hpol_e7[count];
	}
	stream.close();


	core::io::database::open( stream, filename[2] );
	//read in database file. This is the haro file, put files into the array constructed above
	for( core::Size count=0; utility::io::getline(stream, line); ++count ){
		std::istringstream l( line );
		l >> haro_e1[count] >> haro_e2[count] >> haro_e3[count] >> haro_e4[count] >> haro_e5[count] >> haro_e6[count] >> haro_e7[count];
	}
	stream.close();

	core::io::database::open( stream, filename[3] );
	//read in database file. This is the haro file, put files into the array constructed above
	for( core::Size count=0; utility::io::getline(stream, line); ++count ){
		std::istringstream l( line );
		l >> orb_e1[count] >> orb_e2[count] >> orb_e3[count] >> orb_e4[count] >> orb_e5[count] >> orb_e6[count] >> orb_e7[count];
		//std::cout << orb_e1[count] << " <- orb_e1, orb_e2->" << orb_e2[count] << std::endl;
	}
	stream.close();



	//assign the arrays as a MathMatrix. Necessary step if using the
	//Bicubic spline class. Notice that the sizes of the matrix are
	//50, 21. This is because I have 50 distances and 21 angles. The
	//distance range from 0.1 - 50 and are incremented in 0.1 bins.
	//The angles range from the arcos -1 to 1 and are incremented in
	//0.1 bins
	numeric::MathMatrix<core::Real> matrix_hpol_e1(50, 21, hpol_e1);
	numeric::MathMatrix<core::Real> matrix_hpol_e2(50, 21, hpol_e2);
	numeric::MathMatrix<core::Real> matrix_hpol_e3(50, 21, hpol_e3);
	numeric::MathMatrix<core::Real> matrix_hpol_e4(50, 21, hpol_e4);
	numeric::MathMatrix<core::Real> matrix_hpol_e5(50, 21, hpol_e5);
	numeric::MathMatrix<core::Real> matrix_hpol_e6(50, 21, hpol_e6);
	numeric::MathMatrix<core::Real> matrix_hpol_e7(50, 21, hpol_e7);

	numeric::MathMatrix<core::Real> matrix_haro_e1(50, 21, haro_e1);
	numeric::MathMatrix<core::Real> matrix_haro_e2(50, 21, haro_e2);
	numeric::MathMatrix<core::Real> matrix_haro_e3(50, 21, haro_e3);
	numeric::MathMatrix<core::Real> matrix_haro_e4(50, 21, haro_e4);
	numeric::MathMatrix<core::Real> matrix_haro_e5(50, 21, haro_e5);
	numeric::MathMatrix<core::Real> matrix_haro_e6(50, 21, haro_e6);
	numeric::MathMatrix<core::Real> matrix_haro_e7(50, 21, haro_e7);

	numeric::MathMatrix<core::Real> matrix_orb_e1(50, 21, orb_e1);
	numeric::MathMatrix<core::Real> matrix_orb_e2(50, 21, orb_e2);
	numeric::MathMatrix<core::Real> matrix_orb_e3(50, 21, orb_e3);
	numeric::MathMatrix<core::Real> matrix_orb_e4(50, 21, orb_e4);
	numeric::MathMatrix<core::Real> matrix_orb_e5(50, 21, orb_e5);
	numeric::MathMatrix<core::Real> matrix_orb_e6(50, 21, orb_e6);
	numeric::MathMatrix<core::Real> matrix_orb_e7(50, 21, orb_e7);

	//the spline will behave as a first derivative. This means that the borders of the spline will be smoothed out if the first_derivative
	//is set to 0. This is controlled through an enum and put into an array.
	numeric::interpolation::spline::BorderFlag behavior[2] =
	{
			numeric::interpolation::spline::e_FirstDer,
			numeric::interpolation::spline::e_FirstDer
	};


	const core::Real start[2] = {0.1, -1.0}; //starting position of values in the matrix.
	const core::Real delta[2] = {0.1, 0.1}; //change in values. Needed to increment the values
	const bool linear_cont[2] ={true, true}; //if outside of range continue linearly




	//what is the characteristic of the spline at 0 values. Should be smoothed out at edges. Powerful if used with e_FristDer
	const std::pair<core::Real, core::Real> first_deriv[2] =
	{
			std::pair<core::Real, core::Real>(0,0),
			std::pair<core::Real, core::Real>(0,0)
	};

	//create all the splines that are needed
	numeric::interpolation::spline::BicubicSpline *hpol_e1_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e2_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e3_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e4_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e5_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e6_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *hpol_e7_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;

	numeric::interpolation::spline::BicubicSpline *haro_e1_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e2_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e3_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e4_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e5_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e6_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *haro_e7_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;

	numeric::interpolation::spline::BicubicSpline *orb_e1_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e2_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e3_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e4_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e5_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e6_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;
	numeric::interpolation::spline::BicubicSpline *orb_e7_BicubicSpline = new numeric::interpolation::spline::BicubicSpline;



	//train all the splines. This needs to be done once
	hpol_e1_BicubicSpline->train(behavior, start, delta, matrix_hpol_e1, linear_cont, first_deriv);
	hpol_e2_BicubicSpline->train(behavior, start, delta, matrix_hpol_e2, linear_cont, first_deriv);
	hpol_e3_BicubicSpline->train(behavior, start, delta, matrix_hpol_e3, linear_cont, first_deriv);
	hpol_e4_BicubicSpline->train(behavior, start, delta, matrix_hpol_e4, linear_cont, first_deriv);
	hpol_e5_BicubicSpline->train(behavior, start, delta, matrix_hpol_e5, linear_cont, first_deriv);
	hpol_e6_BicubicSpline->train(behavior, start, delta, matrix_hpol_e6, linear_cont, first_deriv);
	hpol_e7_BicubicSpline->train(behavior, start, delta, matrix_hpol_e7, linear_cont, first_deriv);

	haro_e1_BicubicSpline->train(behavior, start, delta, matrix_haro_e1, linear_cont, first_deriv);
	haro_e2_BicubicSpline->train(behavior, start, delta, matrix_haro_e2, linear_cont, first_deriv);
	haro_e3_BicubicSpline->train(behavior, start, delta, matrix_haro_e3, linear_cont, first_deriv);
	haro_e4_BicubicSpline->train(behavior, start, delta, matrix_haro_e4, linear_cont, first_deriv);
	haro_e5_BicubicSpline->train(behavior, start, delta, matrix_haro_e5, linear_cont, first_deriv);
	haro_e6_BicubicSpline->train(behavior, start, delta, matrix_haro_e6, linear_cont, first_deriv);
	haro_e7_BicubicSpline->train(behavior, start, delta, matrix_haro_e7, linear_cont, first_deriv);

	orb_e1_BicubicSpline->train(behavior, start, delta, matrix_orb_e1, linear_cont, first_deriv);
	orb_e2_BicubicSpline->train(behavior, start, delta, matrix_orb_e2, linear_cont, first_deriv);
	orb_e3_BicubicSpline->train(behavior, start, delta, matrix_orb_e3, linear_cont, first_deriv);
	orb_e4_BicubicSpline->train(behavior, start, delta, matrix_orb_e4, linear_cont, first_deriv);
	orb_e5_BicubicSpline->train(behavior, start, delta, matrix_orb_e5, linear_cont, first_deriv);
	orb_e6_BicubicSpline->train(behavior, start, delta, matrix_orb_e6, linear_cont, first_deriv);
	orb_e7_BicubicSpline->train(behavior, start, delta, matrix_orb_e7, linear_cont, first_deriv);

	//store all splines in a private member look up table. This allows for looking up the values in the actual score and provides an
	//interface for the OrbitalsScore class.
	hpol_e1_spline_ = hpol_e1_BicubicSpline;
	hpol_e2_spline_ = hpol_e2_BicubicSpline;
	hpol_e3_spline_ = hpol_e3_BicubicSpline;
	hpol_e4_spline_ = hpol_e4_BicubicSpline;
	hpol_e5_spline_ = hpol_e5_BicubicSpline;
	hpol_e6_spline_ = hpol_e6_BicubicSpline;
	hpol_e7_spline_ = hpol_e7_BicubicSpline;

	haro_e1_spline_ = haro_e1_BicubicSpline;
	haro_e2_spline_ = haro_e2_BicubicSpline;
	haro_e3_spline_ = haro_e3_BicubicSpline;
	haro_e4_spline_ = haro_e4_BicubicSpline;
	haro_e5_spline_ = haro_e5_BicubicSpline;
	haro_e6_spline_ = haro_e6_BicubicSpline;
	haro_e7_spline_ = haro_e7_BicubicSpline;

	orb_e1_spline_ = orb_e1_BicubicSpline;
	orb_e2_spline_ = orb_e2_BicubicSpline;
	orb_e3_spline_ = orb_e3_BicubicSpline;
	orb_e4_spline_ = orb_e4_BicubicSpline;
	orb_e5_spline_ = orb_e5_BicubicSpline;
	orb_e6_spline_ = orb_e6_BicubicSpline;
	orb_e7_spline_ = orb_e7_BicubicSpline;

	//make everything into a map. The lookup key is the mm_atom_name, or the first variable. The spline is stored in the second variable
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_1", *hpol_e1_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_2", *hpol_e2_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_3", *hpol_e3_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_4", *hpol_e4_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_5", *hpol_e5_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_6", *hpol_e6_spline_) );
	hpol_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_7", *hpol_e7_spline_) );

	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_1", *haro_e1_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_2", *haro_e2_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_3", *haro_e3_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_4", *haro_e4_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_5", *haro_e5_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_6", *haro_e6_spline_) );
	haro_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_7", *haro_e7_spline_) );

	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_1", *orb_e1_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_2", *orb_e2_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_3", *orb_e3_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_4", *orb_e4_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_5", *orb_e5_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_6", *orb_e6_spline_) );
	orb_splines_.insert( std::pair<std::string, numeric::interpolation::spline::BicubicSpline>("EL_7", *orb_e7_spline_) );



/*	for(core::Real i=0.1; i <= 50; i+=0.1){
core::Size number=0;
		for(core::Real j=-1; j<=1; j+=0.1){
			++number;
			if( number == 21){
				std::cout << hpol_e1_BicubicSpline->F(numeric::MakeVector(i,j)) << std::endl;

			}else{
				std::cout << hpol_e1_BicubicSpline->F(numeric::MakeVector(i,j)) << " ";
				//std::cout << i << " " << j << std::endl;
			}
		}
	}*/



}//end orbitalsLookup

//get the energy from the bicubic spline. This is done by using the mm atom name as a key in the map. We pass in the distance and angle.
core::Real OrbitalsLookup::get_energy
(
		const h_type & h_enum,
		const std::string & mm_atom_name,
		const core::Real & distance,
		const core::Real & angle
)
const{
	core::Real energy(0);
	numeric::interpolation::spline::BicubicSpline spline;
	if(h_enum == hpol){
		spline = hpol_splines_.find(mm_atom_name)->second;
		energy = spline.F(numeric::MakeVector(distance, angle));
	}
	if(h_enum == haro){
		spline = haro_splines_.find(mm_atom_name)->second;
		energy = spline.F(numeric::MakeVector(distance, angle));

	}
	if(h_enum == orb){
		spline = orb_splines_.find(mm_atom_name)->second;
		energy = spline.F(numeric::MakeVector(distance, angle));
	}
	if(energy >0){
		energy =0;
	}

	return energy;


}


//I really dont know what I am doing here. I copied most of this code from the carbon hbond
//seems to work. Feel free to explain to me what I just did.
Vector OrbitalsLookup::get_derivative
(
		const h_type & h_enum,
		const std::string & mm_atom_name,
		const core::Real & distance,
		const core::Real & angle,
		Vector  & r_vec,
		Vector  & z_i,
		Vector  & B_A_vector
)
const{
	core::Real d_x(0);
	core::Real d_y(0);
	core::Real derivative(0.0);
	core::Real energy(0.0);




	numeric::interpolation::spline::BicubicSpline spline;
	if(h_enum == hpol){
		spline = hpol_splines_.find(mm_atom_name)->second;
		d_x += spline.dFdx(numeric::MakeVector(distance, angle));
		d_y += spline.dFdy(numeric::MakeVector(distance, angle));
		energy += spline.F(numeric::MakeVector(distance, angle));

	}
	if(h_enum == haro){
		spline = haro_splines_.find(mm_atom_name)->second;
		d_x += spline.dFdx(numeric::MakeVector(distance, angle));
		d_y += spline.dFdy(numeric::MakeVector(distance, angle));
		energy += spline.F(numeric::MakeVector(distance, angle));

	}
	if(h_enum == orb){
		spline = orb_splines_.find(mm_atom_name)->second;
		d_x += spline.dFdx(numeric::MakeVector(distance, angle));
		d_y += spline.dFdy(numeric::MakeVector(distance, angle));
		energy += spline.F(numeric::MakeVector(distance, angle));
	}
	if(energy >0){
		energy =0;
	}

	Vector  H_A_norm_vector ( r_vec ); H_A_norm_vector.normalize();
	Vector  D_H_norm_vector ( z_i ); D_H_norm_vector.normalize();
	Vector  B_A_norm_vector ( B_A_vector ); B_A_norm_vector.normalize();
	Real neg_cos_theta =  dot ( H_A_norm_vector, D_H_norm_vector );
	Real neg_cos_psi   = -dot ( H_A_norm_vector, B_A_norm_vector );




	Real theta_fade_value( 1.0 ), theta_fade_deriv( 0.0 );
	Real psi_fade_value( 1.0 ), psi_fade_deriv( 0.0 );
	core::scoring::rna::get_fade_correction( neg_cos_theta, 0.5, 1.5, 0.2, theta_fade_value, theta_fade_deriv );
	core::scoring::rna::get_fade_correction( neg_cos_psi,   0.4, 1.5, 0.2, psi_fade_value,   psi_fade_deriv );

	Vector deriv_vector = -d_x * theta_fade_value * psi_fade_value * H_A_norm_vector;


	core::Real neg_sin_theta = -(cross(D_H_norm_vector, H_A_norm_vector)).length();
	Vector theta_i = -cross( cross(D_H_norm_vector, H_A_norm_vector), H_A_norm_vector);
	theta_i.normalize();
	Vector theta_fade_gradient = (1./ r_vec.length()) * theta_fade_deriv * neg_sin_theta * theta_i;
	deriv_vector += energy * psi_fade_value * theta_fade_gradient;


	Real neg_sin_psi = -(cross(B_A_norm_vector, H_A_norm_vector)).length();
	Vector psi_i = cross( cross(B_A_norm_vector, H_A_norm_vector), H_A_norm_vector);
	psi_i.normalize();
	Vector psi_fade_gradient = (1. / r_vec.length()) * psi_fade_deriv * neg_sin_psi * psi_i;
	deriv_vector += energy * theta_fade_value * psi_fade_gradient;

return deriv_vector;
/*
	core::Real const r= r_vec.length();
	core::Real const z = dot(z_i, r_vec);

	core::Real dEdr = energy * d_x;
	//dEdr += energy * d_y;

	Vector x_i = r_vec - z_i * dot(r_vec,z_i);
	core::Real const x = x_i.length();
	x_i = x_i.normalize();

	Real const dEdx = dEdr * (x/r) + ( - x * z ) * d_y / (r * r * r);
	Real const dEdz = dEdr * (z/r) + ( r*r - z*z) * d_y / (r * r * r);


	Vector deriv_vector = dEdx * x_i + dEdz * z_i;
	return deriv_vector;*/
	//return deriv_vector;

	//Vector B = d_y * 1/r_vec.length() * (z_i.normalize() - r_vec.normalize() * dot(z_i, r_vec/r_vec.length()));
	//return B;


}




}//namespace orbitals

}//namespace scoring

}//namespace core
