// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/movers/PrepackMover.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

// Unit headers
#include <protocols/ProteinInterfaceDesign/movers/PrepackMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/PrepackMoverCreator.hh>

// Project headers
#include <core/scoring/ScoreFunction.hh>
#include <utility/Tag/Tag.hh>
#include <protocols/moves/DataMap.hh>
#include <core/pack/task/operation/NoRepackDisulfides.hh>
#include <core/pack/task/operation/RestrictToInterface.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/PackerTask.hh>
#include <protocols/moves/RotamerTrialsMinMover.hh>
#include <protocols/moves/PackRotamersMover.hh>

#include <core/pack/rotamer_set/UnboundRotamersOperation.hh>
#include <core/scoring/constraints/RotamerConstraint.hh>

#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/conformation/Conformation.hh>
// Auto-header: duplicate removed #include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <protocols/moves/MinMover.hh>
#include <core/options/option.hh>
#include <core/kinematics/MoveMap.hh>
#include <protocols/moves/RigidBodyMover.hh>

//Auto Headers
#include <core/pose/Pose.hh>


namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace core;
using namespace std;
using namespace core::scoring;
using namespace protocols::moves;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.PrepackMover" );

std::string
PrepackMoverCreator::keyname() const
{
	return PrepackMoverCreator::mover_name();
}

protocols::moves::MoverOP
PrepackMoverCreator::create_mover() const {
	return new PrepackMover;
}

std::string
PrepackMoverCreator::mover_name()
{
	return "Prepack";
}

PrepackMover::PrepackMover() :
	protocols::moves::PackRotamersMover( PrepackMoverCreator::mover_name() ),
	jump_num_( 0 )
{}

PrepackMover::PrepackMover(
	core::scoring::ScoreFunctionCOP scorefxn,
	core::Size jump_num
) :
	protocols::moves::PackRotamersMover( PrepackMoverCreator::mover_name() ),
	scorefxn_( scorefxn ),
	jump_num_( jump_num )
{}


protocols::moves::MoverOP
PrepackMover::clone() const {
	return( protocols::moves::MoverOP( new PrepackMover( *this ) ) );
}

protocols::moves::MoverOP
PrepackMover::fresh_instance() const {
	return protocols::moves::MoverOP( new PrepackMover );
}

PrepackMover::~PrepackMover() {}

/// @details separate bound partners (if any), minimize, do rotamer trials, and re-minimize.
void PrepackMover::apply( pose::Pose & pose )
{

	// separate any bound partners
	protocols::moves::RigidBodyTransMoverOP translate;
	if( (jump_num_ > 0) && (pose.conformation().num_chains() > 1) ) {
		TR<<"Translating along jump #"<<jump_num_<<std::endl;
		translate = new protocols::moves::RigidBodyTransMover( pose, jump_num_ ) ;
		translate->step_size( 1000.0 );
		translate->apply( pose );
	}

	// pre-minimize sidechains
	TR << "Pre-minimizing structure..." << std::endl;
	core::kinematics::MoveMapOP mm( new core::kinematics::MoveMap );
	mm->clear();
	mm->set_chi( true );
	for ( core::Size i = 1; i <= pose.total_residue(); ++i) {
		if ( !pose.residue(i).is_protein() ) {
			mm->set_chi( i, false );
			continue;
		}
		// Check for disulfide bonded cysteines
		if( pose.residue(i).type().name() == "CYD" ) mm->set_chi( i, false );
	}
	protocols::moves::MinMover min_mover( mm, scorefxn_, "dfpmin_armijo_nonmonotone", 1e-5, true/*nblist*/, false/*deriv_check*/  );
	min_mover.apply( pose );

	// rotamer trials
	// make a local packertask, reading resfiles and including current rotamers, excluding disulfides
	TR << "Performing repack..." << std::endl;
	using namespace core::pack::task;
	TaskFactoryOP tf;
	if( task_factory() ) tf = new TaskFactory( *task_factory() );
	else tf = new TaskFactory;
	tf->push_back( new operation::InitializeFromCommandline );
	tf->push_back( new operation::IncludeCurrent );
	tf->push_back( new operation::RestrictToRepacking );
	tf->push_back( new operation::NoRepackDisulfides );

	// incorporating Ian's UnboundRotamer operation.
	// note that nothing happens if unboundrot option is inactive!
	core::pack::rotamer_set::UnboundRotamersOperationOP unboundrot = new core::pack::rotamer_set::UnboundRotamersOperation();
	unboundrot->initialize_from_command_line();
	core::pack::task::operation::AppendRotamerSetOP unboundrot_operation = new core::pack::task::operation::AppendRotamerSet( unboundrot );
	tf->push_back( unboundrot_operation );
	core::scoring::constraints::load_unboundrot(pose); // adds scoring bonuses for the "unbound" rotamers, if any

	if (core::options::option[ core::options::OptionKeys::docking::norepack1 ]()) tf->push_back( new operation::DockingNoRepack1( jump_num_) );
	if (core::options::option[ core::options::OptionKeys::docking::norepack2 ]()) tf->push_back( new operation::DockingNoRepack2( jump_num_) );

	//in case there is a resfile, information in this resfile overrides the computed task
	if( core::options::option[core::options::OptionKeys::packing::resfile].user() ) {
		tf->push_back( new operation::ReadResfile );
	}

	if( core::options::option[core::options::OptionKeys::docking::dock_rtmin].user() ) {
		protocols::moves::RotamerTrialsMinMover rtmin( scorefxn_, tf );
		rtmin.apply( pose );
	}
	else {
		PackerTaskOP task = tf->create_task_and_apply_taskoperations( pose );
		protocols::moves::PackRotamersMover pack( scorefxn_, task );
		pack.apply( pose );
	}

	// post-minimize
	// using packer include_current() will make sure that these minimized rotamers are used.
	TR << "Post-minimizing structure..." << std::endl;;
	min_mover.apply( pose );
	TR << "Done!\n";

	// move back together
	if( (jump_num_ > 0) && (pose.conformation().num_chains() > 1) ) {
		translate->trans_axis().negate();
		translate->apply( pose );
	}

	//final rescore to get everyone on the same page
	(*scorefxn_)(pose);
	TR.flush();
}

std::string
PrepackMover::get_name() const {
	return PrepackMoverCreator::mover_name();
}

void
PrepackMover::parse_my_tag( TagPtr const tag, DataMap & data, protocols::filters::Filters_map const & filters, Movers_map const & movers, core::pose::Pose const & pose )
{
	std::string const scorefxn( tag->getOption<string>( "scorefxn", "score12" ));
	jump_num_ = tag->getOption<core::Size>("jump_number", 1 );
	scorefxn_ = new ScoreFunction( *data.get< ScoreFunction * >( "scorefxns", scorefxn) );
	parse_task_operations( tag, data, filters, movers, pose );
	TR << "Prepack mover with scorefxn " << scorefxn << " over jump number " << jump_num_ << std::endl;
}


} //movers
} //ProteinInterfaceDesign
} //protocols
