// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.


/// @file protocols/ProteinInterfaceDesign/movers/SetupHotspotConstraintsMover.cc
/// @brief
/// @author Jacob Corn (jecorn@u.washington.edu), Sarel Fleishman (sarelf@uw.edu)

#include <protocols/ProteinInterfaceDesign/movers/SetupHotspotConstraintsMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/SetupHotspotConstraintsMoverCreator.hh>
#include <core/pose/Pose.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/hotspot_hashing/HotspotStubSet.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/types.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreType.hh>
#include <core/util/Tracer.hh>
#include <utility/Tag/Tag.hh>
#include <protocols/moves/DataMap.hh>

//Auto Headers
#include <complex>


namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace protocols::moves;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.SetupHotspotConstraintsMover" );

std::string
SetupHotspotConstraintsMoverCreator::keyname() const
{
	return SetupHotspotConstraintsMoverCreator::mover_name();
}

protocols::moves::MoverOP
SetupHotspotConstraintsMoverCreator::create_mover() const {
	return new SetupHotspotConstraintsMover;
}

std::string
SetupHotspotConstraintsMoverCreator::mover_name()
{
	return "SetupHotspotConstraints";
}

SetupHotspotConstraintsMover::SetupHotspotConstraintsMover() :
	protocols::moves::Mover( "SetupHotspotConstraintsMover" ),
	chain_to_design_( 2 ),
	CB_force_constant_( 1.0 ),
	worst_allowed_stub_bonus_( 0.0 ),
	apply_self_energies_( true ),
	bump_cutoff_( 4.0 ),
	apply_ambiguous_constraints_( true ),
	colonyE_( false )
	{}

protocols::moves::MoverOP
SetupHotspotConstraintsMover::clone() const{
	return protocols::moves::MoverOP( new SetupHotspotConstraintsMover( *this ) );
}

protocols::moves::MoverOP
SetupHotspotConstraintsMover::fresh_instance() const{
	return protocols::moves::MoverOP( new SetupHotspotConstraintsMover() );
}

SetupHotspotConstraintsMover::SetupHotspotConstraintsMover(
	protocols::hotspot_hashing::HotspotStubSetCOP hotspot_stub_set,
	core::Size const chain_to_design,
	core::Real const & CB_force_constant,
	core::Real const & worst_allowed_stub_bonus,
	bool const apply_self_energies,
	core::Real const & bump_cutoff,
	bool const apply_ambiguous_constraints,
	bool const colonyE
) :
	protocols::moves::Mover( "SetupHotspotConstraintMover" ),
	chain_to_design_(chain_to_design),
	CB_force_constant_(CB_force_constant),
	worst_allowed_stub_bonus_(worst_allowed_stub_bonus),
	apply_self_energies_(apply_self_energies),
	bump_cutoff_(bump_cutoff),
	apply_ambiguous_constraints_(apply_ambiguous_constraints),
	colonyE_( colonyE)
{
//		packer_task_ = packer_task->clone();
	hotspot_stub_set_ = new protocols::hotspot_hashing::HotspotStubSet( *hotspot_stub_set );
}

SetupHotspotConstraintsMover::SetupHotspotConstraintsMover( SetupHotspotConstraintsMover const & init ) :
	protocols::moves::Mover( init ),
	chain_to_design_( init.chain_to_design_),
	CB_force_constant_(init.CB_force_constant_),
	worst_allowed_stub_bonus_(init.worst_allowed_stub_bonus_),
	apply_self_energies_(init.apply_self_energies_),
	bump_cutoff_(init.bump_cutoff_),
	apply_ambiguous_constraints_(init.apply_ambiguous_constraints_),
	colonyE_( init.colonyE_ )
{
	hotspot_stub_set_ = new protocols::hotspot_hashing::HotspotStubSet( *init.hotspot_stub_set_ );
}

void
SetupHotspotConstraintsMover::apply( core::pose::Pose & pose ) {
	if ( colonyE_ ) {
		protocols::hotspot_hashing::HotspotStubSetOP colonyE_set = hotspot_stub_set_->colonyE();
		hotspot_stub_set_ = colonyE_set;
	}
	if ( std::abs(CB_force_constant_) > 1E-9 ) {
		hotspot_stub_set_->add_hotspot_constraints_to_pose( pose, chain_to_design_, hotspot_stub_set_,
			CB_force_constant_, worst_allowed_stub_bonus_, apply_self_energies_, bump_cutoff_, apply_ambiguous_constraints_ );
	} else {
		core::scoring::constraints::ConstraintSetOP empty_constraint_set = new core::scoring::constraints::ConstraintSet;
		pose.constraint_set( empty_constraint_set );
	}
}

std::string
SetupHotspotConstraintsMover::get_name() const {
return "SetupHotspotConstraintsMover";
}

/// This needs to be parsed before all other movers b/c it changes scorefxns
void
SetupHotspotConstraintsMover::parse_my_tag( TagPtr const tag, DataMap & data, protocols::filters::Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	using core::Real;
  std::string const hotspot_fname( tag->getOption<std::string>( "stubfile", "stubs.pdb" ) );
  chain_to_design_ = tag->getOption<Size>( "redesign_chain", 2 );

  CB_force_constant_ = tag->getOption<Real>( "cb_force", 0.5 );
  worst_allowed_stub_bonus_ = tag->getOption<Real>( "worst_allowed_stub_bonus", 0 );
  apply_self_energies_ = tag->getOption<bool>( "apply_stub_self_energies", 0 );
  bump_cutoff_ = tag->getOption<Real>( "apply_stub_bump_cutoff", 10. );
  apply_ambiguous_constraints_ = tag->getOption<bool>( "pick_best_energy_constraint", 1 );
	core::Real const bb_stub_cst_weight( tag->getOption< core::Real >( "backbone_stub_constraint_weight", 1.0 ) );

  colonyE_ = tag->getOption<bool>( "colonyE", 0 );

  hotspot_stub_set_ = new hotspot_hashing::HotspotStubSet;
  hotspot_stub_set_->read_data( hotspot_fname );

	TR<<"applying hotspot hashing constraints top pose with stubs file "<<hotspot_fname<<" cb_force weight of "<<CB_force_constant_<<
			", apply ambiguous constraints set to "<<apply_ambiguous_constraints_<< " and colonyE set to " << colonyE_ << "\n";
	data.add( "constraints" , "hotspot_stubset", hotspot_stub_set_ );

	for( std::map< std::string, utility::pointer::ReferenceCountOP >::const_iterator it = (data)[ "scorefxns" ].begin(); it!=(data)[ "scorefxns" ].end(); ++it ){
		using namespace core::scoring;
		ScoreFunctionOP scorefxn( *data.get< ScoreFunction * >( "scorefxns", it->first) );
		core::Real const weight( scorefxn->get_weight( backbone_stub_constraint ) );
		if( weight == 0.0 ){
			scorefxn->set_weight( backbone_stub_constraint, bb_stub_cst_weight );
			TR<<"Setting bacbkone_stub_constraint weight in scorefxn "<<it->first<<" to "<<bb_stub_cst_weight<<'\n';
		}
		else
			TR<<"Skipping resetting of backbone_stub_constraint weight in "<<it->first<<" which is already preset to "<<weight<<'\n';
	}
	TR.flush();
}

SetupHotspotConstraintsMover::~SetupHotspotConstraintsMover() {}

} //movers
} //ProteinInterfaceDesign
} //protocols
