// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file DockingHighRes
/// @brief protocols that are specific to high resolution docking
/// @detailed
///		This contains the functions that create initial positions for docking
///		You can either randomize partner 1 or partner 2, spin partner 2, or
///		perform a simple perturbation.
/// 	Also contains docking mcm protocol
/// @author Monica Berrondo
/// @author Modified by Sergey Lyskov
/// @author Modified by Sid Chaudhury
/// @author Modified by Jacob Corn


// Package Headers
#include <protocols/docking/DockingHighRes.hh>
#include <protocols/docking/DockTaskFactory.hh>

// Project Headers
#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

// For some reason, the full header has to be included in the header. since it's there, it doesn't need to be included here too.
// Ideally the forward declaration would be included in the header and the full header would be here.
#include <core/pack/task/TaskFactory.hh> 

// ObjexxFCL Headers

//Utility Headers
#include <utility/tools/make_vector1.hh>

#include <core/options/option.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/util/Tracer.hh>

// C++ Headers

using core::util::T;

using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockingHighRes");

//     originally from dock_structure.cc Jeff Gray April 2001

using namespace core;

namespace protocols {
namespace docking {

// default constructor
DockingHighRes::DockingHighRes() : Mover()
{
	init( utility::tools::make_vector1<core::SSize>(1) ); // operate on the first jump
	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "docking", "docking_min" );
	scorefxn_pack_ = core::scoring::ScoreFunctionFactory::create_score_function( "standard" );
}
// default constructor
DockingHighRes::DockingHighRes(
	core::Size const rb_jump
) : Mover()
{
	init( utility::tools::make_vector1<core::SSize>(rb_jump) ); // operate on the first jump
	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "docking", "docking_min" );
	scorefxn_pack_ = core::scoring::ScoreFunctionFactory::create_score_function( "standard" );
}

// constructor with arguments
// only one movable jump
DockingHighRes::DockingHighRes(
	core::Size const rb_jump,
	core::scoring::ScoreFunctionCOP scorefxn
) : Mover(), scorefxn_(scorefxn), scorefxn_pack_(scorefxn)
{
	init( utility::tools::make_vector1<core::SSize>(rb_jump) );
}

// constructor with arguments
// only one movable jump, scoring and packing defined
DockingHighRes::DockingHighRes(
	core::Size const rb_jump,
	core::scoring::ScoreFunctionCOP scorefxn,
	core::scoring::ScoreFunctionCOP scorefxn_pack
) : Mover(), scorefxn_(scorefxn), scorefxn_pack_(scorefxn_pack)
{
	init( utility::tools::make_vector1<core::SSize>(rb_jump) );
}

// constructor with arguments
// vector of  movable jumps, scoring and packing defined
DockingHighRes::DockingHighRes(
	DockJumps const movable_jumps,
	core::scoring::ScoreFunctionCOP scorefxn,
	core::scoring::ScoreFunctionCOP scorefxn_pack
) : Mover(), scorefxn_(scorefxn), scorefxn_pack_(scorefxn_pack)
{
	init( movable_jumps );
}
	
DockingHighRes::DockingHighRes( DockingHighRes const & old_instance ) : Mover( old_instance )
{
	sc_min_ = old_instance.sc_min_;
	rt_min_ = old_instance.rt_min_;
	partners_ = old_instance.partners_;

	scorefxn_ = old_instance.scorefxn_->clone();
	scorefxn_pack_ = old_instance.scorefxn_pack_->clone();

	movable_jumps_ = old_instance.movable_jumps_;

	tf_ = new core::pack::task::TaskFactory( *old_instance.tf_ );
	tf2_ = new DockTaskFactory( *old_instance.tf2_ );
}

void DockingHighRes::init( DockJumps const movable_jumps )
{
	moves::Mover::type( "DockingHighRes" );

	partners_ = "_";
	sc_min_ = false;
	rt_min_ = false;

	movable_jumps_ = movable_jumps;
	tf2_ = new DockTaskFactory();

	init_from_options();
}
//destructor
DockingHighRes::~DockingHighRes() {}

void DockingHighRes::init_from_options()
{
	using namespace core::options;
	if( option[ OptionKeys::docking::dock_rtmin ].user() )
		set_rt_min(option[ OptionKeys::docking::dock_rtmin ]());

	if( option[ OptionKeys::docking::sc_min ].user() )
		set_sc_min(option[ OptionKeys::docking::sc_min ]());

	if( option[ OptionKeys::docking::partners ].user() )
		set_partners(option[ OptionKeys::docking::partners ]());
}

void DockingHighRes::register_options()
{
	using namespace core::options;
	
	option.add_relevant( OptionKeys::docking::dock_rtmin );
	option.add_relevant( OptionKeys::docking::sc_min );
	option.add_relevant( OptionKeys::docking::partners );
}
	
	
void
DockingHighRes::set_task_factory( core::pack::task::TaskFactoryCOP tf )
{
	tf_ = new core::pack::task::TaskFactory( *tf );
}

protocols::docking::DockTaskFactoryCOP DockingHighRes::tf2() 
{ 
	return tf2_; 
}

void DockingHighRes::set_scorefxn( core::scoring::ScoreFunctionCOP scorefxn )
{
	scorefxn_ = scorefxn;
}

void DockingHighRes::set_scorefxn_pack( core::scoring::ScoreFunctionCOP scorefxn_pack )
{ 
	scorefxn_pack_ = scorefxn_pack; 
}

core::scoring::ScoreFunctionCOP DockingHighRes::scorefxn()
{ 
	return scorefxn_; 
}

core::scoring::ScoreFunctionCOP DockingHighRes::scorefxn_pack()
{ 
	return scorefxn_pack_; 
}
	
core::pack::task::TaskFactoryCOP DockingHighRes::task_factory()
{ 
	return tf_;
}

std::string
DockingHighRes::get_name() const {
	return "DockingHighRes";
}

} // namespace docking
} // namespace protocols
