// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet;
//
// ThiS file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   DockingProtocol.cc
///
/// @brief
/// @author Monica Berrondo
/// @author Sid Chaudhury
/// @author Jeff Gray


// Unit Headers
#include <protocols/docking/DockingProtocol.hh>

// Package Headers
#include <protocols/docking/util.hh>
#include <protocols/docking/metrics.hh>
#include <protocols/docking/DockFilters.hh>
#include <protocols/docking/DockingLowRes.hh>
#include <protocols/docking/DockMinMover.hh>
#include <protocols/docking/DockMCMProtocol.hh>
#include <protocols/docking/DockingHighResLegacy.hh>
#include <protocols/docking/SidechainMinMover.hh>
#include <protocols/docking/DockingInitialPerturbation.hh>


// Project Headers
#include <core/chemical/ChemicalManager.fwd.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/io/pdb/file_data.hh>

#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableDataType.hh>

#include <core/pack/task/PackerTask.hh>
//#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/NoRepackDisulfides.hh>
#include <core/pack/task/operation/RestrictToInterface.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh> // PreventRepackingRLT
#include <core/pack/task/operation/OperateOnCertainResidues.hh>

#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.tmpl.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/InterchainPotential.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/ConstraintSet.hh>

#include <protocols/moves/ConformerSwitchMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/ScoreMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/ConstraintSetMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>
#include <protocols/moves/RotamerTrialsMinMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>

#include <protocols/ScoreMap.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/ProteinInterfaceDesign/dock_design_filters.hh>

// Utility Headers
#include <utility/file/FileName.hh>
#include <utility/tools/make_vector1.hh>

#include <core/types.hh>
#include <core/util/prof.hh>
#include <core/util/Tracer.hh>
#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/DiagnosticData.hh>

#include <core/options/option.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/cluster.OptionKeys.gen.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>

// Numeric Headers

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/format.hh>

// C++ headers
#include <algorithm>

using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockingProtocol");
using namespace core;

namespace protocols {
namespace docking {

// Constructors
DockingProtocol::DockingProtocol()
{
	user_defined_ = false;
	init(utility::tools::make_vector1<core::SSize>(1), false, false, true, NULL, NULL);
}

DockingProtocol::DockingProtocol(
	Size const rb_jump,
	bool const low_res_protocol_only,
	bool const docking_local_refine,
	bool const autofoldtree,
	core::scoring::ScoreFunctionCOP docking_score_low,
	core::scoring::ScoreFunctionCOP docking_score_high
	) : Mover()
{
	user_defined_ = true;
	init(utility::tools::make_vector1<core::SSize>(rb_jump), low_res_protocol_only, docking_local_refine, autofoldtree, docking_score_low, docking_score_high);
}

DockingProtocol::DockingProtocol(
	DockJumps const movable_jumps,
	bool const low_res_protocol_only,
	bool const docking_local_refine,
	bool const autofoldtree,
	core::scoring::ScoreFunctionCOP docking_score_low,
	core::scoring::ScoreFunctionCOP docking_score_high
	) : Mover()
{
	user_defined_ = true;
	init(movable_jumps, low_res_protocol_only, docking_local_refine, autofoldtree, docking_score_low, docking_score_high);
}
// End of constructors


void DockingProtocol::init(
	DockJumps const movable_jumps,
	bool const low_res_protocol_only,
	bool const docking_local_refine,
	bool const autofoldtree,
	core::scoring::ScoreFunctionCOP docking_score_low,
	core::scoring::ScoreFunctionCOP docking_score_high
	)
{
	Mover::type( "DockingProtocol" );
	movable_jumps_ = movable_jumps;

	// setup all the booleans with default values
	// they will get overwritten by the options and/or passed values
	set_default();
	register_options();
	init_from_options();
	if ( user_defined_ ) {
		low_res_protocol_only_ = low_res_protocol_only;
		docking_local_refine_ = docking_local_refine;
		autofoldtree_ = autofoldtree;
	}

	// correctly set up the score functions from either passed in values or defaults
	if ( docking_score_low() == NULL ) {
		docking_scorefxn_low_ = core::scoring::ScoreFunctionFactory::create_score_function( "interchain_cen" );
	}
	else{
		docking_scorefxn_low_ = docking_score_low;
	}
	if ( docking_score_high() == NULL ) {
		docking_scorefxn_high_ = core::scoring::ScoreFunctionFactory::create_score_function( "docking", "docking_min" ) ;
		docking_scorefxn_pack_ = core::scoring::ScoreFunctionFactory::create_score_function( "standard") ;
		docking_scorefxn_output_ = core::scoring::ScoreFunctionFactory::create_score_function( "docking" );
	}
	else {
		docking_scorefxn_high_ = docking_score_high;
		docking_scorefxn_pack_ = docking_score_high;
		docking_scorefxn_output_ = docking_score_high;
	}

	// set up objects based on the boolean values defined above
	setup_objects();
}

void
DockingProtocol::set_default()
{
	//using namespace core::options;
	using namespace core::scoring;

	reporting_ = true;
	autofoldtree_ = true;

	low_res_protocol_only_ = false;
	docking_local_refine_ = false;
	dock_min_ = false;
	rt_min_ = false;
	sc_min_ = false;
	partners_ = "_";
	cst_weight_ = 0.0;
	cst_file_ = false;
	score_cutoff_ = 1000000.0;
	no_filters_ = false;
	use_new_protocol_ = false;

	lowres_inner_cycles_ = 50;
	lowres_outer_cycles_ = 10;

	// initialize the ensemble movers and file strings
	ensemble1_ = "";
	ensemble2_ = "";
	ensemble1_mover_ = NULL;
	ensemble2_mover_ = NULL;
}
	
void DockingProtocol::setup_objects()
{
	// initialize to null
	lowres_filter_ = NULL;
	highres_filter_ = NULL;

	fold_tree_ = NULL;
	perturber_ = NULL;
	docking_lowres_mover_ = NULL;
	docking_highres_mover_ = NULL;

	// Residue movers
	to_centroid_ = new protocols::moves::SwitchResidueTypeSetMover( core::chemical::CENTROID );
	to_all_atom_ = new protocols::moves::SwitchResidueTypeSetMover( core::chemical::FA_STANDARD );
	sync_objects_with_flags();
}

void DockingProtocol::sync_objects_with_flags()
{
	if ( !docking_local_refine_ )
	{
		if ( !perturber_ ){
			perturber_ = new DockingInitialPerturbation( movable_jumps_, true /*slide into contact*/ );
		}
		if ( !docking_lowres_mover_ ){
			docking_lowres_mover_ = new DockingLowRes( docking_scorefxn_low_, movable_jumps_ );
		}
		if ( !no_filters_ && !lowres_filter_ ) {
			lowres_filter_ = new protocols::docking::DockingLowResFilter();
		}
	}
	else
	{
		perturber_ = NULL;
		docking_lowres_mover_ = NULL;
	}

	if ( !low_res_protocol_only_ ) {
		if ( dock_min_ )
		{
			if ( docking_highres_mover_ )
			{
				if ( docking_highres_mover_->get_name() != "DockMinMover" ) docking_highres_mover_ = NULL;
			}
			if ( !docking_highres_mover_ ) docking_highres_mover_ = new DockMinMover( movable_jumps_, docking_scorefxn_high_ );
		}
		else if ( use_new_protocol_ ){
			if ( docking_highres_mover_ )
			{
				if ( docking_highres_mover_->get_name() != "DockMCMProtocol" ) docking_highres_mover_ = NULL;
			}
			if ( !docking_highres_mover_ ){
				// uses docking_scorefxn_output because three scorefunction still exist
				// After move to new protocol is complete, docking_scorefxn_output will be the same as docking_scorefxn_high
				docking_highres_mover_ = new DockMCMProtocol( movable_jumps_, docking_scorefxn_output_, docking_scorefxn_pack_ );
			}
		}
		else{
			if ( docking_highres_mover_ )
			{
				if ( docking_highres_mover_->get_name() != "DockingHighResLegacy" ) docking_highres_mover_ = NULL;
			}
			if ( !docking_highres_mover_ ){
				docking_highres_mover_ = new DockingHighResLegacy( movable_jumps_, docking_scorefxn_high_, docking_scorefxn_pack_ );			}
		}
		if ( !no_filters_ && !highres_filter_ ) {
			highres_filter_ = new protocols::docking::DockingHighResFilter();
		}
	}
	else {
		docking_highres_mover_ = NULL;
	}
	if ( no_filters_ ){
		highres_filter_ = NULL;
		lowres_filter_ = NULL;
	}
	flags_and_objects_are_in_sync_ = true;
	first_apply_with_current_setup_ = true;
}

void
DockingProtocol::init_from_options()
{
	using namespace core::options;

	// check for movable jumps option
	if( option[ OptionKeys::docking::multibody ].user() )
		set_movable_jumps( option[ OptionKeys::docking::multibody ]() );

	// These options default to false
	if( option[ OptionKeys::docking::low_res_protocol_only ].user() )
		set_low_res_protocol_only(option[ OptionKeys::docking::low_res_protocol_only ]());

	if( option[ OptionKeys::docking::docking_local_refine ].user() )
		set_docking_local_refine(option[ OptionKeys::docking::docking_local_refine ]());

	if( option[ OptionKeys::docking::dock_min ].user() )
		set_dock_min(option[ OptionKeys::docking::dock_min ]());

	// Sets the member directly so sync_objects_with_flags won't be triggered prematurely.
	// A public setter exists for this member.
	if( option[ OptionKeys::docking::dock_rtmin ].user() )
		rt_min_ = option[ OptionKeys::docking::dock_rtmin ]();

	// Sets the member directly so sync_objects_with_flags won't be triggered prematurely.
	// A public setter exists for this member.
	if( option[ OptionKeys::docking::sc_min ].user() )
		sc_min_ = option[ OptionKeys::docking::sc_min ]();

	// This defaults to "_"
	if( option[ OptionKeys::docking::partners ].user() )
		set_partners(option[ OptionKeys::docking::partners ]());

	// Defaults to 0
	if( option[ OptionKeys::constraints::cst_weight ].user() )
		set_cst_weight(option[ OptionKeys::constraints::cst_weight ]());

	if( option[ OptionKeys::constraints::cst_file ].user() )
		set_cst_file(option[ OptionKeys::constraints::cst_file ].user());

	//set native pose if asked for
	if ( option[ OptionKeys::in::file::native ].user() ) {
		core::pose::PoseOP native_pose = new core::pose::Pose();
		core::io::pdb::pose_from_pdb( *native_pose, option[ OptionKeys::in::file::native ]() );
		set_native_pose( native_pose );
	}
	else
		set_native_pose(NULL);

	if ( option[ OptionKeys::cluster::output_score_filter ].user() )
		score_cutoff_ = option[ OptionKeys::cluster::output_score_filter ]();

	if ( option[ OptionKeys::docking::no_filters ].user() )
		set_no_filters(option[ OptionKeys::docking::no_filters ]);

	if ( option[ OptionKeys::docking::use_new_protocol ].user() )
		set_use_new_protocol( option[ OptionKeys::docking::no_filters ] );

	// docking low res options
	if ( option[ OptionKeys::docking::docking_centroid_inner_cycles ].user() )
		set_inner_cycles(option[ OptionKeys::docking::docking_centroid_inner_cycles ]());

	if ( option[ OptionKeys::docking::docking_centroid_outer_cycles ].user() )
		set_outer_cycles(option[ OptionKeys::docking::docking_centroid_outer_cycles ]());

	// above cycles must be called before ensemble since the cycle number is overwritten if in ensemble mode
	if ( option[ OptionKeys::docking::ensemble1 ].user() )
		set_ensemble1(option[ OptionKeys::docking::ensemble1 ]());

	if ( option[ OptionKeys::docking::ensemble2 ].user() )
		set_ensemble2(option[ OptionKeys::docking::ensemble2 ]());

}

void
DockingProtocol::register_options()
{
	using namespace core::options;

	option.add_relevant( OptionKeys::docking::multibody );
	option.add_relevant( OptionKeys::docking::low_res_protocol_only );
	option.add_relevant( OptionKeys::docking::docking_local_refine );
	option.add_relevant( OptionKeys::docking::dock_min );
	option.add_relevant( OptionKeys::docking::dock_rtmin );
	option.add_relevant( OptionKeys::docking::sc_min );
	option.add_relevant( OptionKeys::docking::partners );
	option.add_relevant( OptionKeys::docking::no_filters );
	option.add_relevant( OptionKeys::constraints::cst_weight );
	option.add_relevant( OptionKeys::constraints::cst_file );
	option.add_relevant( OptionKeys::run::score_only );
	option.add_relevant( OptionKeys::in::file::native );
	option.add_relevant( OptionKeys::docking::use_new_protocol );

	// low res protocol options
	option.add_relevant( OptionKeys::docking::docking_centroid_inner_cycles );
	option.add_relevant( OptionKeys::docking::docking_centroid_outer_cycles );
	option.add_relevant( OptionKeys::docking::ensemble1 );
	option.add_relevant( OptionKeys::docking::ensemble2 );
}

void
DockingProtocol::finalize_setup( pose::Pose & pose ) //setupobjectsrequiringposeduringapply
{
	if( autofoldtree_ ) {
		TR << "Setting docking foldtree" << std::endl;
		TR << "old fold tree: " << pose.fold_tree() << std::endl;
		util::setup_foldtree( pose, partners_, movable_jumps_ );
		TR << "new fold tree: " << pose.fold_tree() << std::endl;
	}

	fold_tree_ = pose.fold_tree();

	// check for native and input pose
	// input pose is used further down for the recover sidehchains mover
	if ( !get_input_pose() ) {
		core::pose::PoseOP input_pose = new core::pose::Pose(pose);
		set_input_pose( input_pose );
	}

	if ( !get_native_pose() ) {
		TR << "Danger Will Robinson! Native is an impostor!" << std::endl;
		core::pose::PoseOP native_pose = new core::pose::Pose(pose);
		set_native_pose( native_pose );
	}

	// if an ensemble file is defined for either partner, both partners must be defined
	if ( ensemble1_ != "" || ensemble2_ != "" ) {
		if ( ensemble1_ == "" || ensemble2_ == "" ) utility_exit_with_message( "Must define ensemble file for both partners");
		runtime_assert( movable_jumps_.size() == 1 ); // ensemble mode is only allowed for traditional docking
		core::Size const rb_jump = movable_jumps_[1];
		Size start_res(1), end_res(1), cutpoint(pose.fold_tree().cutpoint_by_jump( rb_jump ));

		lowres_inner_cycles_ = 25;

		TR << "Ensemble 1: " << ensemble1_ << std::endl;
		start_res = 1;
		end_res = cutpoint;

		ensemble1_mover_ = new protocols::moves::ConformerSwitchMover( start_res, end_res, rb_jump, docking_scorefxn_low_, ensemble1_ );

		TR << "Ensemble 2: " << ensemble2_ << std::endl;
		start_res = cutpoint + 1;
		end_res = pose.total_residue();

		ensemble2_mover_ = new protocols::moves::ConformerSwitchMover( start_res, end_res, rb_jump, docking_scorefxn_low_, ensemble2_ );

		// recover sidechains mover is not needed with ensemble docking since the sidechains are recovered from the partners in the ensemble file
		recover_sidechains_ = NULL;
	} else {
		recover_sidechains_ = new protocols::moves::ReturnSidechainMover( *get_input_pose() );
	}

	// pass the ensemble movers to the lowres protocol
	if ( docking_lowres_mover_ ) {
		docking_lowres_mover_->set_ensemble1_mover( ensemble1_mover_ );
		docking_lowres_mover_->set_ensemble2_mover( ensemble2_mover_ );
		docking_lowres_mover_->set_inner_cycles( lowres_inner_cycles_ );
		docking_lowres_mover_->set_outer_cycles( lowres_outer_cycles_ );
	}

	//initialize docking filters
	if ( highres_filter_ ) highres_filter_->set_score_cutoff( score_cutoff_ );

	if ( cst_file_ ) {
		// only do this if ConstraintSet isn't defined.
		TR << "Danger Will Robinson! Constraints not supported in docking.  Do not use.  You will be sad." << std::endl;
		if ( !pose.constraint_set() ) {
			protocols::moves::ConstraintSetMoverOP docking_constraint = new protocols::moves::ConstraintSetMover();
			docking_constraint->apply( pose );
		}
		if( lowres_filter_ ) lowres_filter_->set_use_constraints( cst_weight_ );
	}

	if ( cst_weight_!=0.0 ) {
		TR << "Danger Will Robinson! Constraints not supported in docking.  Do not use.  You will be sad." << std::endl;
		core::scoring::ScoreFunctionOP docking_scorefxn_cst, docking_highres_cst;
		docking_scorefxn_cst = new core::scoring::ScoreFunction( *docking_scorefxn_low_ );
		docking_scorefxn_cst->set_weight( core::scoring::atom_pair_constraint, cst_weight_ );
		set_lowres_scorefxn(docking_scorefxn_cst);
		// pass the scorefunction to the low res mover
		docking_lowres_mover_->set_scorefxn( docking_scorefxn_low_ );

		docking_highres_cst = new core::scoring::ScoreFunction( *docking_scorefxn_output_ ); // needs to use the non-min score function to match assemble_domains test with constraints
		docking_highres_cst->set_weight( core::scoring::atom_pair_constraint, cst_weight_ );
		set_highres_scorefxn( docking_highres_cst, docking_scorefxn_pack_ ); // sets csts for mc and minimization, but not packing
		// pass the score function to the high res mover
		docking_highres_mover_->set_scorefxn( docking_scorefxn_high_ );
	}
}

//destructor
DockingProtocol::~DockingProtocol() {}

//clone
protocols::moves::MoverOP
DockingProtocol::clone() const {
	return( new DockingProtocol( movable_jumps_, low_res_protocol_only_, docking_local_refine_, autofoldtree_, docking_scorefxn_low_, docking_scorefxn_high_ ) );
}

void DockingProtocol::set_lowres_scorefxn( core::scoring::ScoreFunctionOP docking_scorefxn_low )
{
	docking_scorefxn_low_ = docking_scorefxn_low;

}

void DockingProtocol::set_highres_scorefxn( core::scoring::ScoreFunctionOP docking_scorefxn_high )
{
	docking_scorefxn_high_ = docking_scorefxn_high;
	docking_scorefxn_pack_ = docking_scorefxn_high;
	docking_scorefxn_output_ = docking_scorefxn_high;
}

void DockingProtocol::set_highres_scorefxn( // delete
	core::scoring::ScoreFunctionCOP docking_scorefxn_high,
	core::scoring::ScoreFunctionCOP docking_scorefxn_pack )
{
	docking_scorefxn_high_ = docking_scorefxn_high;
	docking_scorefxn_pack_ = docking_scorefxn_pack;
	docking_scorefxn_output_ = docking_scorefxn_high;
}

void DockingProtocol::set_highres_scorefxn(
	core::scoring::ScoreFunctionCOP docking_scorefxn_high,
	core::scoring::ScoreFunctionCOP docking_scorefxn_pack,
	core::scoring::ScoreFunctionCOP docking_scorefxn_output )
{
	docking_scorefxn_high_ = docking_scorefxn_high;
	docking_scorefxn_pack_ = docking_scorefxn_pack;
	docking_scorefxn_output_ = docking_scorefxn_output;
}

void DockingProtocol::set_sc_min( bool sc_min )
{
	assert ( !low_res_protocol_only_ );
	if ( !docking_highres_mover_ ){
		sync_objects_with_flags();
	}
	sc_min_ = sc_min;
	docking_highres_mover_->set_sc_min( sc_min );
}

void DockingProtocol::set_rt_min( bool rt_min )
{
	assert ( !low_res_protocol_only_ );
	if ( !docking_highres_mover_ ){
		sync_objects_with_flags();
	}
	rt_min_ = rt_min;
	docking_highres_mover_->set_rt_min( rt_min );
}

void DockingProtocol::set_dock_min( bool const dock_min )
{
	assert ( !low_res_protocol_only_ );
	dock_min_ = dock_min;
}

void DockingProtocol::set_no_filters( bool no_filters )
{
	if ( no_filters != no_filters_ ){
		no_filters_ = no_filters;
		flags_and_objects_are_in_sync_ = false;
	}
}

void DockingProtocol::set_low_res_protocol_only( bool const low_res_protocol_only )
{
	if ( low_res_protocol_only != low_res_protocol_only_ ){
		low_res_protocol_only_ = low_res_protocol_only;
		flags_and_objects_are_in_sync_ = false;
	}
}

void DockingProtocol::set_docking_local_refine( bool const docking_local_refine )
{
	if ( docking_local_refine != docking_local_refine_ ){
		docking_local_refine_ = docking_local_refine;
		flags_and_objects_are_in_sync_ = false;
	}
}

void DockingProtocol::set_use_new_protocol( bool const use_new_protocol )
{
	if ( use_new_protocol != use_new_protocol_ ){
		use_new_protocol_ = use_new_protocol;
		flags_and_objects_are_in_sync_ = false;
	}

}

void DockingProtocol::set_cst_weight( core::Real const cst_weight )
{
	if ( cst_weight != cst_weight_ ){
		cst_weight_ = cst_weight;
		flags_and_objects_are_in_sync_ = false;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Docking apply
///
/// @brief main docking protocol
///
/// @detailed
/// Main function for docking. Includes the following steps:
///      0) prepack mode: prepare a starting structure for later runs
///   OR:
///      1) perturbation of decoy (see docking_perturb_decoy): changes
///         orientation of docking partners
///      2) low-resolution search:
///         refine structure with rigid-body, centroid-mode MC cycles
///      3) high-resolution search:
///         further refinement of structure in fullatom mode
void
DockingProtocol::apply( pose::Pose & pose )
{
	using namespace scoring;

	bool passed_lowres_filter = true;
	bool passed_highres_filter = true;
	std::map < std::string, core::Real > lowres_scores;

	if ( !flags_and_objects_are_in_sync_ ){
		sync_objects_with_flags();
	}

	if ( first_apply_with_current_setup_ ){
		finalize_setup( pose );
		first_apply_with_current_setup_ = false;
	}

	pose.fold_tree( fold_tree_ );

	show(TR);

	core::util::prof_reset();
	protocols::jd2::JobOP job( protocols::jd2::JobDistributor::get_instance()->current_job() );

	// Low resolution docking
	if ( docking_lowres_mover_ ) {
		// convert to centroid mode
		to_centroid_->apply( pose );
		TR << pose.fold_tree() << std::endl;
		// make starting perturbations based on command-line flags over each movable jump
		if ( perturber_ ) {
			perturber_->apply( pose );
			TR << "finished initial perturbation" << std::endl;
		}

		// add scores to jd2 output
		if ( reporting_ && get_native_pose() ) job->add_string_real_pair("st_rmsd", metrics::calc_Lrmsd( pose, *get_native_pose(), movable_jumps_ ));

		docking_lowres_mover_->apply( pose );

		// add scores to jd2 output
		if( reporting_ ) {
			if ( get_native_pose() ) {
				// calculate and store the rms no matter which mode was used
				job->add_string_real_pair("rms", metrics::calc_Lrmsd( pose, *get_native_pose(), movable_jumps_ ));
				job->add_string_real_pair("cen_rms", metrics::calc_Lrmsd( pose, *get_native_pose(), movable_jumps_ ));
			}
			if ( ensemble1_mover_ ) job->add_string_real_pair("conf_num1", ensemble1_mover_->get_current_confnum() );
			if ( ensemble2_mover_ ) job->add_string_real_pair("conf_num2", ensemble2_mover_->get_current_confnum() );
			// store the low res scores for output
			ScoreMap::score_map_from_scored_pose( lowres_scores, pose );
		}
		if ( lowres_filter_ ) passed_lowres_filter = lowres_filter_->apply( pose );
	}

	// High resolution docking
	if ( passed_lowres_filter && docking_highres_mover_ ) {
		if ( !pose.is_fullatom() ) {
			// Convert pose to high resolution and recover sidechains
			to_all_atom_->apply( pose );
			if ( recover_sidechains_ && get_input_pose() ) {
				recover_sidechains_->apply( pose );
			} else {
				if ( ensemble1_mover_ ) {
					ensemble1_mover_->recover_conformer_sidechains( pose );
				}
				if ( ensemble2_mover_ ) {
					ensemble2_mover_->recover_conformer_sidechains( pose );
				}
			}
		}
		docking_highres_mover_->apply( pose );
		( *docking_scorefxn_output_ )( pose );
		if ( highres_filter_ ) passed_highres_filter = highres_filter_->apply( pose );

		if ( reporting_ ) {
			// calculate and store the rms no matter which mode was used
			// this will overwrite the "rms" column from low res if it was calculated
			if ( get_native_pose() ) job->add_string_real_pair("rms", metrics::calc_Lrmsd( pose, *get_native_pose(), movable_jumps_ ) );

			// output low res scores if low res was run
			if ( lowres_scores.size() > 0 ) {
				for ( std::map< std::string, core::Real >::const_iterator pair=lowres_scores.begin(); pair!=lowres_scores.end(); ++pair ) {
					if ( pair->first != "total_score" )	job->add_string_real_pair( pair->first, pair->second );
				}
			}

			// @TODO metrics doesn't need to always take scorefunctions (for example, not necessary for irms or fnat)
			job->add_string_real_pair("I_sc", metrics::calc_interaction_energy( pose, docking_scorefxn_output_, movable_jumps_ ) );
			if ( get_native_pose() ) {
				job->add_string_real_pair("Irms", metrics::calc_Irmsd(pose, *get_native_pose(), docking_scorefxn_output_, movable_jumps_ ));
				job->add_string_real_pair("Fnat", metrics::calc_Fnat( pose, *get_native_pose(), docking_scorefxn_output_, movable_jumps_ ));
			}
		}
	}

	if ( !passed_lowres_filter || !passed_highres_filter ){
		set_last_move_status( protocols::moves::FAIL_RETRY );
	}
	else{
		set_last_move_status( protocols::moves::MS_SUCCESS );
	}

	core::util::prof_show();
}

/// @details  Show the complete setup of the docking protocol
void
DockingProtocol::show( std::ostream & out ) {
	if ( !flags_and_objects_are_in_sync_ ){
		sync_objects_with_flags();
	}
	out << *this;
}

std::ostream & operator<<(std::ostream& out, const DockingProtocol & dp )
{
	using namespace ObjexxFCL::fmt;

	// All output will be 80 characters - 80 is a nice number, don't you think?
	std::string line_marker = "///";
	out << "////////////////////////////////////////////////////////////////////////////////" << std::endl;
	out << line_marker << A( 47, "Rosetta 3 Docking Protocol" ) << space( 27 ) << line_marker << std::endl;
	out << line_marker << space( 74 ) << line_marker << std::endl;
	// Display the movable jumps that will be used in docking
	out << line_marker << " Dockable Jumps: ";

	core::Size spaces_so_far = 23;
	bool first = true;
	for ( DockJumps::const_iterator it = dp.movable_jumps_.begin() ; it != dp.movable_jumps_.end() ; ++it ){
		if (!first) {
			out << ", ";
			spaces_so_far += 2;
		}
		else first = false;

		out << I( 1, *it );
		spaces_so_far += 1;
	}
	core::Size remaining_spaces = 80 - spaces_so_far;
	if ( remaining_spaces > 0 )	out << space( 80 - spaces_so_far );
	out << line_marker << std::endl;

	// Display the state of the low resolution docking protocol that will be used
	out << line_marker << " Low Resolution Docking Protocol:  " << ( ( dp.docking_lowres_mover_ ) ? ( " on " ) : ( "off " ) );
	out << space( 35 ) << line_marker << std::endl;

	// Display the state of the low resolution docking protocol that will be used
	out << line_marker << " High Resolution Docking Protocol: " << ( ( dp.docking_highres_mover_ ) ? ( " on " ) : ( "off " ) );
	out << space( 35 ) << line_marker << std::endl;

	// Display the state of the filters (on or off)
	out << line_marker << " Low Resolution Filter:  " << ( ( dp.lowres_filter_ ) ? ( " on " ) : ( "off " ) );
	out << space( 45 ) << line_marker << std::endl;
	out << line_marker << " High Resolution Filter: " << ( ( dp.highres_filter_ ) ? ( " on " ) : ( "off " ) );
	out << space( 45 ) << line_marker << std::endl;

	// Close the box I have drawn
	out << "////////////////////////////////////////////////////////////////////////////////" << std::endl;
	return out;
}

} //docking
} //protocols
