// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file ./src/protocols/fldsgn/topology/SS_Info2.hh
/// @brief
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

#ifndef INCLUDED_protocols_fldsgn_topology_SS_Info2_HH
#define INCLUDED_protocols_fldsgn_topology_SS_Info2_HH

/// Unit headers
#include <protocols/fldsgn/topology/SS_Info2.fwd.hh>

/// Package headers
#include <protocols/fldsgn/topology/BB_Pos.hh>

// Project headers
#include <core/types.hh>
#include <core/util/datacache/CacheableData.hh>

/// Numeric headers
#include <numeric/xyzVector.hh>

// C++ headers
#include <string>

namespace protocols {
namespace fldsgn {
namespace topology {

class SS_Base : public utility::pointer::ReferenceCount {
public:


	typedef core::Size Size;
	typedef core::Real Real;
	typedef core::Vector Vector;
	typedef protocols::fldsgn::topology::BB_Pos BB_Pos;


public:// construct/destruct


	/// @brief default constructor
	SS_Base();

	/// @brief value constructor
	SS_Base( Size const & begin, Size const & end );

	/// @brief value constructor
	// SS_Base( Size const & begin, Size const & end, Vector const & v );

	/// @brief copy constructor
	SS_Base(	SS_Base const & s );

	/// @brief destructor
	~SS_Base();


public:// accessors


	inline Size begin() const {	return begin_; }

	inline Size end() const {	return end_; }

	inline Size length() const { return end_ - begin_ + 1; }

	inline Vector orient() const { return orient_; }

	inline Vector Nend_pos() const { return Nend_pos_; }

	inline Vector Cend_pos() const { return Cend_pos_; }


public: // calculator


	Vector mid_pos() const;


public:// accessors


	inline bool is_geometry_initialized() const { return is_geometry_initialized_; }


protected: // setters


	void orient( Vector const & v ) { orient_ = v; }

	/// @brief positional vector of N-terminal
	void Nend_pos( Vector const & v ) { Nend_pos_ = v; }

	/// @brief positional vector of C-terminal
	void Cend_pos( Vector const & v ) { Cend_pos_ = v; }

	void is_geometry_initialized( bool const v ) { is_geometry_initialized_ = v; }


private: ///data


	/// @brief begintial residue of strand
	Size begin_;

	/// @brief end residue of strand
	Size end_;

	/// @brief
	bool is_geometry_initialized_;

	/// @brief vector between Calpha of edge residues
	Vector orient_;

	/// @brief positional vector of N- and C- teriminals
	Vector Nend_pos_, Cend_pos_;


};

//////////////////////////////////////////////////////////////////////////////////////////////////////
class Strand : public SS_Base {


	typedef SS_Base Parent;


public:// construct/destruct


	/// @brief default constructor
	Strand();

	/// @brief value constructor
	Strand( Size const & begin, Size const & end );

	/// @brief copy constructor
	Strand(	Strand const & s );

	/// @brief destructor
	~Strand();

  /// @brief
	friend std::ostream & operator<<(std::ostream & out, const Strand & st );


public:


	// @brief
	void set_geometric( BB_Pos const & bbpos );


};

//////////////////////////////////////////////////////////////////////////////////////////////////////
class Helix : public SS_Base {


	typedef SS_Base Parent;
	typedef core::Vector Vector;


public: // constructor/destructor


	/// @brief default constructor
	Helix();

	/// @brief value constructor
	Helix( Size const & begin, Size const & end );

	/// @brief copy constructor
	Helix( Helix const & s );

	/// @brief destructor
	~Helix();

	/// @brief
	Real bend() const { return bend_; }

  /// @brief
	friend std::ostream & operator<<( std::ostream & out, const Helix & hx );


public: //


	/// @brief
	void set_geometric( BB_Pos const & bbpos );


private: // data


	Real bend_;


};

//////////////////////////////////////////////////////////////////////////////////////////////////////
class SS_Info2 : public core::util::datacache::CacheableData {


	typedef core::Size Size;
	typedef core::pose::Pose Pose;
	typedef std::string String;
	typedef protocols::fldsgn::topology::BB_Pos BB_Pos;


public: // constructor/destructor


 	/// @brief default constructor
	SS_Info2();

	/// @brief value constructor
	SS_Info2( String const & secstruct );

	/// @brief value constructor
	SS_Info2( Pose const & pose, String const & secstruct = "" );

	/// @brief copy constructor
	SS_Info2( SS_Info2 const & s );

	/// @brief destructor
	~SS_Info2();


public:


	/// @brief make clone
	core::util::datacache::CacheableDataOP clone() const;

  /// @brief
	friend
	std::ostream & operator<<(std::ostream & out, const SS_Info2 & ssinfo );

	/// @brief initialize parameters of this class
	void initialize( String const & secstruct );

	/// @brief initialize parameters of this class
	void initialize( Pose const & pose, String const & secstruct = "" );


public:


	/// @brief
	inline
	bool
	bbpos_is_set() const
	{
		return bbpos_is_set_;
	}

	/// @brief  return secondary structures
	inline
	String
	secstruct() const
	{
		return secstruct_;
	}

	/// @brief get positional information of backbone
	inline
	BB_Pos const &
	bb_pos() const
	{
		return bb_pos_;
	}

	/// @brief
	inline
	Strands const &
	strands() const
	{
		return strands_;
	}

	/// @brief
	inline
	Helices const &
	helices() const
	{
		return helices_;
	}

	/// @brief
	StrandOP const &
	strand( Size is ) const
	{
		runtime_assert( is <= strands_.size() );
		return strands_[ is ];
	}

	/// @brief
	HelixOP const &
	helix( Size ih ) const
	{
		runtime_assert( ih <= helices_.size() );
		return helices_[ ih ];
	}

	/// @brief
	inline
	Size
	strand_id( Size const nres ) const
	{
		return strand_id_[ nres ];
	}

	/// @brief
	inline
	Size
	helix_id( Size const nres ) const
	{
		return helix_id_[ nres ];
	}

	/// @brief
	inline
	Size
	loop_id( Size const nres ) const
	{
		return loop_id_[ nres ];
	}

	/// @brief
	inline
	Size
	ss_element_id( Size const nres ) const
	{
		return ss_element_id_[ nres ];
	}



public:


	/// @brief set orientation vector of secondary structures given a pose
	void set_SSorient( Pose const & pose );

	/// @brief set orientation vector of secondary structures given a pose which is defined in the constructor
	void set_SSorient();

	/// @brief clear data
	void clear_data();


private:


	/// @brief
	void
	resize( Size const nres );

	/// @brief identify secondary structures
	void identify_ss( String const & secstruct );


private: //data


	/// @brief
	bool bbpos_is_set_;

	/// @brief secondary structures
	String secstruct_;

	/// @brief
	BB_Pos bb_pos_;

	/// @brief
	Strands strands_;
	utility::vector1< Size > strand_id_;

	/// @brief
	Helices helices_;
	utility::vector1< Size > helix_id_;

	/// @brief
	utility::vector1< Size > loop_id_;

	/// @brief
	utility::vector1< Size > ss_element_id_;


};

} // namespace topology
} // namespace fldsgn
} // naemspace protocols

#endif
