// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file ./src/protocols/flxbb/BluePrint.cc
/// @brief the idea of BluePrint is based on the remodel Possu wrote in rosetta++.
/// @author Nobuyasu Koga

// Unit header
#include <protocols/flxbb/BluePrint.hh>

//
#include <protocols/flxbb/BluePrint.hh>
#include <protocols/fldsgn/topology/SS_Info2.hh>
#include <protocols/fldsgn/topology/StrandPairing.hh>
#include <protocols/fldsgn/topology/HelixPairing.hh>

// Project header
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <core/pose/Pose.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/string_util.hh>
#include <boost/lexical_cast.hpp>

//Auto Headers
#include <utility/options/keys/BooleanOptionKey.hh>


static core::util::Tracer TR("protocols.flxbb.BluePrint");

using namespace core;

namespace protocols {
namespace flxbb {

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief default constructor
BluePrint::BluePrint() :
	total_residue_( 0 ),
	ss_info_( new SS_Info2 )
{}


/// @brief value constructor
BluePrint::BluePrint( std::string const & filename ) :
	ss_info_( new SS_Info2 )
{
	if ( ! read_blueprint( filename ) ) {
    TR.Error << "Error in reading blueprint file " << filename << std::endl;
    exit(0);
  }
}


/// @brief destructor
BluePrint::~BluePrint()
{}


/// @brief copy constructor
BluePrint::BluePrint( BluePrint const & src ):
	ReferenceCount(),
	total_residue_( src.total_residue_ ),
	sequence_( src.sequence_ ),
	secstruct_( src.secstruct_ ),
	resnum_( src.resnum_ ),
	resname_( src.resname_ ),
	sstype_( src.sstype_ ),
	buildtype_( src.buildtype_ ),
	resnum_map_( src.resnum_map_ ),
	ss_info_( src.ss_info_ ),
	strand_pairings_( src.strand_pairings_ ),
	helix_pairings_( src.helix_pairings_ )
{}


/// @brief total residue number defined in blueprint file
BluePrint::Size
BluePrint::total_residue() const
{
	return total_residue_;
}


/// @brief sequence defined in blueprint file
BluePrint::String
BluePrint::sequence() const
{
	return sequence_;
}


/// @brief amino acid type at a position in blueprint file
char
BluePrint::sequence( core::Size seqpos ) const
{
	assert( seqpos >= 1 );
	return resname_[ seqpos ];
}


/// @brief secondary structures of blueprint file
BluePrint::String
BluePrint::secstruct() const
{
	return secstruct_;
}


/// @brief secondary structure at a position in blueprint file
char
BluePrint::secstruct( core::Size seqpos ) const
{
	assert( seqpos >= 1 && seqpos <= buildtype_.size() );
	return sstype_[ seqpos ];
}


/// @brief residue number of each position in blueprint file
BluePrint::Size
BluePrint::resnum( core::Size seqpos ) const
{
	assert( seqpos >= 1 && seqpos <= buildtype_.size() );
	return resnum_[ seqpos ];
}


/// @brief translate residue number of pose to that of blueprint file
BluePrint::Size
BluePrint::resnum_map( Size resnum_pose ) const
{
	std::map<Size,Size>::const_iterator itr;
	itr = resnum_map_.find( resnum_pose );
	if( itr != resnum_map_.end() ){
		Size resnum_blueprint = itr->second;
		return resnum_blueprint;
	}else{
		return 0;
	}
}


/// @brief build type at each position in blueprint
char
BluePrint::buildtype( core::Size seqpos ) const
{
	assert( seqpos >= 1 && seqpos <= buildtype_.size() );
	return buildtype_[ seqpos ];
}


/// @brief strand pairings defined at the line of SSPAIR in blueprint
BluePrint::StrandPairings
BluePrint::strand_pairings() const
{
	return strand_pairings_;
}


/// @brief helix pairings defined at the line of HHPAIR in blueprint
BluePrint::HelixPairings
BluePrint::helix_pairings() const
{
	return helix_pairings_;
}


/// @brief secondary structure information
BluePrint::SS_Info2_OP
BluePrint::ssinfo() const
{
	return ss_info_;
}


/// @brief read blueprint file
bool
BluePrint::read_blueprint( std::string const & filename )
{
	using utility::string_split;

  utility::io::izstream data( filename );
  if ( !data ) {
    TR.Error << "can not open blueprint file " << filename << std::endl;
    return false;
  }

	// StranPairing info read from the line of SSPAIR
	StrandPairings spairs;

  String line;
  Size linecount( 0 ), count( 0 );
  while( getline( data, line ) ) {
    linecount++;
    std::vector< String > tokens ( utility::split( line ) );

		// skip reading line that is commented out
    if( tokens[0][0] == '#' ) continue;

		if( tokens[0] == "FOLDINFO" || tokens[0] == "SSPAIR" || tokens[0] == "HHPAIR" ) {
			// read the line of SSPAIR( FOLDINFO ), HHPAIR

			if( tokens.size() > 2 ) {
				TR.Error << "error parsing at " << linecount << " in " << filename << std::endl;
				return false;
			}

			std::vector< String > bpairs( string_split( tokens[1], ';' ) );
			for( std::vector< String >::const_iterator iter = bpairs.begin(); iter != bpairs.end() ; ++iter) {

				std::vector< String > parts( string_split( *iter, '.' ) );
				if( ( tokens[0] == "SSPAIR" && parts.size() != 3 ) || ( tokens[0] == "HHPAIR" && parts.size() != 2 ) ){
					TR.Error << "error parsing: " << *iter << " at " << linecount << " in " << filename << std::endl;
					return false;
				}

				std::vector< String > st( string_split( parts[0], '-' ) );
				Size st1 = boost::lexical_cast<Size>( st[0] );
				Size st2 = boost::lexical_cast<Size>( st[1] );
				runtime_assert( st1 < st2 );

				char para = parts[1][0];
				runtime_assert( para == 'P' || para == 'A' );

				if( tokens[0] == "HHPAIR" ) {
					helix_pairings_.push_back( new HelixPairing( st1, st2, para ) );
				} else {
					Real rsf = boost::lexical_cast<Real>( parts[2] );
					spairs.push_back( new StrandPairing( st1, st2, rsf, para ) );
				}
			} // for ( Vecotr<String> )

		} else {
			// read the line where residue number, amino acid type, secondary structure and build type are written

			// don't need to specify build type in blueprint
			runtime_assert( tokens.size() == 3 || tokens.size() == 4 );

			count ++;
			Size ii = boost::lexical_cast<Size>( tokens[0] );
			char aa ( tokens[1][0] );
			char sec( tokens[2][0] );

			// check the 3rd column where secondary structure info is written
			if ( sec != 'E' && sec != 'H' && sec != 'L' && sec != 'D' ) {
				TR.Error << "unrecognized secstruct char : " << sec << " at lines "	<< linecount << " in " << filename << std::endl;
				return false;
			}

			resnum_.push_back( ii );
			resname_.push_back( aa );
			sstype_.push_back( sec );
			if( ii != 0 ){
				resnum_map_.insert( std::map<Size, Size>::value_type( ii, count ) );
			}

			if( tokens.size() >= 4 ) {

				char build( tokens[3][0] );
				if( build != '.' && build != 'R' && build != 'I' && build != 'X' &&
						build != 'F' && build != 'P' && build != 'C' ) {
					TR.Error << "unrecognized build char : " << build << " at lines "	<< linecount << " in " << filename << std::endl;
					return false;
				}
				if( ii == 0 && build == '.' ) {
					TR.Error << "Cannot have simultaneously '0' at 1st column and '.' in 4th column in blueprint " << std::endl;
					return false;
				}
				buildtype_.push_back( build );

			}else{
				buildtype_.push_back( '.' );
			} // tokens.size() >=4

		}// tokens.size() == 2 || tokens.size() == 3
  } // while ( getline )

  assert( resname_.size() == sstype_.size() );
  total_residue_ = sstype_.size();
  assert( total_residue_ > 0 );

  for( utility::vector1< char >::const_iterator iter = sstype_.begin(); iter != sstype_.end() ; ++iter) {
    secstruct_ += *iter;
  }
  for( utility::vector1< char >::const_iterator iter = resname_.begin(); iter != resname_.end() ; ++iter) {
    sequence_ += *iter;
  }

  TR << secstruct_ << std::endl;
  TR << sequence_ << std::endl;

	ss_info_->initialize( secstruct_ );
	if( spairs.size() > 0 ){
		strand_pairings_ = set_strand_pairings( ss_info_, spairs );
	}

  return true;

} // read_blueprint


/// @brief
BluePrint::StrandPairings
BluePrint::set_strand_pairings( SS_Info2_OP const & ss_info, StrandPairings const & spairs ) const
{
	StrandPairings newspairs;
	for ( utility::vector1< StrandPairingOP >::const_iterator it=spairs.begin(); it!=spairs.end(); ++it ) {

		StrandPairing spair=**it;

		Size istrand( spair.s1() );
		Size jstrand( spair.s2() );
		Real rsf( spair.rgstr_shift() );

    Real ir_ist = ss_info->strand( istrand )->begin();
    Real len_ist = ss_info->strand( istrand )->length();
    Real ir_jst = ss_info->strand( jstrand )->begin();
    Real er_jst = ss_info->strand( jstrand )->end();
    Real len_jst = ss_info->strand( jstrand )->length();

    Real start_ist, start_jst, len,inc;
		char orient;

		if( spair.is_parallel() ){ // parallel

			orient = 'P';
			inc = 1;

			if( rsf >= 0 ) {
				start_ist = ir_ist + rsf;
				start_jst = ir_jst;
				if( len_ist >= (len_jst+rsf) ) {
					//  i =========>
					//  j   ==========>
					len = len_jst;
				}else{
					//  i =========>
					//  j   =====>
					len = len_ist - rsf;
				}
			} else {
				start_ist = ir_ist;
				start_jst = ir_jst - rsf;
				if( len_ist >= (len_jst+rsf) ) {
					//  i      =====>
					//  j   ==========>
					len = len_jst - rsf;
				} else {
					//  i      ==========>
					//  j   ==========>
					len = len_ist ;
				}
			}
		} else { // anti parallel

			orient = 'A';
			inc = -1;

			if( rsf >= 0 ) {
				start_ist = ir_ist;
				start_jst = er_jst - rsf;
				if( len_ist >= (len_jst-rsf) ) {
					//  i     =========>
					//  j  <=========
					len = len_jst - rsf;
				} else {
					//  i     =========>
					//  j       <==========
					len = len_ist;
				}
      } else {
				start_ist = ir_ist - rsf;
				start_jst = er_jst;
				if( len_ist >= (len_jst-rsf) ) {
					//  i   ==========>
					//  j     <=====
					len = len_jst ;
				} else {
					//  i   =========>
					//  j      <=========
					len = len_ist + rsf;
				}
      } // if( rsf )
		} // if is_parallel ?

    StrandPairingOP newspair = new StrandPairing( istrand, jstrand, rsf, orient );
    for( Size i=1; i<=len ;i++ ) {
			runtime_assert( start_ist > 0 && start_jst > 0 );
			newspair->elongate( Size( start_ist ), Size( start_jst ), 0, 0 );
			start_ist ++;
      start_jst += inc;
    }

		// add new sspair
		newspairs.push_back( newspair );

	} // for( spairs )

	return newspairs;

} // BluePrint::set_respairing


/// @brief
void
BluePrint::insert_ss_into_pose( core::pose::Pose & pose )
{
	runtime_assert( pose.total_residue() <= sstype_.size() );
	for ( Size i = 1; i <= total_residue_;/*pose.total_residue();*/ ++i ) {
		pose.set_secstruct( i, sstype_[ i ] );
	}
}

} // ns flxbb
} // ns protocols
