// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Nobuyasu Koga

// Unit Headers
#include <protocols/flxbb/DesignLayerOperation.hh>

// Project Headers
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/pose/Pose.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/PackerTask_.hh>
#include <core/scoring/dssp/Dssp.hh>
#include <core/scoring/sasa.hh>
#include <core/options/option.hh>
#include <core/util/Tracer.hh>

// Utility Headers
#include <utility/vector1.hh>
#include <string>

using core::util::T;
using core::util::Error;
using core::util::Warning;

// option key includes
#include <core/options/keys/flxbb.OptionKeys.gen.hh>

//Auto Headers
#include <ObjexxFCL/format.hh>

using namespace core::options;
using namespace core::options::OptionKeys;

using namespace core;
using namespace core::pack::task;
using namespace protocols::flxbb;

static core::util::Tracer TR("protocols.flxbb.DesignLayerOperation");

DesignLayerOperation::DesignLayerOperation( bool dsgn_core, bool dsgn_boundary, bool dsgn_surface ):
	dsgn_core_( dsgn_core ), dsgn_boundary_( dsgn_boundary ), dsgn_surface_( dsgn_surface ),
	exclude_ala_(false), exclude_met_(false), use_original_( false ),
	pore_radius_( 2.0 ), burial_( 20.0 ), surface_( 40.0 )
{
	if( option[ OptionKeys::flxbb::layer::pore_radius ].user() ){
		pore_radius_ = option[ OptionKeys::flxbb::layer::pore_radius ]() ;
	}
	if( option[ OptionKeys::flxbb::layer::burial ].user() ){
		burial_ = option[ OptionKeys::flxbb::layer::burial ]() ;
	}
	if( option[ OptionKeys::flxbb::layer::surface ].user() ){
		surface_ = option[ OptionKeys::flxbb::layer::surface ]() ;
	}
	TR << " pore_radius : " <<  pore_radius_ << std::endl;
	TR << " burial : " << burial_ << std::endl;
	TR << " surface : " << surface_ << std::endl;
}

DesignLayerOperation::~DesignLayerOperation() {}


void
DesignLayerOperation::apply( pose::Pose const & pose, PackerTask & task ) const {

	// define atom_map for main-chain and CB
	id::AtomID_Map< bool > atom_map;
	id::initialize( atom_map, pose, false );
	for ( Size ir = 1; ir <= pose.total_residue(); ++ir ) {
		for ( Size j = 1; j<=5; ++j ) {
			id::AtomID atom( j, ir );
			atom_map.set( atom, true );
		}
	}

	// calc sasa
	id::AtomID_Map< Real > atom_sasa;
	utility::vector1< Real > rsd_sasa;
	scoring::calc_per_atom_sasa( pose, atom_sasa, rsd_sasa, pore_radius_, false, atom_map );

	// calc dssp
	core::scoring::dssp::Dssp dssp( pose );
	dssp.dssp_reduced();

	// find the position of residues of helix capping and intial residue of helix
	bool flag( false );
	utility::vector1< bool > helix_capping( pose.total_residue(), false );
	utility::vector1< bool > initial_helix( pose.total_residue(), false );
	for( Size i=1; i<=pose.total_residue(); ++i ){
		char ss( dssp.get_dssp_secstruct( i ) );
		if( ss == 'H' && flag == false && i != 1 ){
			initial_helix[ i ] = true;
			helix_capping[ i-1 ] = true;
			flag = true;
		}
		if( ss != 'H' && flag == true ){
			flag = false;
		}
	}

	// terminal residues set to be allaa
	utility::vector1<bool> restrict_to_aa( 20, true );
	task.nonconst_residue_task( 1 ).restrict_absent_canonical_aas( restrict_to_aa );
	task.nonconst_residue_task( pose.total_residue() ).restrict_absent_canonical_aas( restrict_to_aa );

	// assign design residues
	for( Size i=2;i<=pose.total_residue()-1; ++i ){
		char ss( dssp.get_dssp_secstruct( i ) );

		TR << "Resid=" << i << " ,ss=" << ss << " ,Sasa=" << ObjexxFCL::fmt::F( 6, 2, rsd_sasa[ i ] );

		if( task.residue_task( i ).command_string().find( "PIKAA" ) != std::string::npos ){
			TR << " ,Resfile info is used." << std::endl;
			continue;
		}

		if( rsd_sasa[ i ] <= burial_ && dsgn_core_ ){

			utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
			restrict_to_aa[chemical::aa_from_name( "TRP" )] = true;
			restrict_to_aa[chemical::aa_from_name( "PHE" )] = true;
			restrict_to_aa[chemical::aa_from_name( "ILE" )] = true;
			restrict_to_aa[chemical::aa_from_name( "LEU" )] = true;
			restrict_to_aa[chemical::aa_from_name( "VAL" )] = true;
			if( exclude_met_ == false ){
				restrict_to_aa[chemical::aa_from_name( "MET" )] = true;
			}
			if( exclude_ala_ == false ){
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = true;
			}
			if( ss == 'E' ){
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = false;
			}
			if( initial_helix[ i ] == true ){
				restrict_to_aa[chemical::aa_from_name( "PRO" )] = true;
			}
			task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
			TR << " ,Core " << std::endl;

		}else if( rsd_sasa[ i ] >= surface_ && dsgn_surface_ ){

			utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
			if( helix_capping[ i ] == true ){
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				TR << " ,Helix Capping " << std::endl;
			}else{
				restrict_to_aa[chemical::aa_from_name( "GLU" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ARG" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "LYS" )] = true;
				restrict_to_aa[chemical::aa_from_name( "HIS" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "GLN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;
				restrict_to_aa[chemical::aa_from_name( "GLY" )] = true;

				if( initial_helix[ i ] == true ){
					restrict_to_aa[chemical::aa_from_name( "PRO" )] = true;
				}
				if( ss == 'E' || ss == 'H'){
					restrict_to_aa[chemical::aa_from_name( "GLY" )] = false;
				}
				TR << " ,Surface " << std::endl;
			}
			task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );


		}else if( rsd_sasa[ i ] < surface_ && rsd_sasa[ i ] > burial_ && dsgn_boundary_ ){

			if( helix_capping[ i ] == true ){
				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,Helix Capping " << std::endl;
			}else{
				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, true );
				if( exclude_ala_ == true ){
					restrict_to_aa[chemical::aa_from_name( "ALA" )] = false;
				}
				restrict_to_aa[chemical::aa_from_name( "CYS" )] = false;
				restrict_to_aa[chemical::aa_from_name( "PHE" )] = false;
				restrict_to_aa[chemical::aa_from_name( "TRP" )] = false;
				restrict_to_aa[chemical::aa_from_name( "MET" )] = false;
				restrict_to_aa[chemical::aa_from_name( "HIS" )] = false;

				if( ss == 'E' || ss == 'H' ){
					restrict_to_aa[chemical::aa_from_name( "GLY" )] = false;
				}

				if( ss == 'E' ){
					restrict_to_aa[chemical::aa_from_name( "ALA" )] = false;
					restrict_to_aa[chemical::aa_from_name( "PRO" )] = false;
				}
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,Boundary " << std::endl;
			}

		}else{

			if( use_original_ ){
				task.nonconst_residue_task( i ).restrict_to_repacking();
				TR << " ,Original sequence used" << std::endl;
			}else{
				utility::vector1<bool> restrict_to_aa( 20, false );
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = true;
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,No design" << std::endl;
			}

		}
	}

}


operation::TaskOperationOP
DesignLayerOperation::clone() const {
	return new DesignLayerOperation( *this );
}

