// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/parser/DataLoader.cc
/// @brief  Implementation of the XML parser's DataLoader base class (ctor & dstor)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)

// Unit Headers
#include <protocols/jd2/parser/ScoreFunctionLoader.hh>
#include <protocols/jd2/parser/StandardLoaderCreators.hh>

// Project Headers
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/util/Tracer.hh>

// Utility headers
#include <utility/Tag/Tag.hh>


namespace protocols {
namespace jd2 {
namespace parser {

static core::util::Tracer TR( "protocols.jd2.parser.ScoreFunctionLoader" );

ScoreFunctionLoader::ScoreFunctionLoader() {}
ScoreFunctionLoader::~ScoreFunctionLoader() {}

void ScoreFunctionLoader::load_data(
	core::pose::Pose const &,
	utility::Tag::TagPtr const tag,
	moves::DataMap & data
) const
{
	using namespace utility::Tag;
	typedef utility::vector0< TagPtr > TagPtrs;

	TagPtrs const scorefxn_tags( tag->getTags() );

	for( TagPtrs::const_iterator scorefxn_tag_it=scorefxn_tags.begin(); scorefxn_tag_it!=scorefxn_tags.end();
			++scorefxn_tag_it ) {//read user defined scorefxns
		TagPtr const scorefxn_tag = *scorefxn_tag_it;
		using namespace core::scoring;

		ScoreFunctionOP in_scorefxn;
		std::string const scorefxn_name( scorefxn_tag->getName() );
		std::string const scorefxn_weights( scorefxn_tag->getOption<std::string>( "weights", "standard" ) );
		if(  scorefxn_tag->hasOption( "weights" ) && scorefxn_tag->hasOption( "patch" ) ) {
			std::string const scorefxn_patch( scorefxn_tag->getOption<std::string>( "patch" ) );
			in_scorefxn = ScoreFunctionFactory::create_score_function( scorefxn_weights, scorefxn_patch);
			TR << "defined score function \"" << scorefxn_name << "\" with weights \""
				<< scorefxn_weights << "\" and patch \"" << scorefxn_patch << "\"\n";
		} else if ( scorefxn_tag->hasOption( "weights" ) ) {
			in_scorefxn = ScoreFunctionFactory::create_score_function( scorefxn_weights );
			TR << "defined score function \"" << scorefxn_name << "\" with weights \""
				<< scorefxn_weights << "\"\n";
		} else {
			in_scorefxn = new ScoreFunction;
			in_scorefxn->reset();
			TR << "***WARNING***: No weights/patch defined. Defining " << scorefxn_name << " with all-zero weights.\n";
		}
		utility::vector0< TagPtr > const scorefxn_mod_tags( scorefxn_tag->getTags() );
		for( utility::vector0< TagPtr >::const_iterator mod_it=scorefxn_mod_tags.begin(); mod_it!=scorefxn_mod_tags.end(); ++mod_it ) {
			TagPtr const mod_tag = *mod_it;
			if( mod_tag->getName() == "Reweight" ) {
				std::string const scoretype_name( mod_tag->getOption<std::string>( "scoretype" ) );
				core::Real const weight( mod_tag->getOption<core::Real>( "weight" ) );
				TR<<"setting "<<scorefxn_name<<" weight " << scoretype_name << " to " << weight<<'\n';
				core::scoring::ScoreType const type = score_type_from_name( scoretype_name );
				in_scorefxn->set_weight( type, weight );
			}
		}

		// weights for arbitrary ScoreFunctions should be tampered with only as a consequence of user input--NEVER by default

		// hotspot hash constraint
		if ( scorefxn_tag->hasOption("hs_hash") ) {
			core::Real hotspot_hash( 0.0 ); // APL FIX THIS!  This used to be initialized when the HotspotHashingConstraints were read in.
			core::Real const hs_hash( scorefxn_tag->getOption<core::Real>( "hs_hash", hotspot_hash ) );
			TR<<"setting "<<scorefxn_name<<" backbone_stub_constraint to "<<hs_hash<<'\n';
			in_scorefxn->set_weight( backbone_stub_constraint, hs_hash );
		}

		data.add( "scorefxns" , scorefxn_name, in_scorefxn );
	}//end user-defined scorefxns
	TR.flush();
}

DataLoaderOP
ScoreFunctionLoaderCreator::create_loader() const { return new ScoreFunctionLoader; }

std::string
ScoreFunctionLoaderCreator::keyname() const { return "SCOREFXNS"; }


} //namespace parser
} //namespace jd2
} //namespace protocols
