// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/ligand_docking/LigandDesign.cc
///
/// @brief
/// @Gordon Lemmon

#include <protocols/ligand_docking/LigandDesign.hh>
#include <protocols/ligand_docking/LigandDesignCreator.hh>

#include <protocols/ligand_docking/ligand_options/chain_functions.hh>


#include <protocols/moves/Mover.hh>

#include <core/options/option.hh>

#include <core/types.hh>
#include <core/util/prof.hh>

#include <core/util/Tracer.hh>

#include <utility/file/file_sys_util.hh>

// option key includes

#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/chemical/ResidueSelector.hh>
#include <core/chemical/ChemicalManager.hh>

#include <utility/Tag/Tag.hh>

//Auto Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>

#include <core/options/keys/in.OptionKeys.gen.hh>
#include <numeric/random/random.hh>

namespace protocols {
namespace ligand_docking {

static core::util::Tracer ligand_design_tracer("protocols.ligand_docking.LigandDesign", core::util::t_debug);

std::string
LigandDesignCreator::keyname() const
{
	return LigandDesignCreator::mover_name();
}

protocols::moves::MoverOP
LigandDesignCreator::create_mover() const {
	return new LigandDesign;
}

std::string
LigandDesignCreator::mover_name()
{
	return "LigandDesign";
}

LigandDesign::LigandDesign(): Mover("LigandDesign"){
	Mover::type( "LigandDesign" );
	set_fragments();
}

LigandDesign::LigandDesign(LigandDesign const & that):
		protocols::moves::Mover( that ),
		option_file_(that.option_file_),
		fragments_(that.fragments_)
{}

LigandDesign::~LigandDesign() {}

void
LigandDesign::set_fragments(){
	core::chemical::ResidueSelector rs;
	rs.set_property("FRAGMENT");
	core::chemical::ChemicalManager *cm= core::chemical::ChemicalManager::get_instance();
	core::chemical::ResidueTypeSetCAP rsd_set= cm->residue_type_set( core::chemical::FA_STANDARD );
	core::chemical::ResidueTypeCAPs fragment_types= rs.select( *rsd_set );
	ligand_design_tracer<< fragment_types.size()<< " fragment_types"<< std::endl;
	core::chemical::ResidueTypeCAPs::const_iterator begin= fragment_types.begin();
	for(; begin!= fragment_types.end(); ++begin){
		core::conformation::Residue* temp= new core::conformation::Residue(**begin, true);
		fragments_.push_back(temp);
		ligand_design_tracer<< "frag_name: "<< temp->name()<< std::endl;
	}
}

protocols::moves::MoverOP LigandDesign::clone() const {
	return new LigandDesign( *this );
}

protocols::moves::MoverOP LigandDesign::fresh_instance() const {
	return new LigandDesign;
}

std::string LigandDesign::get_name() const{
	return "LigandDesign";
}

///@brief parse XML (specifically in the context of the parser/scripting scheme)
void
LigandDesign::parse_my_tag(
		utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & /*datamap*/,
		protocols::filters::Filters_map const & /*filters*/,
		protocols::moves::Movers_map const & /*movers*/,
		core::pose::Pose const & /*pose*/
)
{
	if ( tag->getName() != "LigandDesign" ) {
		ligand_design_tracer << " received incompatible Tag " << tag << std::endl;
		assert(false);
		return;
	}
	if ( tag->hasOption("option_file") ) {
		option_file_ = tag->getOption<std::string>("option_file");
	}
	//parse_score_function( tag, datamap, filters, movers, pose );
	//parse_task_operations( tag, datamap, filters, movers, pose );
}

bool grow(core::pose::Pose pose, core::Size start, core::Size end){
	return passes_filters(pose,start,end) && has_incomplete_connections(pose,start,end);
}

bool has_incomplete_connections(core::pose::Pose pose, core::Size start, core::Size const end){
	for(;start <= end; ++start){
		core::conformation::Residue const & res= pose.residue(start);
		ligand_design_tracer<< res.name();
		if(res.has_incomplete_connection()){
			ligand_design_tracer<<" has incomplete connection"<< std::endl;
			return true;
		}
		ligand_design_tracer<<" completely connected"<< std::endl;
	}
	ligand_design_tracer<< "no more connections"<< std::endl;
	return false;
}

bool passes_filters(core::pose::Pose const & pose, core::Size start, core::Size const end){
	/// User Defined Filters
	// example
	if(	ligand_options::num_heavy_atoms(start,end,pose)>20 ){
		ligand_design_tracer<< "Reached heavy atom limit"<< std::endl;
		return false;
	}
	if (ligand_options::num_chi_angles(start,end,pose)>10){
		ligand_design_tracer<< "Reached chi angle limit"<< std::endl;
		return false;
	}
	else return true;
}

core::Size
random_connection(core::conformation::Residue const & residue){
	//find connections that are incomplete
	utility::vector1<core::Size> incomplete_connections= get_incomplete_connections(residue);
	return numeric::random::random_element(incomplete_connections);
}

// should be a helper function of residue class
utility::vector1<core::Size>
get_incomplete_connections(core::conformation::Residue const & residue){
	utility::vector1<core::Size> incomplete_connections;
	for(core::Size i=1; i<= residue.n_residue_connections(); ++i){
		if(residue.connection_incomplete(i)){
			incomplete_connections.push_back(i);
		}
	}
	return incomplete_connections;
}

utility::vector1<core::Size>
find_unconnected_residues(
		core::pose::Pose const & pose,
		core::Size start,
		core::Size const end
){
	utility::vector1<core::Size> unconnected_residues;
	for(; start <= end; ++start){
		if( pose.residue(start).has_incomplete_connection() ){
			unconnected_residues.push_back(start);
		}
	}
	return unconnected_residues;
}

void
LigandDesign::apply( core::pose::Pose & pose )
{
	assert(!fragments_.empty());
	using namespace core::options::OptionKeys;
	using core::options::option;
	core::Size ligand_residue_id= pose.n_residue();
	core::conformation::Residue const & ligand= pose.residue(ligand_residue_id);
	assert(ligand.is_ligand());
	assert( ligand.n_residue_connections() > 0);
	core::Size const & chain_id= pose.chain(ligand_residue_id);
	core::Size const start = pose.conformation().chain_begin(chain_id);
	core::Size end = pose.conformation().chain_end(chain_id);
	while(grow(pose, start, end)){
		utility::vector1<core::Size> unconnected_residues = find_unconnected_residues(pose, start,end);
		core::Size const & grow_from= numeric::random::random_element(unconnected_residues);
		core::conformation::ResidueCOP const growth= numeric::random::random_element(fragments_);;
		core::Size grow_from_connection= random_connection(pose.residue(grow_from));
		core::Size growth_connection= random_connection(*growth);
		pose.append_residue_by_bond(*growth, true, growth_connection, grow_from, grow_from_connection);
		//end = pose.conformation().chain_end(chain_id);
		++end;
	}

}

void LigandDesign::fragments_to_string() const{
	core::conformation::ResidueCOPs::const_iterator begin= fragments_.begin();
	for(; begin != fragments_.end(); ++begin){
		core::conformation::Residue const & res= **begin;
		std::string name= res.name();
		core::Size total= res.n_residue_connections();
		core::Size incomplete= get_incomplete_connections(res).size();
		ligand_design_tracer<< "name, total, incomplete"<< name<< " "<< total<<" "<< incomplete<< std::endl;
	}
}

void LigandDesign::add_scores_to_job(
	core::pose::Pose & /*pose*/
)
{
//	using namespace core::scoring;
//	ScoreFunctionCOP sfxn = command_map_.get_score_function();
//
//	core::Real const tot_score = sfxn->score( pose );
//
//	// Which score terms to use
//	typedef utility::vector1<ScoreType> ScoreTypeVec;
//	ScoreTypeVec score_types;
//	for(int i = 1; i <= n_score_types; ++i) {
//		ScoreType ii = ScoreType(i);
//		if ( sfxn->has_nonzero_weight(ii) ) score_types.push_back(ii);
//	}
//
//	protocols::jd2::JobOP job( protocols::jd2::JobDistributor::get_instance()->current_job() );
//	for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii) {
//		job->add_string_real_pair(name_from_score_type(*ii), sfxn->get_weight(*ii) * pose.energies().total_energies()[ *ii ]);
//	}
//	job->add_string_real_pair(name_from_score_type(core::scoring::total_score), tot_score);
}

} // namespace ligand_docking
} // namespace protocols
