// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   /grid/src/protocols/ligand_docking/Transform.cc
/// @author Sam DeLuca


#include <protocols/ligand_docking/TransformCreator.hh>
#include <protocols/ligand_docking/Transform.hh>


#include <core/conformation/Conformation.hh>
#include <core/util/Tracer.hh>


#include <protocols/ligand_docking/ligand_options/chain_functions.hh>
#include <protocols/qsar/scoring_grid/GridManager.hh>
#include <protocols/qsar/qsarMap.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/geometry/RB_geometry.hh>
#include <numeric/numeric.functions.hh>
#include <numeric/random/random.hh>
#include <protocols/moves/DataMap.hh>
#include <utility/Tag/Tag.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/Job.hh>

namespace protocols {
namespace ligand_docking {

static core::util::Tracer transform_tracer("protocols.ligand_docking.ligand_options.transform", core::util::t_debug);

std::string TransformCreator::keyname() const
{
	return TransformCreator::mover_name();
}

protocols::moves::MoverOP TransformCreator::create_mover() const
{
	return new Transform;
}

std::string TransformCreator::mover_name()
{
	return "Transform";
}

Transform::Transform(): Mover("Transform"), grid_manager_(0), transform_info_()
{

}

Transform::~Transform()
{
	//
}

protocols::moves::MoverOP Transform::clone() const
{
	return new Transform (*this);
}

protocols::moves::MoverOP Transform::fresh_instance() const
{
	return new Transform;
}

std::string Transform::get_name() const
{
	return "Transform";
}

void Transform::parse_my_tag
(
		utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & data_map,
		protocols::filters::Filters_map const & /*filters*/,
		protocols::moves::Movers_map const & /*movers*/,
		core::pose::Pose const & /*pose*/
)
{
	if ( tag->getName() != "Transform" )
	{
		utility_exit_with_message("This should be impossible");
	}
	//tag->write(transform_tracer,1);
	if ( ! tag->hasOption("chain") ) utility_exit_with_message("'Transform' mover requires chain tag");
	//if ( ! tag->hasOption("distribution") ) utility_exit_with_message("'Translate' mover requires distribution tag");
	if ( ! tag->hasOption("move_distance") ) utility_exit_with_message("'Transform' mover requires move_distance tag");
	if (! tag->hasOption("box_size") ) utility_exit_with_message("'Transform' mover requires box_size tag");
	if ( ! tag->hasOption("angle") ) utility_exit_with_message("'Transform' mover requires angle tag");
	if ( ! tag->hasOption("cycles") ) utility_exit_with_message("'Transform' mover requires cycles tag");

	if(data_map.has("scoringgrid","default"))
	{
		grid_manager_ = data_map.get<qsar::scoring_grid::GridManager *>("scoringgrid","default");
	}
	else
	{
		utility_exit_with_message("ERROR: you are using Transform without specifying a GridManger");

		//grid_manager_ = new qsar::scoring_grid::GridManager(40.0, 0.25);
	}

	transform_info_.chain = tag->getOption<std::string>("chain");
	//std::string distribution_str = tag->getOption<std::string>("distribution");
	//transform_info_.distribution = get_distribution(distribution_str);
	transform_info_.move_distance = tag->getOption<core::Real>("move_distance");
	transform_info_.box_size = tag->getOption<core::Real>("box_size");
	transform_info_.angle = tag->getOption<core::Real>("angle");
	transform_info_.cycles = tag->getOption<core::Size>("cycles");
}

void Transform::apply(core::pose::Pose & pose)
{
	assert(transform_info_.chain.size() == 1);
	transform_info_.chain_id = ligand_options::get_chain_id_from_chain(transform_info_.chain, pose);
	transform_info_.jump_id = ligand_options::get_jump_id_from_chain_id(transform_info_.chain_id, pose);
	core::Size const begin(pose.conformation().chain_begin(transform_info_.chain_id));
	core::Vector const center(protocols::geometry::downstream_centroid_by_jump(pose, transform_info_.jump_id));
	assert(grid_manager_ != 0); //something has gone hopelessly wrong if this triggers


	if(!grid_manager_->is_qsar_map_attached())
	{
		core::conformation::ResidueOP residue = new core::conformation::Residue(pose.residue(begin));
		qsar::qsarMapOP qsar_map(new qsar::qsarMap("default",residue));
		qsar_map->fill_with_value(1);
		grid_manager_->set_qsar_map(qsar_map);
	}

	core::conformation::Residue original_residue = pose.residue(begin);

	jd2::JobOP current_job = protocols::jd2::JobDistributor::get_instance()->current_job();
	std::string tag = current_job->input_tag();

	grid_manager_->initialize_all_grids(center);
	grid_manager_->update_grids(pose,center,tag);

	core::pose::Pose best_pose(pose);
	core::Real best_score(grid_manager_->total_score(original_residue));
	core::Real temperature = 100;
	core::Vector original_center(original_residue.xyz(original_residue.nbr_atom()));
	for(core::Size cycle = 1; cycle <= transform_info_.cycles; ++cycle)
	{
		transform_ligand(pose);

		//delete residue;
		core::conformation::Residue residue(pose.residue(begin));

		best_score = grid_manager_->total_score(residue);
		core::Real const boltz_factor((best_score-best_score)/temperature);
		core::Real const probability = std::exp( std::min (40.0, std::max(-40.0,boltz_factor)) ) ;

		core::Vector new_center(residue.xyz(residue.nbr_atom()));

		if(new_center.distance(original_center) > transform_info_.box_size) //Reject the new pose
		{
			pose = best_pose;
		}else if(probability < 1)  // Accept the new pose
		{
			best_score = grid_manager_->total_score(residue);
			best_pose = pose;
		}else if(RG.uniform() < probability)  //Accept the new pose
		{
			best_score = grid_manager_->total_score(residue);
			best_pose = pose;
		}else  //Reject the new pose
		{
			pose = best_pose;
		}
	}

}

void Transform::transform_ligand(core::pose::Pose & pose)
{
	if(transform_info_.angle ==0 && transform_info_.move_distance == 0)
	{
		transform_tracer <<"WARNING: angle and distance are both 0.  Transform will do nothing" <<std::endl;
		return;
	}

	protocols::moves::RigidBodyMoverOP mover;
	mover = new protocols::moves::RigidBodyPerturbMover(transform_info_.jump_id,transform_info_.angle,transform_info_.move_distance);
	//core::Size chain_begin = pose.conformation().chain_begin(transform_info_.chain_id);

	mover->apply(pose);
	pose.update_actcoords();
}

void Transform::change_conformer
(
	core::Size const jump_id,
	core::pose::Pose & pose,
	core::Size const & residue_id
)
{
	//lets implement this later
}

}
}
