// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loophash/MPI_LoopHashRefine_Master.cc
/// @brief
/// @author Mike Tyka

#define TRDEBUG TR.Debug

// MPI headers
#ifdef USEMPI
#include <mpi.h> //keep this first
#endif

#include <protocols/loophash/MPI_LoopHashRefine.hh>
#include <protocols/loophash/MPI_LoopHashRefine_Master.hh>
#include <protocols/loophash/WorkUnit_LoopHash.hh>
#include <protocols/wum/WorkUnit_BatchRelax.hh>
#include <protocols/wum/WorkUnitBase.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/util.hh>
#include <core/io/pose_stream/MetaPoseInputStream.hh>
#include <core/io/pose_stream/util.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/ProteinSilentStruct.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/lh.OptionKeys.gen.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/Tracer.hh>
#include <protocols/wum/SilentStructStore.hh>
#include <ObjexxFCL/format.hh>
/// ObjexxFCL headers
#include <ObjexxFCL//string.functions.hh>
#include <ObjexxFCL/format.hh>

#include <numeric/random/random.hh>

#include <unistd.h>

using namespace ObjexxFCL;
using namespace ObjexxFCL::fmt;

namespace protocols {
namespace loophash {

using namespace protocols::wum;

static core::util::Tracer TR("MPI.LHR.Master");

static numeric::random::RandomGenerator RG(3893251);  // <- Magic number, do not change it (and dont try and use it anywhere else)

void
MPI_LoopHashRefine_Master::set_defaults(){
	using namespace core::options;
	using namespace core::options::OptionKeys;
	max_loophash_per_structure_ = option[ OptionKeys::lh::max_loophash_per_structure ]();
	batch_relax_chunks_         = option[ OptionKeys::lh::mpi_batch_relax_chunks ]();
	batch_relax_absolute_max_   = option[ OptionKeys::lh::mpi_batch_relax_absolute_max ]();
	outbound_wu_buffer_size_    = option[ OptionKeys::lh::mpi_outbound_wu_buffer_size ]();
	loophash_split_size_        = option[ OptionKeys::lh::mpi_loophash_split_size     ]();
	library_expiry_time_        = option[ OptionKeys::lh::library_expiry_time ](); 
}


void
MPI_LoopHashRefine_Master::init(){
	// Are we resuming an old job ?
	if( mpi_resume() != "" ){ 
		TR << "Resuming job from IDENT:  " <<  mpi_resume() << std::endl;
		load_state( mpi_resume() );	
	} else {
		load_structures_from_cmdline_into_library( max_lib_size() * master_rank() );
	}
	TR << "STARTLIB: " << std::endl;
	print_library();
}



void
MPI_LoopHashRefine_Master::go()
{
	// initialize master (this is a virtual functino call and this function is overloaded by the children of this class)
	TR << "Init Master: " << mpi_rank() << std::endl;
	init();

	TR << "Master Node: Waiting for job requests..." << std::endl;
	while(true){
		// process any incoming messages such as incoming
		TRDEBUG << "Master: processing msgs.." << std::endl;
		process_incoming_msgs();

		TRDEBUG << "Master: process incoming" << std::endl;
		process_inbound_wus();

		TRDEBUG << "Master: process outbound" << std::endl;
		process_outbound_wus();

		// ok, we've done all our work, now wait until we hear from our slaves
		process_incoming_msgs( true );

		print_stats_auto();
	}
}



/// @brief figure out what to do with incoming WUs.
/// Some will be returning WUs that need to be resent others will be finished and will need
/// reintegration into the library
void
MPI_LoopHashRefine_Master::process_inbound_wus(){
	using namespace protocols::loops;

	check_library_expiry_dates();

	if( inbound().size() > 0 ){
		TRDEBUG << "Processing inbound WUs on master.." << std::endl;
	}
	while( inbound().size() > 0 )
	{
		WorkUnitBaseOP  next_wu =  inbound().pop_next();
		runtime_assert( next_wu );
		
		// skip returning waiting WUs
		if ( next_wu->get_wu_type() == "waitwu" ) continue;
	
		// Upcast to a StructureModifier WU
		WorkUnit_SilentStructStore* structure_wu = dynamic_cast<  WorkUnit_SilentStructStore * > ( (WorkUnitBase*) (&(*next_wu)) );
		
		// If upcast was unsuccessful - warn and ignore.		
		if ( structure_wu == NULL ){
			TR << "Cannot save structural data for WU: " << std::endl;
			next_wu->print( TR );
			continue;
		}
		
		// Otherwise extract structures and figure out what to do with them
		TRDEBUG << "Saving decoy store.. " << std::endl;
		SilentStructStore &decoys = structure_wu->decoys();

		if ( structure_wu->get_wu_type() == "loophasher" ){
			totaltime_loophash() += structure_wu->get_run_time();
			TR << "LoopHash return: " << decoys.size() << " structs in " << structure_wu->get_run_time() << "s " << std::endl;
			n_loophash_++;
			//to_be_relaxed_.add( decoys );
			if( decoys.size() > 0 ){
				add_relax_batch( decoys );
				total_structures_ += decoys.size();
			}
		} else
		if ( structure_wu->get_wu_type() == "resultpack" ){
			decoys.all_sort_silent_scores();
			// dump structures
			TR << "Emperor sent: " << decoys.size() << " structs" << std::endl;
			print_library();	
			add_structures_to_library( decoys );
			print_library();
			dump_structures( decoys );
		} else 
		if ( structure_wu->get_wu_type() == "batchrelax" ){
			decoys.all_sort_silent_scores();
			totaltime_batchrelax_ += structure_wu->get_run_time();
			n_batchrelax_ ++;
			TR << "BatchRelax return: " << decoys.size() << " structs in " << structure_wu->get_run_time() << "s " << std::endl;
			add_structures_to_library( decoys );
			dump_structures( decoys );
		} else {
			TR.Error << "Unknown workunit received. " << std::endl;
		}


	}

	print_stats();
}



void
MPI_LoopHashRefine_Master::process_outbound_wus(){
	TRDEBUG << "Adding loophash WUs if necessary .. " << std::endl;
	if( outbound().size() < outbound_wu_buffer_size_ ){
		if ( library_central().size() == 0 ){
			TR.Error << "FATAL ERROR:  library_central_ is empty! " << std::endl;
      utility_exit_with_message( "FATAL ERROR:  library_central_ is empty! " );
		}
		// pick a random structure from the library

		core::Size finished_structures;
		for( SilentStructStore::iterator it = library_central().begin(); it !=  library_central().end(); it ++ ){
			if( max_loophash_per_structure_ > (*it)->get_energy("lhcount"))
			{
					TRDEBUG << "Adding: " << (*it) << "  " << (*it)->get_energy("lhcount") << std::endl;
				 (*it)->add_energy( "lhcount",  (*it)->get_energy("lhcount") + 1.0 ); 
				create_loophash_WUs( *it );
			}else{
					finished_structures = 1;
					TRDEBUG << "Already done: " << (*it) << "  " << (*it)->get_energy("lhcount") << std::endl;
			}
		}
		TR << "WARNING: " << finished_structures << "  " << library_central().size() << std::endl;
		if ( finished_structures >= library_central().size() ){
			TR << "WARNING: The starting structs exhausted!" << std::endl;
		}
	}

	save_state_auto();
}


void
MPI_LoopHashRefine_Master::create_loophash_WUs( const core::io::silent::SilentStructOP &start_struct ){

		runtime_assert( start_struct );
		core::pose::Pose start_pose;
		start_struct->fill_pose( start_pose );
  	core::chemical::switch_to_residue_type_set( start_pose, core::chemical::CENTROID);
		core::pose::set_ss_from_phipsi( start_pose );

		core::io::silent::ProteinSilentStruct pss;
		pss.fill_struct( start_pose );
		pss.copy_scores( *start_struct );
	
		// first cound up "round" counter - just counts how many times each structure has been
		// thorugh the loop hasher
		core::Size round = (core::Size)pss.get_energy("round");
		round++;
		pss.add_energy("round", round );
		pss.add_energy( "masterid", mpi_rank() );

		core::Size start_ir = 1;
		core::Size end_ir = 1;

		for( ;start_ir< start_pose.total_residue(); start_ir+=loophash_split_size_ )
		{
			end_ir =  std::min( start_ir + loophash_split_size_ - 1, start_pose.total_residue());
			if( end_ir < start_ir) end_ir = start_ir;
			TR << "Adding a new loophash WU: " << start_ir << " - " << end_ir << std::endl;

			WorkUnit_LoopHashOP new_wu = new WorkUnit_LoopHash( start_ir, end_ir );  
			// this is unsatisfying.. why can't i use the template ? grrr C++ thou are limited.
			new_wu->set_wu_type("loophasher");
			new_wu->decoys().add( pss );
			new_wu->clear_serial_data();
			outbound().add( new_wu );
		}


}


void
MPI_LoopHashRefine_Master::add_relax_batch( SilentStructStore &start_decoys ){
	if( start_decoys.size() == 0 ) return;
	TR << "Adding relax WUs.." << start_decoys.size() << std::endl;

	core::Size chunks  = 1 + core::Size( floor( core::Real(start_decoys.size()) / core::Real( batch_relax_chunks_ ) ) );
	core::Size batchrelax_batchsize_ = (start_decoys.size() / chunks) + 1;
	core::Size dcount=0;
	while( dcount < start_decoys.size() ){

		WorkUnit_BatchRelaxOP new_wu = new WorkUnit_BatchRelax();
		new_wu->set_wu_type("batchrelax");
		core::Size lcount=0;

		for(lcount=0; lcount < batchrelax_batchsize_; lcount++ ){
			if ( dcount < start_decoys.size() ){
				core::io::silent::SilentStructOP new_relax_structure =  start_decoys.get_struct( dcount );
				TRDEBUG << "AddRelaxStructure: " << format_silent_struct(new_relax_structure)  << std::endl;
				new_wu->decoys().add( new_relax_structure ); 
			}
			dcount++;
		}
	
		// Mix up the order
    std::random_shuffle( new_wu->decoys().begin(), new_wu->decoys().end());
		
		// make sure the chunk size doesnt exceed batch_relax_absolute_max_
		core::Size chunk_size = new_wu->decoys().size();
		new_wu->decoys().limit( batch_relax_absolute_max_ );
		
		total_structures_relax_ += new_wu->decoys().size();	
		TR << "Adding " << new_wu->decoys().size() << "/" << chunk_size << " (random pick) structures to be relaxed ..." << std::endl;
		new_wu->clear_serial_data();
		
		// Relax work units have a lot of structures and fill up the queue and lead to memory crashes. Thus they get prioritized and added at the beginning of the queue!!
		outbound().push_front( new_wu );
	}


}


// this goes through the library and identifies structures that have not managed to get replaced
// for some cutoff amount of time. It will send back this structure and request a new structure with the same ssid from
// the emperor.
void 
MPI_LoopHashRefine_Master::check_library_expiry_dates(){

	core::Size current_time = time(NULL);
	for( SilentStructStore::iterator jt =  library_central().begin();
									jt != library_central().end(); jt ++ )
	{
	
		core::Size struct_time = (core::Size)(*jt)->get_energy("ltime");
		if( (int(current_time) - int(struct_time)) > (int)library_expiry_time_ ){
			core::Size ssid = (core::Size)(*jt)->get_energy("ssid");
			TR << "Structure: " << ssid << " LTime:  is expired: " << int(current_time) - int(struct_time) << std::endl;

			// send the expired structure to the emperor (who will in due time send back a new one)
			WorkUnit_SilentStructStoreOP getnewstruct = new WorkUnit_SilentStructStore( );
			getnewstruct->set_wu_type( "getnewstruct" );
			getnewstruct->decoys().add( (*jt) );	
			send_MPI_workunit( getnewstruct, 0 );
			TR << "Reported expired structure to emperor: - waiting for new structure" << std::endl;
			receive_MPI_workunit( 0 ); //receive the reply from master and add it to the normal inbound queue
			(*jt)->add_energy("ltime", time(NULL) );	
		}

	}
}

/// This is a virtual over load of the base class MPI_LoopHashRefine:: add_structure_to_library with an extra behavioural step
/// that reports any successful library add-ons to the emperor. This behaviour is master specific and thus should not be in the base class.

bool 
MPI_LoopHashRefine_Master::add_structure_to_library( core::io::silent::ProteinSilentStruct &pss ){
	bool result = MPI_LoopHashRefine::add_structure_to_library( pss );
	TR.Debug << "MPI_LoopHashRefine_Master::add_structure_to_library: " << std::endl;
	if(result) report_structure_to_emperor( pss );
	return result;
}

void
MPI_LoopHashRefine_Master::report_structure_to_emperor(  core::io::silent::SilentStructOP &ss ) {
	WorkUnit_SilentStructStoreOP resultpack = new WorkUnit_SilentStructStore( );
	resultpack->set_wu_type( "resultpack" );
	resultpack->decoys().add( ss );	
	send_MPI_workunit( resultpack, my_emperor() );
	TR << "Reported structure to emperor: " << format_silent_struct( ss ) << std::endl;
}

void
MPI_LoopHashRefine_Master::report_structure_to_emperor(  core::io::silent::ProteinSilentStruct &pss ) {
	WorkUnit_SilentStructStoreOP resultpack = new WorkUnit_SilentStructStore( );
	resultpack->set_wu_type( "resultpack" );
	resultpack->decoys().add( pss );	
	send_MPI_workunit( resultpack, my_emperor() );
	TR << "Reported structure to emperor: " << pss.get_energy("score") << std::endl;
}











} // namespace loophash
} // namespace protocols


