// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/IndependentLoopMover.cc
/// @brief  loop mover base class
/// @author Mike Tyka

#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/Loops.hh>
#include <protocols/loops/loops_main.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh> // tracer output

#include <core/fragment/FragSet.hh>

#include <core/scoring/ScoreFunction.hh>

#include <protocols/checkpoint/CheckPointer.hh>

//Utility Headers

/// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <iostream>
#include <map>
#include <string>
#if defined(WIN32) || defined(__CYGWIN__)
	#include <ctime>
#endif
// option key includes
#include <core/options/keys/loops.OptionKeys.gen.hh>

#include <core/options/option.hh>
#include <numeric/random/random.fwd.hh>
#include <ObjexxFCL/format.hh>




namespace protocols {
namespace loops {

///////////////////////////////////////////////////////////////////////////////
using namespace core;
using namespace ObjexxFCL;
using namespace ObjexxFCL::fmt;

core::util::Tracer tr("protocol.loops.LoopMover");


//////////////////////////////////////////////////////////////////////////////////
/// @details  Set a loop to extended torsion angles.
void LoopMover::set_extended_torsions(
	core::pose::Pose & pose,
	Loop const & loop
)
{
	Real const init_phi  ( -150.0 );
	Real const init_psi  (  150.0 );
	Real const init_omega(  180.0 );

	static int counter = 0;

	tr.Debug << "Extending loop torsions" << loop.start() << " " << loop.stop()
		<< std::endl;

	idealize_loop(pose, loop );

	Size start_extended = std::max((Size)1,loop.start());
	Size end_extended   = std::min(pose.total_residue(),loop.stop());
	for ( Size i = start_extended; i <= end_extended; ++i ) {
		if ( i != start_extended )	pose.set_phi( i, init_phi );
		if ( i != end_extended ) pose.set_psi( i, init_psi );
		if ( ( i != start_extended ) && ( i != end_extended ) ) pose.set_omega( i, init_omega );
	}

	counter++;
}

std::string
LoopMover::get_name() const {
	return "LoopMover";
}

void LoopMover::add_fragments(
	core::fragment::FragSetOP fragset
) {
	if ( fragset->size() > 0 ) {
		frag_libs_.push_back( fragset->clone() );
	}
}

void LoopMover::clear_fragments() {
	frag_libs_.clear();
}

} // namespace loops
} // namespace protocols
