// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/motifs/MotifLibrary.cc
/// @brief Implementation of interaction motifs
/// @author havranek, sthyme (sthyme@gmail.com)

// Unit Headers
#include <protocols/motifs/MotifLibrary.hh>

// Package Headers
#include <protocols/motifs/motif_utils.hh>
#include <protocols/motifs/Motif.hh>
#include <protocols/motifs/SingleMotif.hh>

// Project Headers

// Utility Headers
#include <utility/file/file_sys_util.hh>
#include <utility/file/FileName.hh>

// C++ Headers
#include <fstream>
#include <iostream>

namespace protocols {
namespace motifs {

MotifLibrary::MotifLibrary()
{}

MotifLibrary::~MotifLibrary()
{}

MotifLibrary::MotifLibrary(
	FileNames & motif_filenames
)
{
	for ( FileNames::const_iterator filename( motif_filenames.begin() );
		filename != motif_filenames.end(); ++filename ) {
		if ( !utility::file::file_exists( *filename ) ) {
			continue;
		}
		SingleMotifOP new_motif = single_motif_from_filename( *filename );
		add_to_library( *new_motif );
	}
}

MotifLibrary::MotifLibrary(
	std::istream & motif_info
)
{
	std::string key_in;
	while( motif_info >> key_in ) {
		if( key_in == "SINGLE" ) {
			SingleMotifOP new_motif = single_motif_from_stream( motif_info );
			add_to_library( *new_motif );
		} else {
			std::cout << "ERROR - BAD MOTIF KEY " << key_in << "\n";
		}
	}
}

void
MotifLibrary::add_to_library( Motif const & add_me )
{
	library_.push_back( add_me.clone() );
}

core::Size
MotifLibrary::nmotifs()
{
	return library_.size();
}

void
MotifLibrary::add_from_file( std::string const & motif_filename )
{
	// Try to open the file
	std::ifstream motif_file;
	motif_file.open( motif_filename.c_str() );
	if( !motif_file ) {
		std::cout << "ERROR:  No motif file " << motif_filename << " - FAILING!\n";
		return;
	}

	// Attempt to read in motifs until exhausted
	MotifLibrary new_library( motif_file );

	// Add to this library
	for( MotifCOPs::const_iterator pmot = new_library.begin() ; pmot != new_library.end() ; ++pmot ) {
		add_to_library( **pmot );
	}
}

std::ostream & operator <<(
	std::ostream & os, MotifLibrary & mlib
)
{
	for( MotifCOPs::const_iterator pmot = mlib.begin() ; pmot != mlib.end() ; ++pmot ) {
		os << (*pmot)->print();
	}
	return os;
}

} // namespace motifs
} // namespace protocols
