// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file ConformerSwitchMover.cc
/// @brief code for the conformer switch mover in ensemble docking
/// @author Sid Chaudhury

#ifndef INCLUDED_protocols_moves_ConformerSwitchMover_HH
#define INCLUDED_protocols_moves_ConformerSwitchMover_HH

// Unit headers
#include <protocols/moves/ConformerSwitchMover.fwd.hh>

// Package headers
#include <protocols/moves/Mover.hh>

#include <core/types.hh>

#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
// AUTO-REMOVED #include <core/scoring/ScoreFunctionFactory.hh>

//Option key includes
// AUTO-REMOVED #include <core/options/keys/docking.OptionKeys.gen.hh>

// ObjexxFCL Headers

// C++ Headers
#include <map>
#include <string>

// Utility Headers
#include <utility/pointer/ReferenceCount.hh>

//Auto Headers
#include <utility/options/keys/BooleanOptionKey.hh>


namespace protocols {
namespace moves {

/// @brief this mover does the conformer swap in RosettaDock's ensemble docking.  It takes
/// in a multi-model PDB file as an ensemble, and does swaps conformers by superpositioning
/// over interface residues, and selects a conformer based on a partition function using
/// a ScoreFunction.
class ConformerSwitchMover : public Mover {

public:

	//default constructor
	ConformerSwitchMover() :
		Mover(),
		start_res_(1),
		end_res_(1),
		conf_size_(0),
		jump_id_(1),
		ensemble_size_(0),
		temperature_(0.8),
		fullatom_( false ),
		partition_switch_ ( true )
	{
		Mover::type( "ConformerSwitchMover" );
	}

	//constructor with arguments
	ConformerSwitchMover(
		Size start_res,
		Size end_res,
		Size jump_id,
		core::scoring::ScoreFunctionCOP score_fxn,
		std::string ensemble_file_path
		);

	virtual void apply( core::pose::Pose & pose );
	virtual std::string get_name() const;
	void set_temperature( core::Real temp_in );
	void set_fullatom( bool fullatom_in );
	void set_partition_switch( bool partition_switch_in );
	core::Size get_current_confnum() { return conf_num_; }

	// @brief recover sidechains for a conformer
	void recover_conformer_sidechains( core::pose::Pose & pose );

private:
	void load_ensemble( std::string ensemble_file_path );

	void GenerateProbTable( core::pose::Pose & pose );

	void SwitchConformer(
		core::pose::Pose & pose,
		core::Size conf_num);

	bool docking_lowres_filter( core::pose::Pose & pose );

	Size start_res_, end_res_, conf_size_, jump_id_, ensemble_size_, conf_num_;
	core::scoring::ScoreFunctionCOP score_fxn_;
	core::Real temperature_;
	bool fullatom_, partition_switch_;
	utility::vector1< core::pose::Pose > ensemble_;
	utility::vector1< core::Real > prob_table_;
}; //mover

} // moves
} // rosetta


#endif
