// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/moves/ReportToDB.cc
///
/// @brief  report all data to a database
/// @author Matthew O'Meara

#include <protocols/moves/ReportToDB.hh>
#include <string>


// Setup Mover
#include <protocols/moves/ReportToDBCreator.hh>
namespace protocols{
namespace moves{

std::string
ReportToDBCreator::keyname() const
{
	return ReportToDBCreator::mover_name();
}

MoverOP
ReportToDBCreator::create_mover() const {
	return new ReportToDB;
}

std::string
ReportToDBCreator::mover_name()
{
	return "ReportToDB";
}

}
}


#ifdef DB_SQLITE3
// enable with './scons.py extras=sqlite'

#include <core/chemical/AtomType.hh>
#include <core/chemical/AA.hh>
#include <core/conformation/Residue.hh>
#include <core/id/AtomID_Map.fwd.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/options/keys/parser.OptionKeys.gen.hh>
#include <core/options/util.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/scoring/geometric_solvation/ExactOccludedHbondSolEnergy.hh>
#include <core/scoring/EnergyMap.fwd.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/etable/EtableEnergy.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/hbonds/HBondSet.hh>
#include <core/scoring/hbonds/hbonds_geom.hh>
#include <core/scoring/hbonds/HBondTypeManager.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/scoring/NV/NVscore.hh>
#include <core/scoring/sasa.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreTypeManager.hh>
#include <core/scoring/TenANeighborGraph.hh>
#include <core/scoring/TwelveANeighborGraph.hh>
#include <core/svn_version.hh>
#include <core/types.hh>
#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/CacheableString.hh>
#include <core/util/Tracer.hh>
#include <protocols/jd2/Job.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/DataMap.hh>
#include <protocols/moves/Mover.hh>

// utility headers
#include <utility/file/FileName.hh>
#include <utility/file/FileName.fwd.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/sql_database/sqlite3_interface.hh>
#include <utility/options/keys/StringOptionKey.hh>
#include <utility/options/keys/BooleanOptionKey.hh>
#include <utility/Tag/Tag.hh>
#include <utility/vector1.hh>


//
#include <sqlite3.h>

// c++ headers
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>


// namespaces
using namespace std;
using namespace boost;
using namespace core;
	using namespace conformation;
	using namespace kinematics;
	using namespace chemical;
	using namespace id;
	using namespace pose;
	using namespace protocols::jd2;
	using namespace scoring;
		using namespace etable;
		using namespace hbonds;
using namespace utility::sql_database;

using core::util::T;
using core::util::Error;
using core::util::Warning;
using utility::file::FileName;
using utility::vector1;

util::Tracer TR("apps.pilot.momeara.ReportToDB");


namespace protocols{
namespace moves{

Size ReportToDB::protocol_id = 0;


ReportToDB::ReportToDB():
	Mover("ReportToDB"),
	database_fname("FeatureStatistics.sl3"),
	schema_fname("FatureStatistics_schema.sql"),
	sample_source("Rosetta: Unknown Protocol"),
	scfxn( ScoreFunctionFactory::create_score_function( STANDARD_WTS ) ),
	geo_sol_energy( ExactOccludedHbondSolEnergy() ),
	emap_temp_( EnergyMap() ),
	initialized( false ),
	db_interface()
{}

ReportToDB::ReportToDB(string const & name):
	Mover(name),
	database_fname("FeatureStatistics.sl3"),
	schema_fname("FatureStatistics_schema.sql"),
	sample_source("Rosetta: Unknown Protocol"),
	scfxn( ScoreFunctionFactory::create_score_function( STANDARD_WTS ) ),
	geo_sol_energy( ExactOccludedHbondSolEnergy() ),
	emap_temp_( EnergyMap() ),
	initialized( false ),
	db_interface()
{}

ReportToDB::ReportToDB(
	string const & name,
	string const & database_fname,
	string const & schema_fname,
	string const & sample_source,
	ScoreFunctionOP scfxn ) :
	Mover(name),
	database_fname(database_fname),
	schema_fname(schema_fname),
	sample_source(sample_source),
	scfxn(scfxn),
	geo_sol_energy( ExactOccludedHbondSolEnergy() ),
	emap_temp_( EnergyMap() ),
	initialized( false ),
	db_interface(database_fname)
{}

ReportToDB::ReportToDB( ReportToDB const & src):
	Mover(src),
	database_fname(src.database_fname),
	schema_fname(src.schema_fname),
	sample_source(src.sample_source),
	scfxn(new ScoreFunction(* src.scfxn)),
	emap_temp_(EnergyMap(src.emap_temp_)),
	initialized(src.initialized),
	db_interface(src.db_interface)
{}



ReportToDB::~ReportToDB(){
}

MoverOP ReportToDB::fresh_instance() const { return new ReportToDB; }

MoverOP ReportToDB::clone() const
{
	return new ReportToDB( *this );
}

// So this this can be called from RosettaScripts
void
ReportToDB::parse_my_tag(
	utility::Tag::TagPtr const tag,
	DataMap & /*data*/,
	Filters_map const & /*filters*/,
	Movers_map const & /*movers*/,
	Pose const & /*pose*/ )
{
	if( tag->hasOption("db") ){
		database_fname = tag->getOption<string>("db");
	} else {
		TR << "Field 'db' required for use of ReportToDB mover in Rosetta Scripts." << endl;
	}
	if( tag->hasOption("schema") ){
		schema_fname = tag->getOption<string>("schema");
	} else {
		TR << "Field 'schema' required for use of ReportToDB in Rosetta Scripts." << endl;
	}
	if( tag->hasOption("sample_source") ){
		sample_source = tag->getOption<string>("sample_source");
	} else {
		TR << "Field 'sample_source' required for use of ReportToDB in Rosetta Scripts." << endl;
		TR << "The sample_source should be a meaniful short string to identify where the samples came from." << endl;
		TR << "For example: 'Rosetta r34521' or 'Top5200 Dataset'." << endl;
	}
}

// Argh why isn't there a good way go get the name of a pose?
string
ReportToDB::pose_name(
	Pose& pose)
{
	//silent files and pdbs set the name of the pose differently
	string name = "No_Name_Found";
	if (pose.pdb_info()){
		name = pose.pdb_info()->name();
	} else if ( pose.data().has( datacache::CacheableDataType::JOBDIST_OUTPUT_TAG ) ) {
		name = static_cast< util::datacache::CacheableString const & >
			( pose.data().get( datacache::CacheableDataType::JOBDIST_OUTPUT_TAG ) ).str();
	} else {
		name = JobDistributor::get_instance()->current_job()->input_tag();
	}
	return name;
}


void
ReportToDB::initialize_database()
{
	if (initialized){
		return;
	} else {

		create_tables();

		if(!protocol_id){
			protocol_id = insert_protocol_row();
		}
		initialized = true;
	}
}

void
ReportToDB::create_tables()
{
	utility::io::izstream schemaf(schema_fname.c_str());
	if( !schemaf){
		TR.Error << "Unable to upen sql schema '" << schema_fname << "' for database '" << database_fname << "'" << endl;
		utility_exit();
	} else {
		TR << "Opened schema '" << schema_fname << "' for database '" << database_fname << "'" << endl;
	}
	stringstream schema;
	string line;
	while ( getline(schemaf, line)){
		schema << line << "\n";
	}
	schemaf.close();
	db_interface.execute_sql(schema.str());
}

Size
ReportToDB::insert_protocol_row()
{

	stringstream option_stream;
	option_stream << options::option;

	db_interface << begin_row("protocols");
	db_interface << Sqlite3Interface::sqlite3_null;
	db_interface << options::option.get_argv();
	db_interface << option_stream.str();
	db_interface << minirosetta_svn_url();
	db_interface << minirosetta_svn_version();
	if (options::option[ options::OptionKeys::parser::protocol ].active() ){
		string const script_fname( options::option[ options::OptionKeys::parser::protocol ] );
		stringstream script;
		script << utility::io::izstream( script_fname.c_str() ).rdbuf();
		db_interface << script.str();
	} else {
		db_interface << Sqlite3Interface::sqlite3_null;
	}
	db_interface << Sqlite3Interface::end_row;

	return db_interface.last_key();
}

Size
ReportToDB::insert_structure_row(
	Pose const & pose,
	bool const & isNative,
	string const & resolution,
	string const & refinementProgram)
{

	//Do not use by default for backwards compatibiliy
	//TR.Debug << "Memory allocated but not freed by Sqlite3 interface: " << db_interface.memory_used() << std::endl;

	db_interface << begin_row("structures");
	db_interface << Sqlite3Interface::sqlite3_null;
	db_interface << protocol_id;

	//filename -> should be using the job distributor!
	FileName fileName_full = JobDistributor::get_instance()->current_job()->input_tag();
	string filename = fileName_full.to_local_name();
	db_interface << filename;


	//tag
	if( pose.data().has( datacache::CacheableDataType::JOBDIST_OUTPUT_TAG ) ) {
		string jobdist_output_tag =	static_cast<util::datacache::CacheableString const & >
			(pose.data().get( datacache::CacheableDataType::JOBDIST_OUTPUT_TAG)).str();
		if( jobdist_output_tag != "EMPTY_JOB_use_jd2" ){
			db_interface << jobdist_output_tag;
		} else {
			// not running ReportToDB under the JobDistributor
			if(pose.pdb_info()){
				db_interface << pose.pdb_info()->name();
			} else {
				db_interface << Sqlite3Interface::sqlite3_null;
			}
		}
	} else {
		if(pose.pdb_info()){
			db_interface << pose.pdb_info()->name();
		} else {
			db_interface << Sqlite3Interface::sqlite3_null;
		}
	}
	db_interface << pose.total_residue();
	db_interface << pose.sequence();
	db_interface << isNative;
	db_interface << resolution;
	db_interface << refinementProgram;
	db_interface << pose.energies().total_energy();
	db_interface << Sqlite3Interface::end_row;

	return db_interface.last_key();
}

void
ReportToDB::insert_structure_score_rows(
	Pose & pose,
	ScoreFunctionCOP scfxn,
	Size const struct_id)
{
	(*scfxn)(pose);
	EnergyMap weights(scfxn->weights());
	EnergyMap energies(pose.energies().total_energies());
	for(Size i=1; i <= n_score_types; ++i){
		ScoreType type(static_cast<ScoreType>(i));
		if( weights[type] == 0 ) continue;
		db_interface << begin_row("structure_scores");
		db_interface << struct_id;
		db_interface << ScoreTypeManager::name_from_score_type(type);
		db_interface << energies[type];
		db_interface << Sqlite3Interface::end_row;
	}
}


///@details
///
/// * Score types are either one body, two body or whole structure and
/// can either dependend on the context or not.
///
/// * Whole structure terms are reported in the structure_scores table
/// along with the totals from the remaining terms.
///
/// * The one and two body terms are broken up into two different
/// tables because they are parametrized differently (one residue vs
/// two residues).
///
/// * Residues are identified by Rosetta's residue numbering scheme.
/// To convert to the PDB residue numbering scheme, join with the
/// residues_pdb table.
///
/// * Although two body terms can be with in the same residue, these
/// 'intrares' score terms are reported with the two body terms where
/// resNum == otherResNum.
///
/// * Two body terms always are reported so that resNum1 <= resNum2.
///
/// * Values for score terms are only reported if they are non-zero.

void
ReportToDB::insert_residue_scores_rows(
	Pose & pose,
	ScoreFunctionOP scfxn,
	Size const struct_id)
{

	scfxn->setup_for_scoring( pose );

	ScoreTypes ci_1b( scfxn->ci_1b_types() );
	ScoreTypes cd_1b( scfxn->cd_1b_types() );
	ScoreTypes ci_2b( scfxn->ci_2b_types() );
	ScoreTypes cd_2b( scfxn->cd_2b_types() );


	for(Size resNum=1; resNum <= pose.total_residue(); ++resNum){
		Residue rsd( pose.residue(resNum) );
		{ // Context Independent One Body Energies
			EnergyMap emap;
			scfxn->eval_ci_1b(rsd, pose, emap);
			for(ScoreTypes::const_iterator st = ci_1b.begin(), ste = ci_1b.end(); st != ste; ++st){
				if(!emap[*st]) continue;
				db_interface << begin_row("residue_scores_1b")
					<< struct_id
					<< resNum
					<< ScoreTypeManager::name_from_score_type(*st)
					<< emap[*st]
					<< "context_independent"
					<< Sqlite3Interface::end_row;
			}
		}
		{ // Context Dependent One Body Energies
			EnergyMap emap;
			scfxn->eval_cd_1b(rsd, pose, emap);
			for(ScoreTypes::const_iterator st = cd_1b.begin(), ste = cd_1b.end(); st != ste; ++st){
				if(!emap[*st]) continue;
				db_interface << begin_row("residue_scores_1b")
					<< struct_id
					<< resNum
					<< ScoreTypeManager::name_from_score_type(*st)
					<< emap[*st]
					<< "context_dependent"
					<< Sqlite3Interface::end_row;
			}
		}

		// Two Body Energies
		for(Size otherResNum=resNum+1; otherResNum <= pose.total_residue(); ++otherResNum){
			if(!scfxn->are_they_neighbors(pose, resNum, otherResNum)) continue;
			Residue otherRsd( pose.residue(otherResNum) );
			{ // Context Independent Two Body Energies
				EnergyMap emap;
				scfxn->eval_ci_2b(rsd, otherRsd, pose, emap);
				for(ScoreTypes::const_iterator st = ci_2b.begin(), ste = ci_2b.end(); st != ste; ++st){
					if(!emap[*st]) continue;
					db_interface << begin_row("residue_scores_2b")
						<< struct_id
						<< resNum
						<< otherResNum
						<< ScoreTypeManager::name_from_score_type(*st)
						<< emap[*st]
						<< "context_independent"
						<< Sqlite3Interface::end_row;
				}
			}
			{ // Context Dependent Two Body Energies
				EnergyMap emap;
				scfxn->eval_cd_2b(rsd, otherRsd, pose, emap);
				for(ScoreTypes::const_iterator st = cd_2b.begin(), ste = cd_2b.end(); st != ste; ++st){
					if(!emap[*st]) continue;
					db_interface << begin_row("residue_scores_2b")
						<< struct_id
						<< resNum
						<< otherResNum
						<< ScoreTypeManager::name_from_score_type(*st)
						<< emap[*st]
						<< "context_dependent"
						<< Sqlite3Interface::end_row;
				}
			}
		} // End Two Body Energies
	} // End res1 for loop
} // End function body

void
ReportToDB::insert_residues_pdb_rows (
	Pose const & pose,
	Size struct_id) {

	if(!pose.pdb_info()){
		//eg if this is a silent file structure

		for(Size resNum=1; resNum <= pose.total_residue(); ++resNum){
			db_interface << begin_row("residues_pdb")
				<< struct_id
				<< resNum
				<< pose.chain(resNum)
				<< resNum
				<< Sqlite3Interface::sqlite3_null
				<< Sqlite3Interface::sqlite3_null
				// should the occupancy be here as well?
				<< Sqlite3Interface::end_row;
		}
	} else {
		for(Size resNum=1; resNum <= pose.total_residue(); ++resNum){
			db_interface << begin_row("residues_pdb")
				<< struct_id
				<< resNum
				<< pose.pdb_info()->chain(resNum)
				<< pose.pdb_info()->number(resNum)
				<< pose.pdb_info()->icode(resNum)
				// should the occupancy be here as well?
				<< Sqlite3Interface::end_row;
		}
	}
}


Size
ReportToDB::insert_site_row(
	Pose const & pose,
	Size const & struct_id,
	Size const & resNum,
	Size const & atmNum,
	bool const & is_donor)
{
	string hb_chem_type;
	if (is_donor){
		HBDonChemType hb_don_chem_type = get_hb_don_chem_type( atmNum, pose.residue(resNum) ); // this is the donor atom
		hb_chem_type = HBondTypeManager::name_from_don_chem_type(hb_don_chem_type);
	} else {
		HBAccChemType hb_acc_chem_type = get_hb_acc_chem_type( atmNum, pose.residue(resNum) );
		hb_chem_type = HBondTypeManager::name_from_acc_chem_type(hb_acc_chem_type);
	}

	db_interface << begin_row("sites");
	db_interface << Sqlite3Interface::sqlite3_null;
	db_interface << struct_id;
	db_interface << resNum;
	db_interface << atmNum;
	db_interface << is_donor;
	db_interface << pose.chain(resNum);
	db_interface << pose.residue_type(resNum).name();
	db_interface << pose.residue(resNum).atom_type(atmNum).name();
	db_interface << hb_chem_type;
	db_interface << Sqlite3Interface::end_row;

	return db_interface.last_key();
}

void
ReportToDB::insert_site_pdb_row(
	Pose const & pose,
	Size const & resNum,
	Size const & atmNum,
	Size const & site_id)
{
	if(!pose.pdb_info()) return; //eg if this is a silent file structure

	db_interface << begin_row("sites_pdb");
	db_interface << site_id;
	db_interface << pose.pdb_info()->chain(resNum);
	db_interface << pose.pdb_info()->number(resNum);
	db_interface << pose.pdb_info()->icode(resNum);
	db_interface << pose.pdb_info()->temperature(resNum,atmNum);
	// should the occupancy be here as well?
	db_interface << Sqlite3Interface::end_row;
}


void
ReportToDB::insert_site_environment_row(
	Pose const & pose,
	Size const & resNum,
	Size const & atmNum,
	Size const & site_id,
	id::AtomID_Map< vector1<HBondCOP> > const & site_partners,
	id::AtomID_Map< Real > const & site_hbond_energies)
{
	Residue rsd( pose.residue(resNum) );

	Size num_nbrs = 1; // NOTE: Current hydrogen bond nbr_count counts from 1!!
	TenANeighborGraph const & tenA_neighbor_graph( pose.energies().tenA_neighbor_graph() );
	for ( graph::Graph::EdgeListConstIter
		ir = tenA_neighbor_graph.get_node( resNum )->const_edge_list_begin(),
		ire = tenA_neighbor_graph.get_node( resNum )->const_edge_list_end();
		ir != ire; ++ir ) {
		++num_nbrs;
	}

	string secstruct(1,pose.secstruct(resNum));

	//ignored for now because it is slow to compute
	//Real geometric_solvation( geo_sol_energy.compute_polar_group_sol_energy(pose, resNum, atmNum));

	db_interface << begin_row("site_environment");
	db_interface << site_id;
	db_interface << pose.secstruct(resNum);
	db_interface << num_nbrs;
	db_interface << Sqlite3Interface::sqlite3_null; // weighted_num_nbrs;
	db_interface << Sqlite3Interface::sqlite3_null; // emap_temp_[neigh_vect_raw] //nbr_vector
	db_interface << Sqlite3Interface::sqlite3_null; // sasa
	db_interface << Sqlite3Interface::sqlite3_null; // geometric_solvation
	db_interface << site_hbond_energies(resNum, atmNum);
	db_interface << site_partners(resNum,atmNum).size();
	db_interface << Sqlite3Interface::sqlite3_null; // gb_radius
	db_interface << Sqlite3Interface::end_row;
}

void
ReportToDB::insert_site_atoms_row(
	Pose const & pose,
	Size const & resNum,
	Size const & atmNum,
	Size const & site_id)
{
	Residue rsd( pose.residue(resNum) );

	db_interface << begin_row("site_atoms");
	db_interface << site_id;
	db_interface << rsd.atom(atmNum).xyz().x();
	db_interface << rsd.atom(atmNum).xyz().y();
	db_interface << rsd.atom(atmNum).xyz().z();
	db_interface << rsd.atom(rsd.atom_base(atmNum)).xyz().x();
	db_interface << rsd.atom(rsd.atom_base(atmNum)).xyz().y();
	db_interface << rsd.atom(rsd.atom_base(atmNum)).xyz().z();
	db_interface << rsd.atom(rsd.atom_base(rsd.atom_base(atmNum))).xyz().x();
	db_interface << rsd.atom(rsd.atom_base(rsd.atom_base(atmNum))).xyz().y();
	db_interface << rsd.atom(rsd.atom_base(rsd.atom_base(atmNum))).xyz().z();
	if (rsd.abase2(atmNum)){
		db_interface << rsd.atom(rsd.abase2(atmNum)).xyz().x();
		db_interface << rsd.atom(rsd.abase2(atmNum)).xyz().y();
		db_interface << rsd.atom(rsd.abase2(atmNum)).xyz().z();
	} else {
		db_interface << Sqlite3Interface::sqlite3_null;
		db_interface << Sqlite3Interface::sqlite3_null;
		db_interface << Sqlite3Interface::sqlite3_null;
	}
	db_interface << Sqlite3Interface::end_row;
}

Size
ReportToDB::insert_hbond_row(
	HBond const & hbond,
	Size const & struct_id,
	AtomID_Map< Size > const & site_ids,                      // This is for ranking hbonds
	AtomID_Map< vector1<HBondCOP> > const & site_partners)    //    "             "
{
	bool found_don_partner( false ); // for debug

	//Zero if unique
	//i in 1 through n if ith lowest energy hbond made with this site
	Size donRank=0, accRank=0;
	vector1< HBondCOP > don_partners( site_partners(hbond.don_res(), hbond.don_hatm()));
	if (don_partners.size() > 1){
		donRank++;
	}
	for( vector1< HBondCOP >::const_iterator i = don_partners.begin(), i_end = don_partners.end();
		i != i_end; ++i){
		HBond const & candidate_hbond( **i);
		if(hbond == candidate_hbond){
			found_don_partner = true;
			break;
		} else {
			++donRank;
		}
	}
	assert(found_don_partner);



	bool found_acc_partner(false); //for debug
	vector1< HBondCOP > acc_partners( site_partners(hbond.acc_res(), hbond.acc_atm()));

	if (acc_partners.size() > 1){
		accRank++;
	}

	for( vector1< HBondCOP >::const_iterator i = acc_partners.begin(), i_end = acc_partners.end();
		i != i_end; ++i){
		HBond const & candidate_hbond( **i);
		if(hbond == candidate_hbond){
			found_acc_partner = true;
			break;
		} else {
			++accRank;
		}
	}
	assert(found_acc_partner);

	db_interface << begin_row("hbonds");
	db_interface << Sqlite3Interface::sqlite3_null;
	db_interface << site_ids(hbond.don_res(), hbond.don_hatm());
	db_interface << site_ids(hbond.acc_res(), hbond.acc_atm());
	db_interface << struct_id;
	db_interface << hbond.eval_type();
	db_interface << hbond.energy();
	db_interface << hbond.weight();
	db_interface << donRank;
	db_interface << accRank;
	db_interface << Sqlite3Interface::end_row;
	return db_interface.last_key();
}

void
ReportToDB::insert_hbond_geom_coords(
	Pose const & pose,
	HBond const & hbond,
	Size const & hbond_id)
{
	const Residue acc_res( pose.residue(hbond.acc_res()));
	const Vector Axyz(acc_res.atom(hbond.acc_atm()).xyz());
	const Vector Bxyz(acc_res.atom(acc_res.atom_base(hbond.acc_atm())).xyz());
	const Vector B2xyz(acc_res.atom(acc_res.abase2(hbond.acc_atm())).xyz());
	const Residue don_res( pose.residue(hbond.don_res()));
	const Vector Hxyz(don_res.atom(hbond.don_hatm()).xyz());
	const Vector Dxyz(don_res.atom(don_res.atom_base(hbond.don_hatm())).xyz());


	// Paraphrase hb_energy_deriv (when called with atomic coordinates):
	Vector HDunit( Dxyz - Hxyz );
	Real const HDdist2( HDunit.length_squared() );
	Real const inv_HDdist = 1.0f / std::sqrt( HDdist2 );
	HDunit *= inv_HDdist;
	Vector BAunit;
	Vector PBxyz; // pseudo base atom
	chemical::Hybridization acc_hybrid( get_hbe_acc_hybrid( hbond.eval_type() ) );
	make_hbBasetoAcc_unitvector(acc_hybrid, Axyz, Bxyz, B2xyz, PBxyz, BAunit);

	// Paraphrase hb_energy_deriv (when called with coords/vectors)
	Vector AH = Hxyz - Axyz;
	Real const AHdist2( AH.length_squared() );
	Real const AHdist = std::sqrt(AHdist2);
	Real const inv_AHdist = 1.0f / AHdist;
	Vector AHunit = AH * inv_AHdist;

	Real const cosAHD = dot( AHunit, HDunit );
	Real const cosBAH = dot( BAunit, AHunit );

	// Paraphrase Ningjin's work
	static Vector v(B2xyz - Bxyz);
	const float vDotBAunit = dot(v, BAunit);
	v = (v - vDotBAunit * BAunit)/sqrt(v.length_squared());
	static Vector AHperpBA( AH - dot(AH, BAunit) * BAunit);
	const float AHperpBAlen = sqrt(AHperpBA.length_squared());
	AHperpBA = AHperpBA / AHperpBAlen;
	const float cosChi = dot(AHperpBA, v);
	const float sinChi = dot(AHperpBA, cross(BAunit, v));
	float chi( sinChi >= 0 ? acos(cosChi) : -acos(cosChi));

	db_interface << begin_row("hbond_geom_coords");
	db_interface << hbond_id;
	db_interface << AHdist;
	db_interface << cosBAH;
	db_interface << cosAHD;
	db_interface << chi;
	db_interface << Sqlite3Interface::end_row;
}

// right now this just reports the lennard jones values for hbond atoms
void
ReportToDB::insert_hbond_lennard_jones_row(
	Pose const & pose,
	HBond const & hbond,
	Size const & hbond_id)
{

	Residue don_res(pose.residue(hbond.don_res()));
	Residue acc_res(pose.residue(hbond.acc_res()));
	Size don_hatmNum(hbond.don_hatm());
	Size acc_atmNum(hbond.acc_atm());
	Size don_datmNum(don_res.atom_base(don_hatmNum));
	Size acc_batmNum(acc_res.atom_base(acc_atmNum));

	EtableEnergy const etable_energy
		( *ScoringManager::get_instance()->etable( scfxn->energy_method_options().etable_type() ),
			scfxn->energy_method_options() );

	Real bb_dummy, dsq_dummy;

	Real don_acc_atrE, don_acc_repE, don_acc_solv;
	etable_energy.atom_pair_energy( don_res.atom(don_datmNum), acc_res.atom(acc_atmNum), /*weight*/ 1,
																	don_acc_atrE, don_acc_repE, don_acc_solv, bb_dummy, dsq_dummy );
	don_acc_atrE *= (*scfxn)[ fa_atr ];
	don_acc_repE *= (*scfxn)[ fa_rep ];
	don_acc_solv *= (*scfxn)[ fa_sol ];

	Real don_acc_base_atrE, don_acc_base_repE, don_acc_base_solv;
	etable_energy.atom_pair_energy( don_res.atom(don_datmNum), acc_res.atom(acc_batmNum), /*weight*/ 1,
																	don_acc_base_atrE, don_acc_base_repE, don_acc_base_solv, bb_dummy, dsq_dummy );
	don_acc_base_atrE *= (*scfxn)[ fa_atr ];
	don_acc_base_repE *= (*scfxn)[ fa_rep ];
	don_acc_base_solv *= (*scfxn)[ fa_sol ];

	Real h_acc_atrE, h_acc_repE, h_acc_solv;
	etable_energy.atom_pair_energy( don_res.atom(don_hatmNum), acc_res.atom(acc_atmNum), /*weight*/ 1,
																	h_acc_atrE, h_acc_repE, h_acc_solv, bb_dummy, dsq_dummy );
	h_acc_atrE *= (*scfxn)[ fa_atr ];
	h_acc_repE *= (*scfxn)[ fa_rep ];
	h_acc_solv *= (*scfxn)[ fa_sol ];

	Real h_acc_base_atrE, h_acc_base_repE, h_acc_base_solv;
	etable_energy.atom_pair_energy( don_res.atom(don_hatmNum), acc_res.atom(acc_batmNum), /*weight*/ 1,
																	h_acc_base_atrE, h_acc_base_repE, h_acc_base_solv, bb_dummy, dsq_dummy );
	h_acc_base_atrE *= (*scfxn)[ fa_atr ];
	h_acc_base_repE *= (*scfxn)[ fa_rep ];
	h_acc_base_solv *= (*scfxn)[ fa_sol ];

	db_interface << begin_row("hbond_lennard_jones");
	db_interface << hbond_id;
	db_interface << don_acc_atrE;
	db_interface << don_acc_repE;
	db_interface << don_acc_solv;
	db_interface << don_acc_base_atrE;
	db_interface << don_acc_base_repE;
	db_interface << don_acc_base_solv;
	db_interface << h_acc_atrE;
	db_interface << h_acc_repE;
	db_interface << h_acc_solv;
	db_interface << h_acc_base_atrE;
	db_interface << h_acc_base_repE;
	db_interface << h_acc_base_solv;
	db_interface << Sqlite3Interface::end_row;
}

void
ReportToDB::report_hbonds(
	Pose& pose,
	Size const struct_id)
{
	HBondSet hbond_set;
	pose.update_residue_neighbors();
	hbond_set.setup_for_residue_pair_energies( pose, false, false );

	TR << "Number of hydrogen bonds found: " << hbond_set.nhbonds() << endl;

	// compute dssp
	protocols::jumping::Dssp dssp( pose );
	dssp.insert_ss_into_pose( pose );

//  // compute sasa pack
//  Real const probe_radius(1.4);
//  id::AtomID_Map< Real > atom_sasa;
//  vector1< Real > residue_sasa;
//  calc_per_atom_sasa( pose, atom_sasa, residue_sasa, probe_radius);

//	//Set up energy functions for scoring
//	geo_sol_energy.setup_for_scoring(pose, *scfxn);

	AtomID_Map< vector1<HBondCOP> > site_partners;
	id::initialize(site_partners, pose);

	AtomID_Map<Real>site_hbond_energies;
	id::initialize(site_hbond_energies, pose);

	for (Size i = 1; i<= hbond_set.nhbonds(); i++) {
		HBondCOP hbond(hbond_set.hbond(i));

		site_partners(hbond->don_res(),hbond->don_hatm()).push_back(hbond);
		site_hbond_energies(hbond->don_res(),hbond->don_hatm()) += hbond->energy()/2;

		site_partners(hbond->acc_res(),hbond->acc_atm()).push_back(hbond);
		site_hbond_energies(hbond->acc_res(),hbond->acc_atm()) += hbond->energy()/2;
	}

	AtomID_Map< Size > site_ids;
	id::initialize(site_ids, pose);
	for( Size resNum =1; resNum <= pose.n_residue(); ++resNum ){
		Residue const res(pose.residue(resNum));
		// donor sites
		for ( AtomIndices::const_iterator
			atmNum  = res.Hpos_polar().begin(),
			atmNume = res.Hpos_polar().end(); atmNum != atmNume; ++atmNum ) {

			Size batmNum = res.atom_base( *atmNum);
			Size site_id = insert_site_row(pose, struct_id, resNum, batmNum, true /*is donor*/);
			site_ids(resNum,*atmNum) = site_id;
			insert_site_pdb_row(pose, resNum, *atmNum, site_id);
			insert_site_environment_row(pose, resNum, *atmNum, site_id, site_partners, site_hbond_energies);
			insert_site_atoms_row(pose, resNum, *atmNum, site_id);

			vector1< HBondCOP >::iterator partners_begin( site_partners(resNum, *atmNum).begin() );
			vector1< HBondCOP >::iterator partners_end( site_partners(resNum, *atmNum).end() );

			sort(partners_begin, partners_end, HBond::hbond_energy_comparer);

		}
		// acceptor sites
		for( chemical::AtomIndices::const_iterator
			atmNum  = res.accpt_pos().begin(),
			atmNume = res.accpt_pos().end(); atmNum != atmNume; ++atmNum ) {
			Size site_id = insert_site_row(pose, struct_id, resNum, *atmNum, false /*is not donor*/);
			site_ids(resNum,*atmNum) = site_id;
			insert_site_pdb_row(pose, resNum, *atmNum, site_id);
			insert_site_environment_row(pose, resNum, *atmNum, site_id, site_partners, site_hbond_energies);
			insert_site_atoms_row(pose, resNum, *atmNum, site_id);

			sort(site_partners(resNum,*atmNum).begin(),
					site_partners(resNum,*atmNum).end(),
					HBond::hbond_energy_comparer);
		}
	}

	for (Size i = 1; i<= hbond_set.nhbonds(); i++) {
		HBond hbond = hbond_set.hbond( i );

		Size const hbond_id = insert_hbond_row(hbond, struct_id, site_ids, site_partners);
		insert_hbond_geom_coords(pose, hbond, hbond_id);
		insert_hbond_lennard_jones_row(pose, hbond, hbond_id);
	}
}

void
ReportToDB::apply( Pose& pose ){

	// Hack to make sure energy objects are initialized
	(*scfxn)(pose);

	db_interface.open_connection( database_fname );

	// begin the transaction for everything this mover does with the database
	db_interface.begin_transaction();

	initialize_database();

	TR<< "Reporting data for " << pose_name(pose) << std::endl;

	bool const isNative = true; // how to set this? maybe by a flag somewhere?
	string const resolution = "resolution_TBD"; // how to set this?
	string const refinementProgram = "refinement_program_TBD"; // how to set this?
	Size const struct_id = insert_structure_row(pose, isNative, resolution, refinementProgram );
	insert_structure_score_rows(pose, scfxn, struct_id );
	insert_residue_scores_rows(pose, scfxn, struct_id );
	insert_residues_pdb_rows(pose, struct_id );

	report_hbonds(pose, struct_id);

	db_interface.end_transaction();
}

} // moves namespace
} // protocols namespace

#endif // DB_SQLITE3
