// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/toolbox/PoseMetricCalculators/FragQualCalculator.cc
/// @brief calculate rmsd fragment quality given a pose
/// Roughly, fragment quality is number of fragments which are close to a pose in rmsd
/// @detailed
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )

// Unit Headers
#include <protocols/toolbox/PoseMetricCalculators/FragQualCalculator.hh>

// Project Headers
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/util.hh>
#include <core/util/MetricValue.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>
#include <core/scoring/rms_util.hh>
#include <core/fragment/FragSet.hh>
#include <core/fragment/FrameIterator.hh>
#include <core/fragment/FrameIteratorWorker_.hh>
#include <core/fragment/Frame.hh>
#include <protocols/moves/DataMap.hh>

// Utility headers
#include <utility/vector1.hh>
#include <utility/string_util.hh>
#include <utility/Tag/Tag.hh>
#include <utility/exit.hh>

//// C++ headers
#include <cmath>
#include <ObjexxFCL/format.hh>

static core::util::Tracer TR("protocols.toolbox.PoseMetricCalculators.FragQualCalculator");

namespace protocols {
namespace toolbox {
namespace PoseMetricCalculators {


/// @brief default constructor
FragQualCalculator::FragQualCalculator():
	rmsd_cutoff_goodfrag_( 1.0 ),
	ratio_cutoff_goodfrag_( 0.3 ),
	total_goodfrags_( 0 ),
	coverage_( 0 ),
	frag_( NULL ),
	begin_( 0 ),
	end_( 0 )
{
	goodfrags_.clear();
}

/// @brief value constructor
FragQualCalculator::FragQualCalculator(
  FragSetOP const & frag,
	Real const rmsd,
	Real const ratio ):
	rmsd_cutoff_goodfrag_( rmsd ),
	ratio_cutoff_goodfrag_( ratio ),
	total_goodfrags_( 0 ),
	coverage_( 0 ),
	frag_( frag ),
	begin_( 0 ),
	end_( 0 )
{
	goodfrags_.clear();
}

/// @brief copy constructor
FragQualCalculator::FragQualCalculator( FragQualCalculator const & rval ):
	Super(),
	rmsd_cutoff_goodfrag_( rval.rmsd_cutoff_goodfrag_ ),
	ratio_cutoff_goodfrag_( rval.ratio_cutoff_goodfrag_ ),
	total_goodfrags_( rval.total_goodfrags_ ),
	coverage_( rval.coverage_ ),
	goodfrags_( rval.goodfrags_ ),
	frag_( rval.frag_ ),
	begin_( rval.begin_ ),
	end_( rval.end_ )
{}

/// @brief destructor
FragQualCalculator::~FragQualCalculator(){}


/// @brief set fragments
void
FragQualCalculator::set_fragset( FragSetOP const & frag )
{
	frag_ = frag;
}

/// @brief rmsd cutoff of good fragments
void
FragQualCalculator::rmsd_cutoff( Real const & val )
{
	rmsd_cutoff_goodfrag_ = val;
}

/// @brief
void
FragQualCalculator::ratio_cutoff( Real const & val )
{
	ratio_cutoff_goodfrag_ = val;
}

/// @brief
void
FragQualCalculator::set_region( Size const val1, Size const val2 )
{
	begin_ = val1;
	end_ = val2;
}

/// @brief
void
FragQualCalculator::lookup(
  String const & key,
  MetricValueBase * valptr
) const
{
	using namespace core;
	if ( key == "num_goodfrag" ) {
		core::util::check_cast( valptr, &total_goodfrags_, "number of fragments within rmsd cutoff " );
		(static_cast<util::MetricValue<Real> *>(valptr))->set( total_goodfrags_ );
	} else if (key == "coverage" ) {
		core::util::check_cast( valptr, &coverage_, "ratio of the region where good fragments are included more than XXX% " );
		(static_cast<util::MetricValue<Real> *>(valptr))->set( coverage_ );
	} else {
		TR << "FragQualCalculator cannot compute the requested metric " << key << std::endl;
		utility_exit();
	}

} //lookup


// @brief
std::string
FragQualCalculator::print( String const & key ) const
{
	String result;
  if ( key == "num_goodfrag" ) {
    result = utility::to_string( total_goodfrags_ );
  } else if ( key == "coverage" ) {
    result = utility::to_string( coverage_ );
  } else {
		core::util::Error() << "FragQualCalculator cannot compute metric " << key << std::endl;
	}
	return result;
} // apply


/// @brief recomute ncontacts
void
FragQualCalculator::recompute( Pose const & pose )
{
	using ObjexxFCL::fmt::RJ;
	using ObjexxFCL::fmt::F;
	using core::scoring::CA_rmsd;
	using core::fragment::FrameIterator;

	// initialization
	if( begin_ == 0 ) begin_ = 1;
	if( end_ == 0 ) end_ = pose.total_residue();
	total_goodfrags_ = 0;
	coverage_ = 0;

	goodfrags_.resize( pose.total_residue() );
	for( Size i=1; i<=pose.total_residue(); i++ ) {
 		goodfrags_[ i ] = 0;
	}
	utility::vector1< bool > frag_region( pose.total_residue(), false );
	utility::vector1< bool > is_covered( pose.total_residue(), false );

	Pose input_pose( pose ), test_pose( pose );
	core::chemical::switch_to_residue_type_set( input_pose, core::chemical::CENTROID );
	core::chemical::switch_to_residue_type_set(  test_pose, core::chemical::CENTROID );

	for ( FrameIterator frame = frag_->begin(); frame != frag_->end(); ++frame ) {

		Size const start ( frame->start() );
		runtime_assert( start <= pose.total_residue() );

		if( begin_ > start || end_ < start ) continue;
		frag_region[ start ] = true;

		for ( Size i=1; i<=frame->nr_frags(); i++ ) {
			// insert fragment
			frame->apply( i, test_pose );
			// calc rmsd
			Real rmsd = CA_rmsd( input_pose, test_pose, start, start + frame->length() - 1 );
			if( rmsd <= rmsd_cutoff_goodfrag_ ) {
				goodfrags_[ start ] += 1;
			}
			// TR << RJ( 6, frame->length() ) << RJ( 6, frame->start() ) << RJ( 6, i ) << F( 10, 4, rmsd ) << std::endl;
		}

		if( goodfrags_[ start ] >= frame->nr_frags()*ratio_cutoff_goodfrag_ ) {
			is_covered[ start ] = true;
		}
	} // FrameIterator

	// calc coverage
	Size count( 0 );
	for( Size i=1; i<=pose.total_residue(); i++ ) {

		if( frag_region[ i ] ) {
			count ++;
			if( is_covered[ i ] ) {
				coverage_++;
			}
		}
		TR.Debug << i << " " << goodfrags_[ i ] << std::endl;
		total_goodfrags_ += Real( goodfrags_[ i ] );
	}

	coverage_ = coverage_ / count;

}

/// @brief parse xml
void
FragQualCalculator::parse_my_tag(
	TagPtr const tag,
	DataMap & data,
	Filters_map const &,
	Movers_map const &,
	Pose const & pose )
{
 	rmsd_cutoff_goodfrag_ = tag->getOption<Real>( "rmsd_cutoff", 1.0 );
	ratio_cutoff_goodfrag_ = tag->getOption<Real>( "ratio_cutoff", 0.3 );

	begin_ = tag->getOption<Size>( "begin", 1 );
	end_ = tag->getOption<Size>( "end", pose.total_residue() );

	String const fset_string ( tag->getOption<String>( "frag", "" ) );
	runtime_assert( ! fset_string.empty() );
	if ( data.has( "fragsets", fset_string ) ) {
		frag_ = data.get< FragSet* >( "fragsets", fset_string );
	} else {
		utility_exit_with_message("fragsets " + fset_string + " not found in DataMap.");
	}
}


} // filters
} // fldsgn
} // protocols

