// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file apps/pilot/matt/rosetta_design_wizard.cc
/// @brief The Mini interface for the PyMol Wizard
/// @author Matt O'Meara (mattjomeara@gmail.com)

// core headers
#include <core/init.hh>
#include <core/types.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>

//#include <core/io/pdb/file_data.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/io/pdb/pdb_dynamic_reader.hh>

#include <core/pack/pack_rotamers.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/ResfileReader.fwd.hh>
#include <core/pack/task/ResfileReader.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/PDBPoseMap.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/util/Tracer.hh>

#include <rdwizard/bindings/rosetta_design_wizard.hh>

#include <utility/exit.hh>
#include <utility/io/izstream.hh>
#include <utility/string_util.hh>
#include <utility/pointer/owning_ptr.fwd.hh>


// c++ headers
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <sstream>
#include <utility>
#include <vector>


// namespaces
using namespace core;
using namespace conformation;
using namespace chemical;
using namespace scoring;
using namespace pose;
using core::util::T;
using core::util::Error;
using core::util::Warning;
//using utility::file::FileName;



ScoreFunctionOP scorefxn;
std::map <const char *, PoseOP> poses;
std::map <const char *, core::pack::task::PackerTaskOP> tasks;
CaptureTStreamOP resfile_reader_tcapture;
std::map <std::string, core::pack::task::ResfileCommandOP > resfile_command_map;


int init_state( 0 );

void init_mini( const char * database)
{
	int argc =8;
	//other possible flags -mute, core
	const char * argv [] = {"init_mini","-database", database, "-ignore_unrecognized_res", "-pack_missing_sidechains", "-mute", "core.pack.task.ResfileReader", "-interactive"};

	//int argc =4;
	//const char * argv [] = {"init_mini","-database", database, "-ignore_unrecognized_res" };

  devel::init( argc, const_cast<char**>( argv ) );



  scorefxn = ScoreFunctionFactory::create_score_function( STANDARD_WTS, SCORE12_PATCH );
	init_state = 1;

	//monitor the tracer output of the resfile reader
	resfile_reader_tcapture = new CaptureTStream();
	core::util::Tracer::set_ios_hook( resfile_reader_tcapture, "core.pack.task.ResfileReader" );

	resfile_command_map = core::pack::task::create_command_map();


}


int get_init_state()
{
	return init_state;
}

int get_init_state_pose( const char * pose_id )
{
	if ( poses.find( pose_id ) != poses.end() ){
		return 1;
	} else {
		return 0;
	}
}

std::string get_tracer()
{
	return resfile_reader_tcapture->read();
}


void set_weights( std::string weights_tag, std::string const & patch_tag )
{
	scorefxn = ScoreFunctionFactory::create_score_function( weights_tag, patch_tag );
}


std::vector < float >  get_weights()
{
	std::vector < float > non_zero_weights;
	EnergyMap const & weights( scorefxn->weights() );
	for ( int i = 1; i <= scoring::n_score_types; ++i ){
		if( weights[ ScoreType( i ) ] != 0.0 ){
			non_zero_weights.push_back( weights[ ScoreType( i ) ] );
		}
	}

	return non_zero_weights;
}


void init_pose( const char * pose_id, const char * pdb )
{
	pose::PoseOP pose = new Pose();
	std::string pdb_string( pdb );

	using namespace chemical;
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );

	core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData( pdb_string );
	fd.build_pose( * pose, * residue_set );
	io::pdb::read_additional_pdb_data( pdb_string, * pose, fd );

	poses[pose_id] = pose;

}


pose::PoseOP locate_pose( const char* pose_id)
{
	if( poses.find(pose_id) != poses.end() ){
		return poses[ pose_id ];
	} else {
		std::string pose_id_string(pose_id);
		std::cout << "Pose " << pose_id_string << " has not been initialized." << std::endl;
	  utility_exit();
		return 0;
	}
}

core::pack::task::PackerTaskOP locate_task( const char* pose_id)
{
	pose::PoseOP pose = locate_pose( pose_id );

	if( tasks.find(pose_id) != tasks.end() ){
		return tasks[ pose_id];
	} else {
		core::pack::task::PackerTaskOP task = core::pack::task::TaskFactory::create_packer_task( * pose );
		tasks[ pose_id ] = task;
		return task;
	}
}

void print_score( const char * pose_id)
{
	pose::PoseOP pose = locate_pose( pose_id );
	Energy score = ( *scorefxn )( *pose );
	std::cout << "score for " << pose_id << " is " << score << "." << std::endl;

}



float get_score( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	Energy score = ( *scorefxn )( *pose );
	return float(score);
}



char * get_detailed_score( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	std::ostringstream oss;

	scorefxn->show( oss, * pose );

	std::string s( oss.str() );
	char * output = new char[ s.size() + 1 ];
	std::strcpy(output, s.c_str() );

	//std::cout << "just for reference, dumping dump_scored_pdb to /users/momeara/LP17_scored.pdb" << std::endl;

	//std::string const filename("/users/momeara/LP17_scored.pdb");
	//pose->dump_scored_pdb( filename, * scorefxn );

	return output;
}

std::vector < std::string > get_score_terms(){

	std::vector < std::string > score_types;

	EnergyMap const & weights( scorefxn->weights() );
	for ( int i = 1; i <= scoring::n_score_types; ++i ){
		if ( weights[ ScoreType( i ) ] != 0.0 ){
			score_types.push_back( scoring::name_from_score_type( ScoreType( i ) ) );
		}
	}

	return score_types;
}

float get_weighted_score( const char * pose_id,
			  char chain,
			  int pdb_seqpos,
			  const char * scoretype_name ){

	pose::PoseOP pose = locate_pose( pose_id );
	if( chain == '\0' ) chain = ' '; // PyMOL trims out blank chain IDs

	//( *scorefxn )( *pose );

	//pose->update_residue_neighbors();

	//scorefxn->accumulate_residue_total_energies( *pose );
	//std::cout << "accumulate residue_total_energies" << std::endl;

	Size seqpos( pose->pdb_info()->pdb2pose( chain, pdb_seqpos) );
	ScoreType scoretype( scoring::score_type_from_name( scoretype_name ) );
	EnergyMap residue_energies( pose->energies().residue_total_energies( seqpos ) );
	Energy raw_energy( residue_energies[ scoretype ] );
	Energy weight( scorefxn->weights()[ scoretype ] );
	//std::cout << "raw_energy for chain " << chain << "/" << (int)chain << " pdb_seqpos " << pdb_seqpos << " ros_seqpos " << seqpos << " -> " << raw_energy << " weight -> " << weight << std::endl;

	return raw_energy*weight;
}

char * get_pdb( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	io::pdb::FileData fd;
	fd.init_from_pose( * pose );
	std::string return_pdb = io::pdb::PDB_DReader::createPDBData( fd );
	char * output = new char[ return_pdb.size() + 1 ];
	std::strcpy( output, return_pdb.c_str() );
	return output;
}

void set_task( const char * pose_id, const char * resfile )
{
    pose::PoseOP pose = locate_pose( pose_id );
    core::pack::task::PackerTaskOP packer_task = core::pack::task::TaskFactory::create_packer_task( * pose );
	core::pack::task::parse_resfile_string(*packer_task, resfile );
	tasks[ pose_id ] = packer_task;
}

void clean_residue_task( const char * pose_id, const char chain, const int pdb_seqpos )
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP task = locate_task( pose_id );
	Size seqpos( pose->pdb_info()->pdb2pose().find( chain, pdb_seqpos) );
	task->clean_residue_task( pose->residue( seqpos), seqpos );
}


void set_residue_task( const char * pose_id,  char chain, int pdb_seqpos, std::string command_str)
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP task = locate_task( pose_id );
	//std::cout << "here is the task before executing set_residue_task" <<std::endl;
	//std::cout << *task << std::endl;

	Size seqpos( pose->pdb_info()->pdb2pose().find( chain, pdb_seqpos) );

	if (chain == '_') chain = ' ';
	std::stringstream residue_command( command_str );
	utility::vector1< std::string > tokens( core::pack::task::tokenize_line( residue_command ) );
	Size which_token=1, ntokens( tokens.size());
	while( which_token <= ntokens ){
		//std::cout << "command = " << core::pack::task::get_token(which_token, tokens) << std::endl;
		core::pack::task::ResfileCommandOP command = resfile_command_map[ core::pack::task::get_token( which_token, tokens) ];
		if (!command ){
			std::stringstream error_msg;
			error_msg << "Error reading the residue task command: " << core::pack::task::get_token(which_token, tokens) << " for chain ->" << chain << " pdb_seqpos -> " << pdb_seqpos << "." << std::endl;
			T("core.pack.task.ResfileReader") << error_msg.str();
			throw core::pack::task::ResfileReaderException( error_msg.str() );
		}
		//std::cout << "executing set_residue_task (" << core::pack::task::get_token(which_token, tokens) << ", " << seqpos << ") with command string " << command_str << "-> " << std::endl;
		command->residue_action( tokens, which_token, *task, seqpos );
	}

	//std::cout << "here is the task after set_residue_task" << std::endl;
	//std::cout << *task << std::endl;
}

char * get_task( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP task = locate_task( pose_id );
	//std::cout << "found the task " << task << " with data " << *task << std::endl;
	std::ostringstream oss;
	oss << task->task_string( *pose );


	std::string s( oss.str() );
	char * output = new char[ s.size() + 1 ];
	std::strcpy( output, s.c_str() );

	return output;
}


std::vector< std::pair< char, int> > get_residues( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	std::vector< std::pair< char, int > > res_ids;
	for( Size ii=1; ii<= pose->total_residue(); ++ii ){
		char chain = pose->pdb_info()->chain(ii);
		int resi = ii;
		std::pair<char, int> res_id( chain, resi );
		res_ids.push_back( res_id );
	}
	return res_ids;
}

std::vector< bool > get_packable( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP task = locate_task( pose_id );
	std::vector< bool > packable;
	for( Size ii=1; ii <= pose->total_residue(); ++ii ){
		packable.push_back( task->residue_task( ii ).being_packed() );
	}
	return packable;
}

std::vector< int > num_res_types( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP task = locate_task( pose_id );
	std::vector< int > num_res_types;
	for( Size ii=1; ii <= pose->total_residue(); ++ii ){
		num_res_types.push_back( 0 );
		for(core::pack::task::ResidueLevelTask::ResidueTypeCAPListConstIter type( task->residue_task( ii ).allowed_residue_types_begin() ),
					end(  task->residue_task( ii ).allowed_residue_types_end() );
				type != end; ++type ) {
			num_res_types[ ii-1 ]++;
		}
	}
	return num_res_types;
}

void run_task( const char * pose_id )
{
	pose::PoseOP pose = locate_pose( pose_id );
	core::pack::task::PackerTaskOP packer_task = locate_task( pose_id );
	core::pack::pack_rotamers( *pose, *scorefxn, packer_task );
}


int
main( int argc, char * argv [] )
{

	if ( argc <= 2 ){
		std::cout << "Please pass in the path to the database as the first argument and a structure as the second and the resfile in as the third!" << std::endl;
	}
  init_mini( argv[ 1 ] );
  char * pdb_filename = argv[ 2 ];
	char * res_filename = argv[ 3 ];
	const char * pose_id = "STRUCT1";

	utility::io::izstream file( pdb_filename );
	std::string pdb;
	if (!file) {
		std::cout << "File:" << pdb_filename << " not found!" << std::endl;
	} else {
		std::cout << "read file: " << pdb_filename << std::endl;
	}
	utility::slurp( file, pdb );

	init_pose( pose_id, pdb.c_str() );
	print_score( "STRUCT1" );
	float abc = get_score( "STRUCT1" );
	std::cout << "got score: "<< abc <<"."<< std::endl;
	//	char * pdb_data = get_pdb( "STRUCT1" );

	utility::io::izstream res_file( res_filename );
	std::string res_string;
	if ( !res_file ) {
		std::cout << "Resfile:" << res_filename << " not found!" << std::endl;
	} else {
		std::cout << "read resfile: " << res_filename << std::endl;
	}
  utility::slurp( res_file, res_string );

	std::cout << "Setting the task..." << std::endl << res_string << std::endl;
	set_task( "STRUCT1", res_string.c_str() );


	std::cout << "Here is the task after it has been read in" << std::endl << *locate_task("STRUCT1") <<std::endl;

	std::cout << "Running the task (this can take a while)..." << std::endl;
	run_task( "STRUCT1" );


	std::cout << "Getting score term ... " << std::endl;
	std::cout << "STRUCT1 A 1 fa_atr -> " << get_weighted_score( "STRUCT1", 'A', 1, "fa_atr" ) << std::endl;

	std::cout << "after_running task:" << get_detailed_score( "STRUCT1" ) << std::endl;

	std::vector< std::pair< char, int > > res_ids = get_residues( "STRUCT1" );
	for ( Size i = 0; i < res_ids.size(); ++ i){
		std::cout << "res_id ("<<res_ids[i].first<<", "<<res_ids[i].second <<")" << std::endl;
	}

}

