// -*- mode:c++;tab-width:2;indent-tabs-mode:nil;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/hbonds/HBondFadeIntervalTest.cxxtest.hh
/// @brief  Test hbond fade interval code
/// @author Matthew O'Meara (mattjomeara@gmail.com)

// Test Headers
#include <cxxtest/TestSuite.h>
#include <test/UTracer.hh>
#include <test/core/init_util.hh>

// Unit Headers
#include <core/scoring/hbonds/FadeInterval.hh>
#include <core/scoring/hbonds/HBondDatabase.hh>
#include <core/scoring/hbonds/HBondTypeManager.hh>
#include <core/scoring/hbonds/types.hh>
#include <core/scoring/hbonds/constants.hh>

// Project Headers
#include <core/io/database/open.hh>
#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/corrections.OptionKeys.gen.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>

// Utility Headers
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/vector1.hh>
#include <utility/string_util.hh>

// Numeric Headers
#include <numeric/numeric.functions.hh>

// Boost Headers
#include <boost/tokenizer.hpp>

// C++ Headers
#include <string>

using namespace std;
using numeric::eq_tol;

using namespace core;
using namespace utility;
using namespace core::scoring::hbonds;

static core::util::Tracer TR("core.scoring.hbonds.HBondFadeIntervalTest.cxxtest");

class HBondFadeIntervalTest : public CxxTest::TestSuite {

public:

	void
	setUp() {
		core_init();
	}

	void
	tearDown(){}


	void test_hbond_fade_interval(){
    //    test::UTracer UT("core/scoring/hbonds/HBondFadeIntervalTest.u");
		vector1< string > tags;
		tags.push_back( "standard_params" );
		tags.push_back( "helix_hb_06_2009" );
		tags.push_back( "newOH_params" );
		tags.push_back( "His_Phil_fix" );
		for( Size i = 1; i <= tags.size(); ++i ){
      //			do_test_definitions(tags[i], UT);
			do_test_hbond_fade_interval(tags[i]);
		}
	}


	void
	do_test_definitions(
    string const & hbdatabase_tag,
    test::UTracer & UT
  ){

		string HBFadeInterval_fname = "scoring/score_functions/hbonds/" + hbdatabase_tag + "/HBFadeIntervals.csv";



		UT << "HBFadeInterval_fname: " << HBFadeInterval_fname << endl;

		utility::io::izstream s;
		if (!core::io::database::open(s, HBFadeInterval_fname)){
			stringstream message;
			message << "Unable to open hbond parameter file HBFadeInterval:" << std::endl;
			message << "'" << HBFadeInterval_fname << "'";
			utility_exit_with_message(message.str());
		}

		string line;
		vector1<string> tokens;
		Size id;
		string fade_interval_name;
		Real min0, fmin, fmax, max0;
		while ( getline( s, line ) ) {
			tokens = string_split( line, ',');
			Size ntokens = 7;
			if (tokens.size() != ntokens){
				stringstream message;
				message << "FadeInterval definition line does not have the expected number of fields " << std::endl;
				message << "Expected '" << ntokens << "' tokens but found '" << tokens.size() << "' tokens. " << std::endl;
				message << line << std::endl;
				utility_exit_with_message(message.str());
			}

			Size i(1);

			{	stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> id; }
			{	stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> fade_interval_name; }
			{ stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> min0; }
			{ stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> fmin; }
			{ stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> fmax; }
			{ stringstream buf;buf.precision(17); buf << tokens[i]; i++; buf >> max0; }

			UT.precision(17);
			UT << "id: "
				<< setw(17)
				<< fade_interval_name
				<< " min0: " << min0
				<< " fmin: " << fmin
				<< " fmax: " << fmax
				<< " max0: " << max0 << endl;


			doubleprint(UT, min0);      UT << endl;
			doubleprint(UT, fmin);      UT << endl;
			doubleprint(UT, fmax);      UT << endl;
			doubleprint(UT, max0);      UT << endl;

		}

		UT << "id: "
			<< setw(17)
			<< "fade_rBB_old"
			<< " min0: " << MIN_R
			<< " fmin: " << MIN_R+0.1
			<< " fmax: " << R_INTERP_EDGE
			<< " max0: " << MAX_R << endl;

		doubleprint(UT, MIN_R);          UT << endl;
		doubleprint(UT, MIN_R+0.1);      UT << endl;
		doubleprint(UT, R_INTERP_EDGE);  UT << endl;
		doubleprint(UT, MAX_R);          UT << endl;

		UT << "id: "
			<< setw(17)
			<< "fade_rshort_old"
			<< " min0: " << MIN_R
			<< " fmin: " << MIN_R+0.1
			<< " fmax: " << R_INTERP_EDGE
			<< " max0: " << MAX_R << endl;

		doubleprint(UT, MIN_R); UT << endl;
		doubleprint(UT, MIN_R+0.1); UT << endl;
		doubleprint(UT, R_INTERP_EDGE); UT << endl;
		doubleprint(UT, MAX_R); UT << endl;


		UT << "id: "
			<< setw(17)
			<< "fade_rlong_old"
			<< " min0: " << LR_MIN_R_CUT()
			<< " fmin: " << LR_MIN_R_FADE()
			<< " fmax: " << LR_MAX_R_FADE()
			<< " max0: " << LR_MAX_R_CUT() << endl;

		doubleprint(UT, LR_MIN_R_CUT());  UT << endl;
		doubleprint(UT, LR_MIN_R_FADE()); UT << endl;
		doubleprint(UT, LR_MAX_R_FADE()); UT << endl;
		doubleprint(UT, LR_MAX_R_CUT());  UT << endl;


		UT << "id: "
			<< setw(17)
			<< "fade_rBB_long_old"
			<< " min0: " << 0.0
			<< " fmin: " << 0.0
			<< " fmax: " << 0.0
			<< " max0: " << 0.0 << endl;


		doubleprint(UT, 0.0);      UT << endl;
		doubleprint(UT, 0.0);      UT << endl;
		doubleprint(UT, 0.0);      UT << endl;
		doubleprint(UT, 0.0);      UT << endl;
		UT << endl;

		UT << "id: "
			<< setw(17)
			<< "fade_xD_old"
			<< " min0: " << MIN_xD_CUT()
			<< " fmin: " << MIN_xD_FADE()
			<< " fmax: " << 1.01
			<< " max0: " << 1.01 << endl;


		doubleprint(UT, MIN_xD_CUT()); UT << endl;
		doubleprint(UT, MIN_xD_FADE()); UT << endl;
		doubleprint(UT, 1.01); UT << endl;
		doubleprint(UT, 1.01); UT << endl;
		UT << endl;

		UT << "id: "
			<< setw(17)
			<< "fade_xH_old"
			<< " min0: " << MIN_xH_CUT()
			<< " fmin: " << MIN_xH_FADE()
			<< " fmax: " << 1.01
			<< " max0: " << 1.01 << endl;


		doubleprint(UT, MIN_xH_CUT());  UT << endl;
		doubleprint(UT, MIN_xH_FADE()); UT << endl;
		doubleprint(UT, 1.01);          UT << endl;
		doubleprint(UT, 1.01);          UT << endl;

	}


	void
	do_test_hbond_fade_interval(
		string const hbdatabase_tag,
		Real absolute_tolerance = .000000000000001
	){
		TR.precision(17);

		HBondDatabaseCOP database(HBondDatabase::get_database(hbdatabase_tag));

		// Eventually, these will be attached to the polynomials.
		FadeInterval const fade_rBB_old(MIN_R, MIN_R+0.1, R_INTERP_EDGE, MAX_R); // used to adjust xD,xH
		FadeInterval const fade_rshort_old(MIN_R, MIN_R+0.1, INTERP_MIN, INTERP_MAX);
		FadeInterval const fade_rlong_old(LR_MIN_R_CUT(), LR_MIN_R_FADE(), LR_MAX_R_FADE(), LR_MAX_R_CUT());
		FadeInterval const fade_rBB_long_old;
		FadeInterval const fade_xD_old(MIN_xD_CUT(), MIN_xD_FADE(), 1.01, 1.01); // xD=theta should fade r,xH sooner!
		FadeInterval const fade_xH_old(MIN_xH_CUT(), MIN_xH_FADE(), 1.01, 1.01); // fades r,xD


		for(Size hbdon_index = 1; hbdon_index <= hbdon_MAX; ++hbdon_index){
			HBDonChemType hbdon(static_cast<HBDonChemType>(hbdon_index));
			string hbdon_name(HBondTypeManager::name_from_don_chem_type(hbdon));
			for(Size hbacc_index = 1; hbacc_index <= hbacc_MAX; ++hbacc_index){
				HBAccChemType hbacc(static_cast<HBAccChemType>(hbacc_index));
				string hbacc_name(HBondTypeManager::name_from_acc_chem_type(hbacc));
				for(Size seq_sep_index = 1; seq_sep_index <= seq_sep_MAX; ++seq_sep_index){
					HBSeqSep seq_sep(static_cast<HBSeqSep>(seq_sep_index));
					string seq_sep_name(HBondTypeManager::name_from_seq_sep_type(seq_sep));
					HBEvalType hbe(HBEval_lookup(hbdon, hbacc, seq_sep));

					if(hbe==0) continue;
					FadeIntervalCOP AHdist_short_fade(database->AHdist_short_fade_lookup(hbe));
					const FadeInterval * AHdist_short_fade_old;
					if(hbe_is_BB_type(hbe)){
						AHdist_short_fade_old = &fade_rBB_old;
					} else {
						AHdist_short_fade_old = &fade_rshort_old;
					}

					FadeIntervalCOP AHdist_long_fade(database->AHdist_long_fade_lookup(hbe));
					const FadeInterval * AHdist_long_fade_old;
					if (!hbe_is_BB_type(hbe) && !core::options::option[ core::options::OptionKeys::corrections::score::hbond_no_lr ]() ){
						AHdist_long_fade_old = &fade_rlong_old;
					} else {
						AHdist_long_fade_old = &fade_rBB_long_old;
					}

					FadeIntervalCOP cosBAH_fade(database->cosBAH_fade_lookup(hbe));
					const FadeInterval * cosBAH_fade_old( &fade_xH_old );

					FadeIntervalCOP cosAHD_fade(database->cosAHD_fade_lookup(hbe));
					const FadeInterval * cosAHD_fade_old( &fade_xD_old );

					bool is_problem(false);


					if( abs(AHdist_short_fade->get_min0()-
									AHdist_short_fade_old->get_min0())> absolute_tolerance ||
							abs(AHdist_short_fade->get_fmin()-
									AHdist_short_fade_old->get_fmin())> absolute_tolerance ||
							abs(AHdist_short_fade->get_fmax()-
									AHdist_short_fade_old->get_fmax())> absolute_tolerance ||
							abs(AHdist_short_fade->get_max0()-
									AHdist_short_fade_old->get_max0())> absolute_tolerance ){
							TR << "\tAHdist_short_fade:\t" << *AHdist_short_fade << endl;
							TR << "\tAHdist_short_fade_old:\t" << *AHdist_short_fade_old << endl;
							TR << abs(AHdist_short_fade->get_min0() - AHdist_short_fade_old->get_min0()) << " "
								<< abs(AHdist_short_fade->get_fmin() - AHdist_short_fade_old->get_fmin()) << " "
								<< abs(AHdist_short_fade->get_fmax() - AHdist_short_fade_old->get_fmax()) << " "
								<< abs(AHdist_short_fade->get_max0() - AHdist_short_fade_old->get_max0()) << endl;
							is_problem = true;
						}
					if( abs(AHdist_long_fade->get_min0()-
									AHdist_long_fade_old->get_min0())> absolute_tolerance ||
							abs(AHdist_long_fade->get_fmin()-
									AHdist_long_fade_old->get_fmin())> absolute_tolerance ||
							abs(AHdist_long_fade->get_fmax()-
									AHdist_long_fade_old->get_fmax())> absolute_tolerance ||
							abs(AHdist_long_fade->get_max0()-
									AHdist_long_fade_old->get_max0())> absolute_tolerance ){
							TR << "\tAHdist_long_fade:\t" << *AHdist_long_fade << endl;
							TR << "\tAHdist_long_fade_old:\t" << *AHdist_long_fade_old << endl;
							TR << abs(AHdist_short_fade->get_min0() - AHdist_short_fade_old->get_min0()) << " "
								<< abs(AHdist_short_fade->get_fmin() - AHdist_short_fade_old->get_fmin()) << " "
								<< abs(AHdist_short_fade->get_fmax() - AHdist_short_fade_old->get_fmax()) << " "
								<< abs(AHdist_short_fade->get_max0() - AHdist_short_fade_old->get_max0()) << endl;

							is_problem = true;
						}

					if( abs(cosBAH_fade->get_min0()-
									cosBAH_fade_old->get_min0())> absolute_tolerance ||
							abs(cosBAH_fade->get_fmin()-
									cosBAH_fade_old->get_fmin())> absolute_tolerance ||
							abs(cosBAH_fade->get_fmax()-
									cosBAH_fade_old->get_fmax())> absolute_tolerance ||
							abs(cosBAH_fade->get_max0()-
									cosBAH_fade_old->get_max0())> absolute_tolerance){
							TR << "\tcosBAH_fade:\t" << *cosBAH_fade << endl;
							TR << "\tcosBAH_fade_old:\t" << *cosBAH_fade_old << endl;
							TR << abs(AHdist_short_fade->get_min0() - AHdist_short_fade_old->get_min0()) << " "
								<< abs(AHdist_short_fade->get_fmin() - AHdist_short_fade_old->get_fmin()) << " "
								<< abs(AHdist_short_fade->get_fmax() - AHdist_short_fade_old->get_fmax()) << " "
								<< abs(AHdist_short_fade->get_max0() - AHdist_short_fade_old->get_max0()) << endl;
							is_problem = true;
					}
					if( abs(cosAHD_fade->get_min0()-
									cosAHD_fade_old->get_min0())> absolute_tolerance ||
							abs(cosAHD_fade->get_fmin()-
									cosAHD_fade_old->get_fmin())> absolute_tolerance ||
							abs(cosAHD_fade->get_fmax()-
									cosAHD_fade_old->get_fmax())> absolute_tolerance ||
							abs(cosAHD_fade->get_max0()-
									cosAHD_fade_old->get_max0())> absolute_tolerance){
							TR << "\tcosAHD_fade:\t" << *cosAHD_fade << endl;
							TR << "\tcosAHD_fade_old:\t" << *cosAHD_fade_old << endl;
							TR << abs(AHdist_short_fade->get_min0() - AHdist_short_fade_old->get_min0()) << " "
								<< abs(AHdist_short_fade->get_fmin() - AHdist_short_fade_old->get_fmin()) << " "
								<< abs(AHdist_short_fade->get_fmax() - AHdist_short_fade_old->get_fmax()) << " "
								<< abs(AHdist_short_fade->get_max0() - AHdist_short_fade_old->get_max0()) << endl;
							is_problem = true;
					}

					if(is_problem){
						TR << "hbe: " << hbe << " [" << hbdon_name << "," << hbacc_name << "," << seq_sep_name << "]" <<endl;
						TS_ASSERT(false);
					}

				} // seqsep
			} // hbacc
		} // hbdon
	} //do_test_hbond_fade_interval

//  doubleprint()   Print the bit representation of a double.
//
//  Useful for debugging exact arithmetic routines.
//  Shewchuk, J.R., Routines for Arbitrary Precision Floating-point
//  Arithmetic and Fast Robust Geometric Predicates, 1996
void doubleprint(
  std::ostream & out,
  double number)
{
	unsigned long long no;
	unsigned long long sign, expo;
	int exponent;
	int i, bottomi;

	no = *(unsigned long long *) &number;
	sign = no & 0x8000000000000000ll;
	expo = (no >> 52) & 0x7ffll;
	exponent = (int) expo;
	exponent = exponent - 1023;
	if (sign) {
    out << "-";
	} else {
    out << " ";
	}
	if (exponent == -1023) {
		out <<
			"0.0000000000000000000000000000000000000000000000000000_     (   )";
	} else {
    out << "1.";
		bottomi = -1;
		for (i = 0; i < 52; i++) {
			if (no & 0x0008000000000000ll) {
        out << "1";
				bottomi = i;
			} else {
        out << "0";
			}
			no <<= 1;
		}
    out << "_" << exponent << "  (" << exponent - 1 - bottomi << ")";
	}
}

}; // HBondFadeIntervalTest
